/******************************************************************************************************************************************
 ccombobox.h

 CClass						CClass
 |--CSerialized					|--CSerialized
    |--CMetaModule				   |--CMetaModule
       |--CObject				      |--CObjectListener
          |--CComponent				         |
             |--CControl				 |
             |  |--CWidget				 |--CWidgetListener
             |     |--CComboBox				    |--CComboBoxListener
	     |--CComboBoxItem

 Like the gtk view representations, the gtkol combo box definition is based on a model. This model is specified as a metaclasses list of
 item field values instances (see citemfieldvalue.h). The gtkol combo box instance works on combobox items that own the field values of
 the specific handled combo box model. The combo box model must be defined before any combo box item is added to the combo box widget so
 that the comb box widget knows how to handle the field values of the owned items. The combo box item field values must map the combo box
 widget defined model or the owner affectation of the combo box item to another combo box item or to the combo box widget would fail 
 because of not corresponding to the expected model. The default model is the simpliest one i.e. a field value for each combo box item
 that is a CItemFieldValueString one and so displays just a string for each item.

 CComboBox ==> { CComboBoxItem }
		      |==> { CItemFieldValue  ( CItemFieldValueString, CItemFieldValuePixbuf ) }

 N.B. Only the m_FieldValue member of the CItemFieldValue classes is handled in the mapped model.
******************************************************************************************************************************************/

#ifndef __CCOMBOBOX_H__
#define __CCOMBOBOX_H__

#include "ccontainer.h"
#include "citemfieldvalue.h"

// what's defined here
class CComboBox;
class CComboBoxItem;
class CComboBoxListener;

//-----------------------------------------------------------------------------------------------------------------------------------------
// ccomboboxitem xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ccomboboxitem>
//   <fields>
//     ...
//   </fields>
// </ccomboboxitem>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_COMBOBOXITEM_ELEMENT			("ccomboboxitem");
static CString XML_COMBOBOXITEM_FIELDS_ELEMENT		("fields");

//-----------------------------------------------------------------------------------------------------------------------------------------
// ccombobox xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ccombobox selection="SInt16">
//   <ccombobox-model>
//       <ccombobox-field name="string_type" tag="tag_type">
//     [ <ccombobox-field name="string_type" tag="tag_type"> ]
//   </ccombobox-model>
//   ...
// </ccombobox>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_COMBOBOX_ELEMENT			("ccombobox");
static CString XML_COMBOBOX_ATTR_SELECTION		("selection");
static CString XML_COMBOBOX_MODEL_ELEMENT		("ccombobox-model");
static CString XML_COMBOBOX_FIELDCLASS_ELEMENT		("ccombobox-field");
static CString XML_COMBOBOX_FIELDCLASS_ATTR_NAME	("name");
static CString XML_COMBOBOX_FIELDCLASS_ATTR_TAG 	("tag");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CComboBoxListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CComboBoxListener : public CWidgetListener
{
	// instanciation section
	public :

		CComboBoxListener		();
		virtual ~CComboBoxListener	() =0;

	// combobox specifics
	public :
		
		// called when the active item of the inSender combobox has changed
		virtual void			OnChange			(CObject *inSender)		{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_cbo', CComboBoxListener, CWidgetListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CComboBoxItem class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CComboBoxItem : public CComponent
{
	// instanciation section
	public :

		CComboBoxItem			(CComboBox *inOwner=NULL, const CItemFieldValues &inFieldValues=CItemFieldValues(),
						 const CObjectListener *inListener=NULL);
		CComboBoxItem			(CComboBoxItem *inOwner,  const CItemFieldValues &inFieldValues=CItemFieldValues(),
						 const CObjectListener *inListener=NULL);
		virtual ~CComboBoxItem		();

	// ccomponent redefinition
	public :

		// expected owner CComboBox or CComboBoxItem
		virtual CMetaClasses 		OwnerMustBe			() const;
		virtual CMetaClasses		ChildMustBe			() const;

		// owner affectation, must be derived from CComboBox or CComboBoxItem
		virtual bool			SetOwner			(CComponent *inOwner, const SInt16 inIndex=-1);
		
	// gtkol combobox item / gtk tree item correspondance
	public :
	
		// get the gtk tree iter associated to the gtkol combo box item instance
		GtkTreeIter *			GetGtkTreeIter			() const;

		// get the gtkol combo box item of specified gtk tree iter if any
		static CComboBoxItem *		GetComboBoxItem			(const GtkTreeIter *inGtkTreeIter);

	// combobox item specifics
	public :

		// item field values access
		virtual Bool			SetItemFieldValues		(const CItemFieldValues &inItemFieldValues);
		CItemFieldValues		GetItemFieldValues		() const;

		// 0 indexed field value direct access, no check on range so be carefull
		CItemFieldValue &		operator []			(const size_t) const;

		// item selection
		virtual void			Select				();
		virtual bool			IsSelected			() const;

	// cserialized redefinition
	public :

		// combobox item serialization
		virtual void			Serialize			(CXMLElementNode *&, const int) THROWABLE;

	// protected attributes
	protected :
		
		// gtk tree iter handled attribute
		GtkTreeIter			m_GtkTreeIter;

		// combobox item field values
		CItemFieldValues		m_ItemFieldValues;

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('cboi', CComboBoxItem, CComponent);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CComboBox class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CComboBox : public CWidget
{
	// instanciation section
	public :

		CComboBox			(CContainer *inOwner=NULL, const CMetaClasses &inModel=_IFVString_, THROWABLE
						 const CComboBoxListener *inListener=NULL);
		virtual ~CComboBox		();

	// protected requests handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// cobject redefinition
	protected :

		// listener affectation, must be derived from CComboBoxListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// ccomponent redefinition
	public :

		// ccontainer as a owner, ccomboboxitem as children
		virtual CMetaClasses		OwnerMustBe			() const;
		virtual CMetaClasses		ChildMustBe			() const;

	// ccombobox specifics
	public :

		// ccombox items field values model affectation
		virtual Bool			SetModel			(const CMetaClasses &inModel);
		CMetaClasses			GetModel			() const;

		// combo box item selection (see gtk_combo_box for indices specifications on tree model)
		virtual void			Select				(const SInt16);
		virtual void			Select				(const CComboBoxItem *);
		virtual SInt16			GetSelectionIndex		() const;
		virtual CComboBoxItem *		GetSelectionItem		() const;

		// popup, popdown the combo
		virtual void			Popup				();
		virtual void			Popdown				();

	// cserialized redefinition
	public :

		// ccombobox serialization
		virtual void			Serialize			(CXMLElementNode *&, const int) THROWABLE;

	// protected attributes and events handling
	protected :

		// combobox model representation (expected combobox items typed field values)
		CMetaClasses			m_Model;

		// static event to dynamic handling
		static void			OnChange			(GtkComboBox *, gpointer);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('cbox', CComboBox, CWidget);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CComboBoxItemFieldValueAPIListener definition, internaly used, should not be used by the front end development
//-----------------------------------------------------------------------------------------------------------------------------------------
class CComboBoxItemFieldValueAPIListener : public CItemFieldValueAPIListener
{
	// public handling
	public :

		// called when the developper explicitely requests some modification on the item field value api
		virtual void			OnStateChange			(CObject *);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('_cpi', CComboBoxItemFieldValueAPIListener, CItemFieldValueAPIListener);

#endif
