/*******************************************************************************************************************************************
 combobox.c

 simple compilation directives :
 c++ -Wno-multichar -O2 -o combobox `pkg-config --cflags libgtkol-1.2` combobox.c `pkg-config --libs libgtkol-1.2`

 This demo module shows three comboboxes. The first one displays items owning a pixbuf and a string each. The second one displays items
 that are handled with a tree model. The last dislpays a comboboxentry instance and does not accept digits.
*******************************************************************************************************************************************/

#include "capplication.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// define a comboboxentry listener
//-----------------------------------------------------------------------------------------------------------------------------------------
class CDemoComboBoxEntryListener : public CComboBoxEntryListener
{
	// called when the specified combobox selection has changed
	virtual void OnChange 	       (CObject *inSender);

	// called when the inSender combobox entry is about to append an item following a key validation
	virtual void OnQueryAppendItem (CObject *inSender, CComboBoxItem *&ioItem, const CString &inNewString, SInt32 &ioIndex);
};

//-----------------------------------------------------------------------------------------------------------------------------------------
// CDemoForm definition
//-----------------------------------------------------------------------------------------------------------------------------------------
class CDemoForm : public CForm
{
	// instanciation section
	public :

		CDemoForm (CApplication *inOwner);
	
	// gui direct access section
	public :
 
		CStatusBar *m_StatusBar;
};

//-----------------------------------------------------------------------------------------------------------------------------------------
// demo form constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CDemoForm::CDemoForm   (CApplication *inOwner) 
	  :CForm       (inOwner), 
	   m_StatusBar (NULL)
{
	// set the form caption
	SetCaption (CString("Combo Box"));

	// set the form bounds (center the form over screen)
	SetBounds  (TBounds ((CApplication::GetScreenSize().w-280)/2, (CApplication::GetScreenSize().h-150)/2, 280, 150));

	// instanciate a vertical box layout under the form and set some properties
	CVBoxLayout *VBoxLayout = new CVBoxLayout (this);
	VBoxLayout -> SetHomogeneous (false);
	VBoxLayout -> SetBoxPack (BOXPACK_START, false, false, 0);

	// instanciate a frame owning the first combobox
	CFrame *Frame0 = new CFrame (VBoxLayout, CString("Stock pixbufs"));

	// instanciate a unique listener to handle the different comboboxes events
	CComboBoxEntryListener *theDemoComboBoxEntryListener = new CDemoComboBoxEntryListener ();

	// instanciate the first combobox overwritting the default model, the given listener derives the CComboBoxEntryListener which
	// derives the CComboBoxListener itself, so it can handle generic comboboxes events too
	CComboBox *ComboBox0 = new CComboBox (Frame0, _IFVPixbuf_+_IFVString_, theDemoComboBoxEntryListener);

	// give it some items respecting the combobox owner model
	CComboBoxItem &Item0 = *new CComboBoxItem (ComboBox0);
	Item0[0] = new CPixbuf (GTK_STOCK_COPY);
	Item0[1] = CString("Copy");
	CComboBoxItem &Item1 = *new CComboBoxItem (ComboBox0);
	Item1[0] = new CPixbuf (GTK_STOCK_CUT);
	Item1[1] = CString("Cut");
	CComboBoxItem &Item2 = *new CComboBoxItem (ComboBox0);
	Item2[0] = new CPixbuf (GTK_STOCK_PASTE);
	Item2[1] = CString("Paste");
	
	// default selection of the combobox (index specification)
	ComboBox0 -> Select ((SInt16)0L);

	// instanciate a frame owning the second combobox
	CFrame *Frame1 = new CFrame (VBoxLayout, CString("Tree model"));

	// instanciate the second combobox
	CComboBox *ComboBox1 = new CComboBox (Frame1);

	// assign it a listener, the given listener derives the CComboBoxEntryListener which derives the CComboBoxListener itself, 
	// so it can handle generic comboboxes events too
	ComboBox1 -> AssignListener (theDemoComboBoxEntryListener);

	// give it some items
	CComboBoxItem &Item10 = *new CComboBoxItem (ComboBox1);
	Item10[0] = CString("France");
	CComboBoxItem &Item01 = *new CComboBoxItem (&Item10);
	Item01[0] = CString("Ile de France");
	CComboBoxItem &Item02 = *new CComboBoxItem (&Item10);
	Item02[0] = CString("Picardie");
	CComboBoxItem &Item03 = *new CComboBoxItem (&Item10);
	Item03[0] = CString("Bretagne");
	CComboBoxItem &Item04 = *new CComboBoxItem (&Item10); 
	Item04[0] = CString("Alsace");
	CComboBoxItem &Item11 = *new CComboBoxItem (ComboBox1);
	Item11[0] = CString("Allemagne");
	CComboBoxItem &Item05 = *new CComboBoxItem (&Item11);
	Item05[0] = CString("Schleswig");
	CComboBoxItem &Item06 = *new CComboBoxItem (&Item11);
	Item06[0] = CString("Niedersachsen");
	CComboBoxItem &Item07 = *new CComboBoxItem (&Item11); 
	Item07[0] = CString("Westfalen");
	CComboBoxItem &Item08 = *new CComboBoxItem (&Item11);
	Item08[0] = CString("Hessen");
	CComboBoxItem &Item09 = *new CComboBoxItem (&Item11);
	Item09[0] = CString("Saarland");
	CComboBoxItem &Item0a = *new CComboBoxItem (&Item11); 
	Item0a[0] = CString("Baden");
	CComboBoxItem &Item0b = *new CComboBoxItem (&Item11);
	Item0b[0] = CString("Bayern");
	CComboBoxItem &Item0c = *new CComboBoxItem (&Item11);
	Item0c[0] = CString("Sachsen");
	CComboBoxItem &Item12 = *new CComboBoxItem (ComboBox1);
	Item12[0] = CString("Etats Unis");
	CComboBoxItem &Item0d = *new CComboBoxItem (&Item12);
	Item0d[0] = CString("Californie");
	CComboBoxItem &Item0e = *new CComboBoxItem (&Item12); 
	Item0e[0] = CString("Washington");
	CComboBoxItem &Item0f = *new CComboBoxItem (&Item12); 
	Item0f[0] = CString("Michigan");

	// default combobox selection (item specification)
	ComboBox1 -> Select (&Item10);

	// instanciate a frame owning the third combobox
	CFrame *Frame2 = new CFrame (VBoxLayout, CString("Entry"));

	// instanciate the third combobox
	CComboBoxEntry *ComboBox2 = new CComboBoxEntry (Frame2);

	// assign it the defined listener which, in this sample, will handle classical comboboxes events and combobox entry events
	ComboBox2 -> AssignListener (theDemoComboBoxEntryListener);

	// give it some items (another way to do)
	new CComboBoxItem (ComboBox2, CItemFieldValues (1, new CItemFieldValueString (CString("One"))));
	new CComboBoxItem (ComboBox2, CItemFieldValues (1, new CItemFieldValueString (CString("Two"))));
	new CComboBoxItem (ComboBox2, CItemFieldValues (1, new CItemFieldValueString (CString("2,5"))));
	new CComboBoxItem (ComboBox2, CItemFieldValues (1, new CItemFieldValueString (CString("Three"))));
			
	// default selection of the combobox (index specification)
	ComboBox2 -> Select ((SInt16)0L);
	
	// instanciate the status bar and set its default caption
	m_StatusBar = new CStatusBar (VBoxLayout);
	m_StatusBar -> SetCaption (CString("Ready."));

	// show the form (CForm is the only gtkol component that is not automatically shown)
	Show ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// combobox OnChange event handler
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDemoComboBoxEntryListener::OnChange (CObject *inSender)
{
	// retreive our combobox instance (in this sample code, this listener instance definition is used on both classical comboboxes
	// and entry comboboxes, so we can just cast to the generic combobox definition : inSender may be a CComboBox or a 
	// CComboBoxEntry; see the CDemoForm constructor section)
	CComboBox *inComboBox = static_cast <CComboBox *> (inSender);

	// get the demo form owner by requesting the expected owner type of the combobox instance
	CDemoForm *inDemoForm = static_cast <CDemoForm *> (inComboBox -> GetOwner (__metaclass(CForm)));

	// see if the status bar is available on the demo form
	if (inDemoForm -> m_StatusBar != NULL)
	{
		// get the current combobox selection
		CComboBoxItem *inSelection (inComboBox -> GetSelectionItem ());

		// be paranoid, check there is a selection
		if (inSelection == NULL) return;

		// get the item field values of the current selection
		CItemFieldValues inItemFieldValues (inSelection -> GetItemFieldValues());

		// go through the field values as this listener is instanciated on comboboxes that handle different models
		for (size_t i=0; i<inItemFieldValues.GetLength(); i++)
		{
			// check the field value type : we are looking for a string to set on the status bar
			if ((*inItemFieldValues[i]) -> ClassIs (__metaclass(CItemFieldValueString)))
			{
				// set the status bar string from given field string
				inDemoForm -> m_StatusBar -> SetCaption (CString("Last selected : ") +
					static_cast <CItemFieldValueString *> (*inItemFieldValues[i]) -> GetFieldValue());

				// stop on the first string field value
				break;
			}
		}
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// called when the combobox entry is about to append an item following a key validation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CDemoComboBoxEntryListener::OnQueryAppendItem (CObject *inSender, CComboBoxItem *&ioItem, const CString &inNewString, SInt32 &)
{
	// check gtkol proposes an item (should never append on the simple single string model but true when the combobox model
	// is a more complex one because gtkol would not know which values to fill in the item in such a case and it would be the
	// listener job to do so)
	if (ioItem == NULL) return;

	// go through the potential specified new string
	for (size_t i=0; i<inNewString.GetLength(); i++)
	{
		// check there is not digit in the string
		if (::isdigit (*inNewString[i]))
		{
			// delete the proposed new item and set it to NULL so that libgtkol will not append it to the combobox
			delete ioItem; ioItem = NULL;

			// retreive the combobox instance
			CComboBox *inComboBox = static_cast <CComboBox *> (inSender);

			// retreive the owner form of the combobox instance
			CDemoForm *inDemoForm = static_cast <CDemoForm *> (inComboBox -> GetOwner (__metaclass(CForm)));

			// display some information on the status bar
			inDemoForm -> m_StatusBar -> SetCaption (CString("Digits not accepted !"));

			// stop
			break;
		}
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// execution entry point
//-----------------------------------------------------------------------------------------------------------------------------------------
int main (int argc, char **argv)
{
	// declare an application
	CApplication Application (argc, argv);

	// instanciate our demo form
	new CDemoForm (&Application);

	// run the application
	Application.Run ();

	// ok
	return 0;
}


