/*
 * multiplexer.vala
 *
 * Authored by Michael 'Mickey' Lauer <mlauer@vanille-media.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 */

//===========================================================================
using GLib;
using CONST;
using Gsm0710;
using Gsm0710mux;

//===========================================================================
// callback forwarders
//

internal static bool at_command_fwd( Context ctx, string command )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    return m.at_command( command );
}

internal static int read_fwd( Context ctx, void* data, int len )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    return m.read( data, len );
}

internal static bool write_fwd( Context ctx, void* data, int len )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    return m.write( data, len );
}

internal static void deliver_data_fwd( Context ctx, int channel, void* data, int len )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    m.deliver_data( channel, data, len );
}

internal static void deliver_status_fwd( Context ctx, int channel, int status )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    m.deliver_status( channel, status );
}

internal static void debug_message_fwd( Context ctx, string msg )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    m.debug_message( msg );
}

internal static void open_channel_fwd( Context ctx, int channel )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    m.open_channel( channel );
}

internal static void close_channel_fwd( Context ctx, int channel )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    m.close_channel( channel );
}

internal static void terminate_fwd( Context ctx )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    m.terminate();
}

internal static void response_to_test_fwd( Context ctx, char[] data )
{
    Multiplexer m = (Multiplexer) ctx.user_data;
    m.response_to_test( data );
}

//===========================================================================
// The Multiplexer class
//
internal class Multiplexer
{
    Manager manager;
    FsoFramework.Logger logger;

    string portname;
    Context ctx;
    int portfd = -1;
    IOChannel portchannel;
    uint portwatch;
    uint portspeed;

    uint pingwatch;

    Timer idle_wakeup_timer;
    uint idle_wakeup_threshold;
    uint idle_wakeup_waitms;

    Channel[] vc = new Channel[MAX_CHANNELS];

    public Multiplexer( bool advanced, int framesize, string portname_, int portspeed_, Manager manager_ )
    {
        portname = portname_;
        portspeed = portspeed_;
        manager = manager_;

        ctx = new Context();

        ctx.mode = advanced? 1 : 0;
        ctx.frame_size = framesize;
        ctx.port_speed = portspeed_;

        ctx.user_data = this;

        ctx.at_command = at_command_fwd;
        ctx.read = read_fwd;
        ctx.write = write_fwd;
        ctx.deliver_data = deliver_data_fwd;
        ctx.deliver_status = deliver_status_fwd;
        ctx.debug_message = debug_message_fwd;
        ctx.open_channel = open_channel_fwd;
        ctx.close_channel = close_channel_fwd;
        ctx.terminate = terminate_fwd;
        ctx.response_to_test = response_to_test_fwd;

        var smk = new FsoFramework.SmartKeyFile();
        smk.loadFromFile( ABYSS_CONFIG_FILE );
        logger = FsoFramework.Logger.createFromKeyFile( smk, ABYSS_CONFIG_SECTION, ABYSS_LOGGING_DOMAIN );
        logger.setReprDelegate( repr );
        logger.debug( "created" );
    }

    ~Multiplexer()
    {
        portchannel = null;
        if ( portfd != -1 )
            Posix.close( portfd );
        logger.debug( "destructed" );
    }

    public string repr()
    {
        return "<Multiplexer 07.10-%s Framesize %d via %s @ %d>".printf( ( ctx.mode == 1? "advanced":"basic" ), ctx.frame_size, portname, ctx.port_speed );
    }

    public bool initSession()
    {
        logger.debug( "initSession()" );

        if ( !openSerial() )
            return false;

        portchannel = new IOChannel.unix_new( portfd );
        portwatch = portchannel.add_watch( IOCondition.IN, device_io_can_read );

        // make sure we're out of MUX mode
        ctx.shutdown();

        bool ok;

        if ( ctx.mode == 0 )
        {
            if (!at_command( "AT+CMUX=0\r\n" ) )
                return false;
            ok = ctx.startup( false );
        }
        else
        {
            ok = ctx.startup( true );
        }

        /*
        if (ok)
            Timeout.add_seconds( GSM_PING_SEND_TIMEOUT, protocol_ping_send_timeout );
        */

        return ok;
    }

    public void closeSession()
    {
        logger.debug( "closeSession()" );
        for ( int i = 1; i < MAX_CHANNELS; ++i )
            if ( vc != null && vc[i] != null )
                vc[i].close();

        if ( ctx != null )
            ctx.shutdown();
    }

    public void allocChannel( string name, int chan, out string path, out int allocated_channel ) throws MuxerError
    {
        int channel = chan;
        if ( chan == 0 )
        {
            // find the first free one
            int i = 1;
            while ( channel == 0 && i < MAX_CHANNELS && vc[i] != null )
                ++i;
            channel = i;
        }

        logger.debug( "allocChannel() requested for name %s, requested channel %d".printf( name, channel ) );
        // lets check whether we already have this channel
        if ( vc[channel] != null )
            throw new MuxerError.CHANNEL_TAKEN( "Channel is already taken." );

        wakeupIfNecessary();

        var ok = ctx.openChannel( channel );
        assert( ok );
        logger.debug( "0710 open channel returned result %d".printf( (int)ok ) );
        vc[channel] = new Channel( this, name, channel );

        var t = new Timer();
        t.start();

        var mc = MainContext.default();
        var ack = false;

        // FIXME: Ok, I don't like that, but until Vala supports asnyc dbus
        // on manager side, we have to live with it.
        do
        {
            mc.iteration( false );
            ack = ( vc[channel].isAcked() );
        }
        while ( !ack && t.elapsed() < GSM_OPEN_CHANNEL_ACK_TIMEOUT );

        if ( ack )
        {
            path = vc[channel].path();
            allocated_channel = channel;
        }
        else
        {
            vc[channel] = null;
            throw new MuxerError.NO_CHANNEL( "Modem does not provide this channel." );
        }
    }

    public void releaseChannel( string name ) throws MuxerError
    {
        logger.debug( "releaseChannel() requested for name %s".printf( name ) );
        bool closed = false;
        for ( int i = 1; i < MAX_CHANNELS; ++i )
        {
            if ( vc[i] != null && vc[i].name() == name )
            {
                vc[i].close();
                closed = true;
            }
        }
        if ( !closed )
            throw new MuxerError.NO_CHANNEL( "Could not find any channel with that name." );
    }

    public void setStatus( int channel, string status ) throws MuxerError
    {
        logger.debug( "setStatus() requested for channel %d".printf( channel ) );
        if ( vc[channel] == null )
            throw new MuxerError.NO_CHANNEL( "Could not find channel with that index." );

        var v24 = stringToSerialStatus( status );
        wakeupIfNecessary();
        ctx.setStatus( channel, v24 );
    }

    public void setWakeupThreshold( uint seconds, uint waitms ) throws MuxerError
    {
        if ( seconds == 0 ) /* disable */
        {
            idle_wakeup_timer = null;
        }
        else /* enable */
        {
            if ( idle_wakeup_timer == null )
            {
                idle_wakeup_timer = new Timer();
                idle_wakeup_timer.start();
            }
        }
        idle_wakeup_threshold = seconds;
        idle_wakeup_waitms = waitms;
    }

    public void testCommand( uint8[] data )
    {
        debug( "muxer: testCommand" );
        wakeupIfNecessary();
        ctx.sendTest( data, data.length );
    }

    //
    // internal helpers
    //
    public bool openSerial()
    {
        portfd = Posix.open( portname, Posix.O_RDWR | Posix.O_NOCTTY | Posix.O_NONBLOCK );
        if ( portfd == -1 )
            return false;

        Posix.fcntl( portfd, Posix.F_SETFL, 0 );

        Posix.termios termios = {};
        Posix.tcgetattr( portfd, termios );

        Posix.speed_t tspeed;

        switch ( portspeed )
        {
            case 0:
                tspeed = Posix.B0;
                break;
            case 50:
                tspeed = Posix.B50;
                break;
            case 75:
                tspeed = Posix.B75;
                break;
            case 110:
                tspeed = Posix.B110;
                break;
            case 134:
                tspeed = Posix.B134;
                break;
            case 150:
                tspeed = Posix.B150;
                break;
            case 200:
                tspeed = Posix.B200;
                break;
            case 300:
                tspeed = Posix.B300;
                break;
            case 600:
                tspeed = Posix.B600;
                break;
            case 1200:
                tspeed = Posix.B1200;
                break;
            case 1800:
                tspeed = Posix.B1800;
                break;
            case 2400:
                tspeed = Posix.B2400;
                break;
            case 4800:
                tspeed = Posix.B4800;
                break;
            case 9600:
                tspeed = Posix.B9600;
                break;
            case 19200:
                tspeed = Posix.B19200;
                break;
            case 38400:
                tspeed = Posix.B38400;
                break;
            case 57600:
                tspeed = Posix.B57600;
                break;
            case 115200:
                tspeed = Posix.B115200;
                break;
            case 230400:
                tspeed = Posix.B230400;
                break;
            case 460800:
                tspeed = Linux.Termios.B460800;
                break;
            case 500000:
                tspeed = Linux.Termios.B500000;
                break;
            case 576000:
                tspeed = Linux.Termios.B576000;
                break;
            case 921600:
                tspeed = Linux.Termios.B921600;
                break;
            case 1000000:
                tspeed = Linux.Termios.B1000000;
                break;
            case 1152000:
                tspeed = Linux.Termios.B1152000;
                break;
            case 1500000:
                tspeed = Linux.Termios.B1500000;
                break;
            case 2000000:
                tspeed = Linux.Termios.B2000000;
                break;
            case 2500000:
                tspeed = Linux.Termios.B2500000;
                break;
            case 3000000:
                tspeed = Linux.Termios.B3000000;
                break;
            case 3500000:
                tspeed = Linux.Termios.B3500000;
                break;
            case 4000000:
                tspeed = Linux.Termios.B4000000;
                break;
            default:
                assert_not_reached();
        }

        Posix.cfsetispeed( termios, tspeed );
        Posix.cfsetospeed( termios, tspeed );

        // local read
        termios.c_cflag |= (Posix.CLOCAL | Posix.CREAD);

        // 8n1
        termios.c_cflag &= ~Posix.PARENB;
        termios.c_cflag &= ~Posix.CSTOPB;
        termios.c_cflag &= ~Posix.CSIZE;
        termios.c_cflag |= Posix.CS8;

        // hardware flow control
        termios.c_cflag |= Linux.Termios.CRTSCTS;
        termios.c_iflag &= ~(Posix.IXON | Posix.IXOFF | Posix.IXANY);

        // raw input
        termios.c_lflag &= ~(Posix.ICANON | Posix.ECHO | Posix.ECHOE | Posix.ISIG);

        // raw output
        termios.c_oflag &= ~Posix.OPOST;

        // no special character handling
        termios.c_cc[Posix.VMIN] = 1;
        termios.c_cc[Posix.VTIME] = 0;
        termios.c_cc[Posix.VINTR] = 0;
        termios.c_cc[Posix.VQUIT] = 0;
        termios.c_cc[Posix.VSTART] = 0;
        termios.c_cc[Posix.VSTOP] = 0;
        termios.c_cc[Posix.VSUSP] = 0;

        Posix.tcsetattr( portfd, Posix.TCSAFLUSH, termios);

        int status = Linux.Termios.TIOCM_DTR | Linux.Termios.TIOCM_RTS;
        Posix.ioctl( portfd, Linux.Termios.TIOCMBIS, &status );

        return true;
    }

    public bool writefd( string command, int fd )
    {
        var readfds = Posix.fd_set();
        var writefds = Posix.fd_set();
        var exceptfds = Posix.fd_set();
        Posix.FD_SET( fd, writefds );
        Posix.timeval t = { 1, 0 };
        logger.debug( "writefd select for fd %d".printf( fd ) );
        int res = Posix.select( fd+1, readfds, writefds, exceptfds, t );
        if ( res < 0 || Posix.FD_ISSET( fd, writefds ) == 0 )
            return false;
        ssize_t bwritten = Posix.write( fd, command, command.size() );
        Posix.tcdrain( portfd );
        logger.debug( "writefd written %d bytes".printf( (int)bwritten ) );
        return ( (int)bwritten == command.size() );
    }

    public string readfd( int fd )
    {
        var readfds = Posix.fd_set();
        var writefds = Posix.fd_set();
        var exceptfds = Posix.fd_set();
        Posix.FD_SET( fd, readfds );
        Posix.timeval t = { 1, 0 };
        char[] buffer = new char[512];
        logger.debug( "readfd select for fd %d".printf( fd ) );
        int res = Posix.select( fd+1, readfds, writefds, exceptfds, t );
        if ( res < 0 || Posix.FD_ISSET( fd, readfds ) == 0 )
            return "";
        ssize_t bread = Posix.read( fd, buffer, 512 );
        logger.debug( "readfd read %d bytes".printf( (int)bread ) );
        return (string) buffer;
    }

    public int channelByName( string name )
    {
        for ( int i = 1; i < MAX_CHANNELS; ++i )
        {
            if ( vc[i] != null && vc[i].name() == name )
                return i;
        }
        return 0;
    }

    public string serialStatusToString( int status ) // module -> application
    {
        var sb = new StringBuilder();
        if ( ( status & SerialStatus.FC ) == SerialStatus.FC )
            sb.append( "FC ");
        if ( ( status & SerialStatus.RTC ) == SerialStatus.RTC )
            sb.append( "DSR ");
        if ( ( status & SerialStatus.RTR ) == SerialStatus.RTR )
            sb.append( "CTS ");
        if ( ( status & SerialStatus.RING ) == SerialStatus.RING )
            sb.append( "RING ");
        if ( ( status & SerialStatus.DCD ) == SerialStatus.DCD )
            sb.append( "DCD ");
        return sb.str;
    }

    public int stringToSerialStatus( string status ) // application -> module
    {
        int v24 = 0;
        var bits = status.split( " " );
        foreach( var bit in bits )
        {
            if ( bit == "DTR" )
                v24 |= SerialStatus.RTC;
            else if ( bit == "RTS" )
                v24 |= SerialStatus.RTR;
        }
        return v24;
    }

    public void clearPingResponseTimeout()
    {
        if ( pingwatch != 0 )
            Source.remove( pingwatch );
    }

    public void wakeupIfNecessary()
    {
        if ( idle_wakeup_timer != null )
        {
            var elapsed = idle_wakeup_timer.elapsed();
            if ( elapsed > idle_wakeup_threshold )
            {
                logger.debug( "channel has been idle for %.2f seconds, waking up".printf( elapsed ) );
                var wakeup = new char[] { 'W', 'A', 'K', 'E', 'U', 'P', '!' };
                ctx.sendTest( wakeup, wakeup.length );
                Thread.usleep( 1000 * idle_wakeup_waitms );
            }
        }
    }

    //
    // callbacks from channel
    //
    public void submit_data( int channel, void* data, int len )
    {
        wakeupIfNecessary();
        ctx.writeDataForChannel( channel, data, len );
    }

    public void channel_closed( int channel )
    {
        wakeupIfNecessary();
        ctx.closeChannel( channel );
        vc[channel] = null;
        manager.channelHasBeenClosed();
    }

    //
    // callbacks from 0710 core
    //
    public bool at_command( string command )
    {
        logger.debug( "0710 -> should send at_command '%s'".printf( command ) );

        while ( readfd( portfd ) != "" ) ;

        // first, send something to wake up
        writefd( "ATE0Q0V1\r\n", portfd );
        // then, read until there is nothing to read
        while ( readfd( portfd ) != "" ) ;
        // now, write the actual command
        writefd( command, portfd );
        // and read the result
        string res = "";
        string r = "";

        do
        {
            r = readfd( portfd );
            res += r;
        } while (r != "");

        logger.debug( " -> answer = %s".printf( res.strip() ) );

        if ( "OK" in res )
        {
            logger.debug( " -> answer OK" );
            return true;
        }
        else
        {
            logger.debug( " -> answer NOT ok" );
            return false;
        }
    }

    public int read( void* data, int len )
    {
        logger.debug( "0710 -> should read max %d bytes to %p".printf( len, data ) );

        var number = Posix.read( portfd, data, len );
        logger.debug( "read %d bytes from fd %d".printf( (int)number, portfd ) );
        hexdump( false, data, (int)number, logger );
        if ( number == -1 )
            logger.error( "read error fd %d: %s".printf( portfd, Posix.strerror( Posix.errno ) ) );
        return (int)number;
    }

    public bool write( void* data, int len )
    {
        logger.debug( "0710 -> should write %d bytes".printf( len ) );
        if ( idle_wakeup_timer != null )
            idle_wakeup_timer.reset();

        hexdump( true, data, len, logger );
        var number = Posix.write( portfd, data, len );
        // FIXME: necessary always?
        Posix.tcdrain( portfd );
        if ( number > 0 )
            logger.debug( "wrote %d/%d bytes to fd %d".printf( (int)number, len, portfd ) );
        else
            logger.warning( "could not write to fd %d: %s".printf( portfd, Posix.strerror( Posix.errno ) ) );
        return ( number > 0 );
    }

    public void deliver_data( int channel, void* data, int len )
    {
        logger.debug( "0710 -> deliver %d bytes for channel %d".printf( len, channel ) );
        if ( vc[channel] == null )
        {
            logger.debug( "should deliver bytes for unknown channel: ignoring" );
        }
        else
        {
            vc[channel].deliverData( data, len );
        }
        clearPingResponseTimeout();
    }

    public void deliver_status( int channel, int serial_status )
    {
        string status = serialStatusToString( serial_status );
        logger.debug( "0710 -> deliver status %d = '%s' for channel %d".printf( serial_status, status, channel ) );
        if ( vc[channel] == null )
        {
            logger.debug( ":::should deliver status for unknown channel: ignoring" );
        }
        else
        {
            if ( !vc[channel].isAcked() )
                vc[channel].acked();

            vc[channel].setSerialStatus( serial_status );
            //depending on delegate?
            //manager.Status( channel, status );
        }
        clearPingResponseTimeout();
    }

    public void debug_message( string msg )
    {
        logger.debug( "0710 -> say '%s".printf( msg ) );
    }

    public void open_channel( int channel )
    {
        logger.debug( "0710 -> open channel %d".printf( channel ) );
        logger.error( "unhandled modem side open channel command" );
    }

    public void close_channel( int channel )
    {
        logger.debug( "0710 -> close channel %d".printf( channel ) );
        var message = new char[] { '\r', '\n', '!', 'S', 'H', 'U', 'T', 'D', 'O', 'W', 'N', '\r', '\n' };
        deliver_data( channel, message, message.length );
        vc[channel] = null;
        manager.channelHasBeenClosed();
    }

    public void terminate()
    {
        logger.debug( "0710 -> terminate" );
        // FIXME send close session signal, remove muxer object
    }

    public void response_to_test( char[] data )
    {
        var b = new StringBuilder();
        foreach( var c in data )
            b.append_printf( "%c", c );
        logger.debug( "0710 -> response to test (%d bytes): %s".printf( data.length, b.str ) );
        clearPingResponseTimeout();
    }
    //
    // callbacks from glib
    //
    public bool device_io_can_read( IOChannel source, IOCondition condition )
    {
        assert( condition == IOCondition.IN );
        logger.debug( "device_io_can_read for fd %d".printf( source.unix_get_fd() ) );

        ctx.readyRead();

        return true; // call me again
    }

    public bool protocol_ping_response_timeout()
    {
        logger.warning( "\n*\n*\n* PING TIMEOUT !!!\n*\n*\n*" );
        return true;
    }

    public bool protocol_ping_send_timeout()
    {
        var data = new char[] { 'P', 'I', 'N', 'G' };
        ctx.sendTest( data, data.length );

        if ( pingwatch != 0 )
            Source.remove( pingwatch );
        pingwatch = Timeout.add_seconds( GSM_PING_RESPONSE_TIMEOUT, protocol_ping_response_timeout );
        return true;
    }
}
