/*
 * manager.vala - Manager implementation
 *
 * Authored by Michael 'Mickey' Lauer <mlauer@vanille-media.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 */

//===========================================================================
using GLib;
using CONST;

extern MainLoop loop;

namespace Gsm0710mux {

//===========================================================================
public enum ChannelType
{
    DELEGATE,
    PTY,
    SOCKET,
}

//===========================================================================
public struct ChannelInfo
{
    public int number;         /* the multiplexing channel number */
    public ChannelType type;    /* how the channel gets exposed */
    public string consumer;     /* consumer identification */
    public string transport;    /* transport identification */
    public FsoFramework.TransportReadFunc readfunc;
    public FsoFramework.TransportHupFunc hupfunc;
}

//===========================================================================
public class Manager : Object
{
    Multiplexer muxer;
    FsoFramework.SmartKeyFile config;
    FsoFramework.Logger logger;

    bool autoopen = false;
    bool autoclose = false;
    bool autoexit = true;
    string session_path = "/dev/ttySAC0";
    uint session_speed = 115200;
    bool session_mode = true;
    uint session_framesize = 64;
    uint wakeup_threshold = 0;
    uint wakeup_waitms = 0;

    uint channelsOpen;

    public Manager()
    {
        config = new FsoFramework.SmartKeyFile();
        if ( config.loadFromFile( ABYSS_CONFIG_FILE ) )
        {
            logger = FsoFramework.Logger.createFromKeyFile( config, ABYSS_CONFIG_SECTION, ABYSS_LOGGING_DOMAIN );
            logger.setReprDelegate( repr );
            logger.debug( "constructed" );

            try
            {
                autoopen = config.boolValue( "omuxerd", "autoopen", autoopen );
                autoclose = config.boolValue( "omuxerd", "autoclose", autoclose );
                autoexit = config.boolValue( "omuxerd", "autoexit", autoexit );
                session_path = config.stringValue( "session", "port", session_path );
                session_speed = config.intValue( "session", "speed", (int)session_speed );
                session_mode = config.boolValue( "session", "mode", session_mode );
                session_framesize = config.intValue( "session", "framesize", (int)session_framesize );
                wakeup_threshold = config.intValue( "device", "wakeup_threshold", (int)wakeup_threshold );
                wakeup_waitms = config.intValue( "device", "wakeup_waitms", (int)wakeup_waitms );
            }
            catch ( GLib.Error e )
            {
                warning( "Manager: config error: %s", e.message );
            }
        }
        else
        {
            warning( "Config file '%s' not present. Using defaults.", ABYSS_CONFIG_FILE );
            logger = new FsoFramework.NullLogger( ABYSS_LOGGING_DOMAIN );
        }
    }

    public string repr()
    {
        return "<Manager>";
    }

    ~Manager()
    {
        logger.debug( "destructed" );
    }

    internal void _shutdown()
    {
        logger.debug( "_shutdown" );
        if ( muxer != null )
        {
            muxer.closeSession();
            muxer = null;
        }
        if ( ( Environment.get_prgname() == "fso-abyss" ) && autoexit )
             loop.quit();
    }

    internal void channelHasBeenClosed()
    {
        channelsOpen--;
        if ( channelsOpen == 0 && autoclose )
            _shutdown();
    }

    //
    // Public API
    //

    public string getVersion()
    {
        return MUXER_VERSION;
    }

    public bool hasAutoSession()
    {
        return autoopen;
    }

    public void openSession( bool advanced, int framesize, string port, int portspeed ) throws MuxerError
    {
        logger.debug( "InitSession requested for mode %s, framesize %d, port %s @ %d".printf( advanced? "advanced":"basic", framesize, port, portspeed ) );
        if ( muxer != null )
        {
            throw new MuxerError.SESSION_ALREADY_OPEN( "Close session before opening another one." );
        }
        else
        {
            muxer = new Multiplexer( advanced, framesize, port, portspeed, this );
            if ( !muxer.initSession() )
            {
                muxer = null;
                throw new MuxerError.SESSION_OPEN_ERROR( "Can't initialize the session" );
            }
        }
    }

    public void closeSession() throws MuxerError
    {
        logger.debug( "CloseSession requested" );
        if ( muxer == null )
        {
            throw new MuxerError.NO_SESSION( "Session has to be initialized first." );
        }
        else
        {
            muxer.closeSession();
            muxer = null;
        }
    }

    public void allocChannel( ref ChannelInfo channel ) throws MuxerError
    {
        logger.debug( "Consumer %s requested channel %d".printf( channel.consumer, channel.number ) );

        // check arguments
        if ( channel.type == ChannelType.DELEGATE )
        {
            if ( channel.readfunc == null || channel.hupfunc == null )
                throw new MuxerError.INVALID_TRANSPORT( "Delegates are not set" );
        }

        if ( channel.type == ChannelType.SOCKET )
            throw new MuxerError.INVALID_TRANSPORT( "Socket not supported yet" );

        if ( autoopen && muxer == null )
        {
            logger.debug( "auto configuring..." );
            openSession( session_mode, (int)session_framesize, session_path, (int)session_speed );

            if ( wakeup_threshold > 0 && wakeup_waitms > 0 )
                setWakeupThreshold( wakeup_threshold, wakeup_waitms );
        }

        if ( channel.number < 0 )
        {
            throw new MuxerError.INVALID_CHANNEL( "Channel has to be >= 0" );
        }

        if ( muxer == null )
        {
            throw new MuxerError.NO_SESSION( "Session has to be initialized first." );
        }
        else
        {
            string transport;
            int number;
            muxer.allocChannel( channel.consumer, channel.number, out transport, out number );
            channel.transport = transport;
            channel.number = number;
            channelsOpen++;
        }
    }

    public void releaseChannel( string name ) throws MuxerError
    {
        logger.debug( "ReleaseChannel requested for name %s".printf( name ) );
        if ( muxer == null )
        {
            throw new MuxerError.NO_SESSION( "Session has to be initialized first." );
        }
        else
            muxer.releaseChannel( name );
    }

    public void setWakeupThreshold( uint seconds, uint waitms ) throws MuxerError
    {
        logger.debug( "SetWakeupThreshold to wakeup before transmitting after %u sec. of idleness, wait period = %u msec.".printf( seconds, waitms ) );
        if ( muxer == null )
        {
            throw new MuxerError.NO_SESSION( "Session has to be initialized first." );
        }
        else
            muxer.setWakeupThreshold( seconds, waitms );
    }

    public void setStatus( int channel, string status ) throws MuxerError
    {
        logger.debug( "SetStatus requested for channel %d, status = %s".printf( channel, status ) );
        if ( muxer == null )
        {
            throw new MuxerError.NO_SESSION( "Session has to be initialized first." );
        }
        else
            muxer.setStatus( channel, status );
    }

    //public signal void Status( int channel, string status );

    public void testCommand( uint8[] data ) throws MuxerError
    {
        logger.debug( "Sending %d test command bytes".printf( data.length ) );
        muxer.testCommand( data );
    }

}

} /* namespace Gsm0710mux */
