/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-----------------------------------------------------------------------
FILENAME:                valid_par.c
LIBRARY:                 libgpiv
EXTERNAL FUNCTIONS:      gpiv_valid_default_parameters
                         gpiv_valid_parameters_logic
                         gpiv_valid_read_parameters
			 gpiv_valid_test_parameter
			 gpiv_valid_print_parameters
			 gpiv_valid_fprint_parameters
                         gpiv_valid_fread_hdf5_parameters
                         gpiv_valid_fwrite_hdf5_parameters

LAST MODIFICATION DATE: $Id: valid_par.c,v 1.2 2006/01/31 13:30:14 gerber Exp $
--------------------------------------------------------------------- */
#include <gpiv.h>


static herr_t 
attr_info(hid_t loc_id, 
          const char *name, 
          GpivValidPar * piv_valid_par
          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Operator function.
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    hid_t attribute_id, atype;
    herr_t status;



/*
 * Outliers
 */
    if (strcmp(name, "auto_thresh") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_valid_par->auto_thresh); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->auto_thresh_logic = TRUE;
    }
    

    if (strcmp(name, "data_yield") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &piv_valid_par->data_yield); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->data_yield_logic = TRUE;
    }
    

    if (strcmp(name, "residu_max") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &piv_valid_par->residu_max); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->residu_max_logic = TRUE;
    }
    

    if (strcmp(name, "neighbors") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_valid_par->neighbors); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->neighbors_logic = TRUE;
    }
    

    if (strcmp(name, "res_stat") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_valid_par->res_stats); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->res_stats_logic = TRUE;
    }
    

    if (strcmp(name, "residu_type") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_valid_par->residu_type); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->residu_type_logic = TRUE;
    }
    

    if (strcmp(name, "subst_type") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_valid_par->subst_type); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->subst_type_logic = TRUE;
    }
    

    if (strcmp(name, "histo_type") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_valid_par->histo_type); 
        status = H5Aclose(attribute_id); 
        piv_valid_par->histo_type_logic = TRUE;
    }


    return 0;
}



/*
 * Public functions
 */

void
gpiv_valid_parameters_logic(GpivValidPar * piv_valid_par,
                           gboolean flag
                           )
/*-----------------------------------------------------------------------------
 * Set flag for piv_eval_par _logic */
{
    piv_valid_par->auto_thresh_logic = flag;
    piv_valid_par->data_yield_logic = flag;
    piv_valid_par->residu_max_logic = flag;
    piv_valid_par->neighbors_logic = flag;
    piv_valid_par->res_stats_logic = flag;
    piv_valid_par->residu_type_logic = flag;
    piv_valid_par->subst_type_logic = flag;
    piv_valid_par->histo_type_logic = flag;
}



void
gpiv_valid_default_parameters(GpivValidPar *piv_valid_par_default,
                              gboolean force
                              )
/*-----------------------------------------------------------------------------
 * Default parameter values
 */
{
    if (!piv_valid_par_default->auto_thresh_logic || force)
        piv_valid_par_default->auto_thresh = GPIV_VALIDPAR_DEFAULT__AUTO_THRESH;
    if (!piv_valid_par_default->data_yield_logic || force)
        piv_valid_par_default->data_yield = GPIV_VALIDPAR_DEFAULT__DATA_YIELD;
    if (!piv_valid_par_default->residu_max_logic || force)
        piv_valid_par_default->residu_max = GPIV_VALIDPAR_DEFAULT__RESIDU_MAX;
    if (!piv_valid_par_default->neighbors_logic || force)
        piv_valid_par_default->neighbors = GPIV_VALIDPAR_DEFAULT__NEIGHBORS;
    if (!piv_valid_par_default->res_stats_logic || force)
        piv_valid_par_default->res_stats = GPIV_VALIDPAR_DEFAULT__RES_STATS;
    if (!piv_valid_par_default->residu_type_logic || force)
        piv_valid_par_default->residu_type = GPIV_VALIDPAR_DEFAULT__RESIDU_TYPE;
    if (!piv_valid_par_default->subst_type_logic || force)
        piv_valid_par_default->subst_type = GPIV_VALIDPAR_DEFAULT__SUBST_TYPE;
    if (!piv_valid_par_default->histo_type_logic || force)
        piv_valid_par_default->histo_type = GPIV_VALIDPAR_DEFAULT__HISTO_TYPE;

    gpiv_valid_parameters_logic(piv_valid_par_default, TRUE);
}



void 
gpiv_valid_read_parameters(FILE * fp_par,
                           GpivValidPar * piv_valid_par,
                           int print_par
                           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Read all parameters for PIV data validation 
 *     if not defined by command line keys
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    
    char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];

    while (fgets(line, GPIV_MAX_CHARS, fp_par) != NULL) {
	 if (line[0] != '#' && line[0] != '\n' && line[0] != ' ' 
	     && line[0] != '\t') {
	      sscanf(line,"%s", par_name);

	      if (piv_valid_par->auto_thresh_logic == FALSE) {
		   piv_valid_par->auto_thresh_logic =  
			gpiv_scan_iph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Auto_Thresh:",
				      &piv_valid_par->auto_thresh, 
				      print_par, 1);
	      }

	      if (piv_valid_par->data_yield_logic == FALSE) {
		   piv_valid_par->data_yield_logic =  
			gpiv_scan_fph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Data_Yield:",
				      &piv_valid_par->data_yield, 
				      print_par, 1);
	      }

/*
 * Optional:
 */
	      if (piv_valid_par->residu_max_logic == FALSE) {
		   piv_valid_par->residu_max_logic =  
			gpiv_scan_fph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Residu_Max:",
				      &piv_valid_par->residu_max, 
				      print_par, 1);
	      }

	      if (piv_valid_par->neighbors_logic == FALSE) {
		   piv_valid_par->neighbors_logic =  
			gpiv_scan_iph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Neighbors:",
				      &piv_valid_par->neighbors, 
				      print_par, 1);
	      }

	      if (piv_valid_par->res_stats_logic == FALSE) {
		   piv_valid_par->res_stats_logic = 
			gpiv_scan_iph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Residu_Stats:", 
				      &piv_valid_par->res_stats, 
				      print_par, 1);
	      }	      

/*
 * Optional:
 */
	      if (piv_valid_par->residu_type_logic == FALSE) {
		   piv_valid_par->residu_type_logic = 
			gpiv_scan_iph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Residu_Type:", 
				      &piv_valid_par->residu_type, 
				      print_par, 1);
	      }
	      	      

	      if (piv_valid_par->subst_type_logic == FALSE) {
		   piv_valid_par->subst_type_logic = 
			gpiv_scan_iph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Subst_Type:", 
				      &piv_valid_par->subst_type, 
				      print_par, 1);
	      }

	      if (piv_valid_par->histo_type_logic == FALSE) {
		   piv_valid_par->histo_type_logic = 
			gpiv_scan_iph(GPIV_VALID_PAR_KEY, line, par_name, 
				      "Histo_Type:", 
				      &piv_valid_par->histo_type, 
				      print_par, 1);
	      }
	 }
    }

}



char *
gpiv_valid_check_parameters_read(GpivValidPar *piv_valid_par,
                                 GpivValidPar piv_valid_par_default
                                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Check out if all parameters have been read
 *     Returns: NULL on success or *err_msg on failure
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
  /* Peaklocking */


/*     if (strcmp(GPIV_VALID_PAR_KEY, "Peaklck.") == 0) { */
        /*     if (gnuplot_logic == FALSE) { */
        /* 	fprintf(stderr, "\n%s error: Gnuplot not set\n", GPIV_VALID_PAR_KEY); */
        /* 	exit(1); */
/*     } */

/* 
 * Outliers
 */
/*     } else if (strcmp(GPIV_VALID_PAR_KEY, "Err_V.") == 0) { */

    err_msg = "Using default:";

    if (piv_valid_par->auto_thresh_logic == FALSE) {
        piv_valid_par->auto_thresh = piv_valid_par_default.auto_thresh;
        gpiv_warning("%s\nValid.Auto_Thresh: %d", 
                     err_msg, piv_valid_par->auto_thresh);
    }
    
    if (piv_valid_par->data_yield_logic == FALSE) {
        piv_valid_par->data_yield = piv_valid_par_default.data_yield;
        gpiv_warning("%s\nValid.Data_Yield: %f", 
                     err_msg, piv_valid_par->data_yield);
    }
    
/*
 * Optional
 */
    if (piv_valid_par->residu_max_logic == FALSE) {
       piv_valid_par->residu_max = piv_valid_par_default.residu_max;
        gpiv_warning("%s\nValid.Residu_Max: %f", 
                     err_msg, piv_valid_par->residu_max);
    }
    
    if (piv_valid_par->neighbors_logic == FALSE) {
       piv_valid_par->neighbors = piv_valid_par_default.neighbors;
        gpiv_warning("%s\nValid.Neighbors: %d", 
                     err_msg, piv_valid_par->neighbors);
    }
    
    if (piv_valid_par->res_stats_logic == FALSE) {
        piv_valid_par->res_stats = piv_valid_par_default.res_stats;
        gpiv_warning("%s\nValid.Residu_Stats: %d", 
                     err_msg, piv_valid_par->res_stats);
    }
    
/*
 * Optional:
 */
    if (piv_valid_par->residu_type_logic == FALSE) {
        piv_valid_par->residu_type = piv_valid_par_default.residu_type;
        gpiv_warning("%s\nValid.Residu_Type: %d", 
                     err_msg, piv_valid_par->residu_type);
    }
    
    if (piv_valid_par->subst_type_logic == FALSE) {
        piv_valid_par->subst_type = piv_valid_par_default.subst_type;
        gpiv_warning("%s\nValid.Subst_Type: %d", 
                     err_msg, piv_valid_par->subst_type);
    }
    
    if (piv_valid_par->histo_type_logic == FALSE) {
        piv_valid_par->histo_type = piv_valid_par_default.histo_type;
        gpiv_warning("%s\nValid.Histo_Type: %d", 
                     err_msg, piv_valid_par->histo_type);
    }


    err_msg = NULL;
    return err_msg;
}



void 
gpiv_valid_test_parameter(GpivValidPar * piv_valid_par
                          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Testing parameters on valid values and initializing derived 
 *     variables
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    if (piv_valid_par->residu_type == GPIV_VALID_RESIDUTYPE__NORMMEDIAN) {
                piv_valid_par->residu_max = GPIV_RESIDU_MAX_NORMMEDIAN;
    }
}



void 
gpiv_valid_print_parameters(GpivValidPar piv_valid_par
                            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints parameters to stdout
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
/* Peaklocking */


/* Outliers */
    if (piv_valid_par.subst_type_logic)
	printf("%s%s %d\n", GPIV_VALID_PAR_KEY, "Subst_Type:", 
               piv_valid_par.subst_type);

    if (piv_valid_par.histo_type_logic)
	printf("%s%s %d\n", GPIV_VALID_PAR_KEY, "Histo_Type:", 
               piv_valid_par.histo_type);

    if (piv_valid_par.residu_type_logic)
	printf("%s%s %d\n", GPIV_VALID_PAR_KEY, "Residu_Type", 
               piv_valid_par.residu_type);

    if (piv_valid_par.res_stats_logic)
	printf("%s%s %d\n", GPIV_VALID_PAR_KEY, "Residu_Stats:", 
               piv_valid_par.res_stats);

    if (piv_valid_par.residu_max_logic)
	printf("%s%s %f\n", GPIV_VALID_PAR_KEY, "Residu_Max:", 
               piv_valid_par.residu_max);

    if (piv_valid_par.neighbors_logic)
	printf("%s%s %d\n", GPIV_VALID_PAR_KEY, "Neighbors:", 
               piv_valid_par.neighbors);

    if (piv_valid_par.auto_thresh_logic)
	printf("%s%s %d\n", GPIV_VALID_PAR_KEY, "Auto_thresh:", 
               piv_valid_par.auto_thresh);

    if (piv_valid_par.data_yield_logic)
	printf("%s%s %f\n", GPIV_VALID_PAR_KEY, "Data_Yield:", 
               piv_valid_par.data_yield);
}



void 
gpiv_valid_fprint_parameters(FILE * fp,
                             GpivValidPar piv_valid_par
                             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints parameters to fp_par_out
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
/* Peaklocking */


/* Outliers */
    if (piv_valid_par.subst_type_logic)
	fprintf(fp, "%s%s %d\n", GPIV_VALID_PAR_KEY, "Subst_Type:", 
                piv_valid_par.subst_type);

    if (piv_valid_par.histo_type_logic)
	fprintf(fp, "%s%s %d\n", GPIV_VALID_PAR_KEY, "Histo_Type:", 
                piv_valid_par.histo_type);

    if (piv_valid_par.residu_type_logic)
	fprintf(fp, "%s%s %d\n", GPIV_VALID_PAR_KEY, "Residu_Type", 
                piv_valid_par.residu_type);

    if (piv_valid_par.res_stats_logic) 
	fprintf(fp, "%s%s %d\n", GPIV_VALID_PAR_KEY, "Residu_Stats:", 
                piv_valid_par.res_stats);

    if (piv_valid_par.residu_max_logic)
	fprintf(fp, "%s%s %f\n", GPIV_VALID_PAR_KEY, "Residu_Max:", 
                piv_valid_par.residu_max);

    if (piv_valid_par.neighbors_logic)
	fprintf(fp, "%s%s %d\n", GPIV_VALID_PAR_KEY, "Neighbors:", 
                piv_valid_par.neighbors);

    if (piv_valid_par.auto_thresh_logic)
	fprintf(fp, "%s%s %d\n", GPIV_VALID_PAR_KEY, "Auto_thresh:", 
                piv_valid_par.auto_thresh);

    if (piv_valid_par.data_yield_logic)
	fprintf(fp, "%s%s %f\n", GPIV_VALID_PAR_KEY, "Data_Yield:", 
                piv_valid_par.data_yield);
}



char *
gpiv_valid_fread_hdf5_parameters(char * fname, 
                                 GpivValidPar * piv_valid_par
                                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads validation parameters from hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     fname:          complete file name
 *
 * OUTPUTS:
 *     piv_valid_par:  parameter structure
 *
 * RETURNS: 
 *    NULL on success or *err_msg on failure
 *-------------------------------------------------------------------------- */
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, group_par, group_id, attribute_id; 
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_VALID_FREAD_HDF5_PARAMETERS: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
/*     group_par = H5Gopen (file_id, "PARAMETERS"); */

    group_id = H5Gopen (file_id, "PIV");
    H5Aiterate(group_id, NULL, attr_info, piv_valid_par);
    status = H5Gclose (group_id);

/*    status = H5Gclose (group_par); */
   status = H5Fclose(file_id); 
   return err_msg;
}



char *
gpiv_valid_fwrite_hdf5_parameters(char *fname, 
                                  GpivValidPar * piv_valid_par
                                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes validation parameters to an existing hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *      fname:         complete file name
 *      piv_valid_par: parameter structure
 *
 * OUTPUTS:
 *
 * RETURNS: 
 *     NULL on success or *err_msg on failure
 *-------------------------------------------------------------------------- */
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, dataspace_id, group_par, group_id, attribute_id; 
    hsize_t     dims[1];
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_VALID_FWRITE_HDF5_PARAMETERS: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
/*     group_par = H5Gopen (file_id, "PARAMETERS"); */
    group_id = H5Gopen (file_id, "PIV");



    dims[0] = 1;
    dataspace_id = H5Screate_simple(1, dims, NULL);

/*
 * Outliers
 */
    if (piv_valid_par->auto_thresh_logic) {
        attribute_id = H5Acreate(group_id, "auto_thresh", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_valid_par->auto_thresh); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_valid_par->data_yield_logic) {
        attribute_id = H5Acreate(group_id, "data_yield", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &piv_valid_par->data_yield); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_valid_par->residu_max_logic) {
        attribute_id = H5Acreate(group_id, "residu_max", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &piv_valid_par->residu_max); 
        status = H5Aclose(attribute_id); 
    }


    if (piv_valid_par->neighbors_logic) {
        attribute_id = H5Acreate(group_id, "neighbors", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_valid_par->neighbors); 
        status = H5Aclose(attribute_id); 
    }


    if (piv_valid_par->res_stats_logic) {
        attribute_id = H5Acreate(group_id, "res_stats", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_valid_par->res_stats); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_valid_par->residu_type_logic) {
        attribute_id = H5Acreate(group_id, "residu_type", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_valid_par->residu_type); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_valid_par->subst_type_logic) {
        attribute_id = H5Acreate(group_id, "subst_type", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_valid_par->subst_type); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_valid_par->histo_type_logic) {
        attribute_id = H5Acreate(group_id, "histo_type", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_valid_par->histo_type); 
        status = H5Aclose(attribute_id); 
    }

    status = H5Sclose(dataspace_id);
    status = H5Gclose (group_id);
/*     status = H5Gclose (group_par); */
    status = H5Fclose(file_id); 
    return err_msg;
}
