/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002,2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.



------------------------------------------------------------------------------
FILENAME:                eval.c
LIBRARY:                 libgpiv
EXTERNAL FUNCTIONS:
                         gpiv_piv_count_pivdata_fromimage
			 gpiv_piv_select_int_point
                         gpiv_piv_interr_reg
                         gpiv_piv_isizadapt
                         gpiv_eval_write_deformed_image
			 gpiv_piv_bounds_cov
			 gpiv_piv_weight_kernel_1
			 gpiv_piv_weight_kernel_lin
			 gpiv_fread_fftw_wisdom
			 gpiv_fwrite_fftw_wisdom

LAST MODIFICATION DATE:  $Id: eval.c,v 1.22 2006/01/31 13:30:13 gerber Exp $
--------------------------------------------------------------------- */

/* #include <stdio.h> */
/* #include <math.h> */
/* #include <assert.h> */
/* #include <rfftw.h> */
/* #include <hdf5.h> */

#include <gpiv.h>


#define FFTWISDOM "gpiv_fftwis"	/* filename containg a string of wisdom for fft */
#define FFTWISDOM_INV "gpiv_fftwis_inv"	/* filename containg a string of wisdom for inverse fft */



/*-------------------------------------------------------------------------
 Levenbergh-Margardt parameter estimation constants */
/* #define MA 4 */			/* Number of parameters to be estimated by
				   marquardt fit of cov peak */
/* #define NPT 9 */			/* Numbers of points to be fitted by mrqmin */
/* #define SPREAD 0.001 */		/* multiplication factor for individual
				   variances in measuremnts, i.e. covariance
				   peak values */
/* #undef NPT */
/* #undef SPREAD */



/*
 * Local functions
 */

static gboolean
assign_img2intarr(gint ipoint_x,
                  gint ipoint_y,
                  guint16 **img_1,
                  guint16 **img_2,
                  gint int_size_1,
                  gint int_size_2,
                  gfloat **int_area_1,
                  gfloat **int_area_2,
                  gint pre_shift_row,
                  gint pre_shift_col,
                  gint nrows,
                  gint ncolumns,
                  gint int_size_0
                  )
/*-----------------------------------------------------------------------------
 * Assigns image data to the interrogation area arrays straightforward
 */
{
    gint m, n;
    gint idum = gpiv_max((int_size_2 - int_size_1) / 2, 0);
    gint arg_int1_r = ipoint_y  - int_size_1 / 2 + 1;
    gint arg_int1_c = ipoint_x  - int_size_1 / 2 + 1;
    gint arg_int2_r = ipoint_y  - int_size_2 / 2 + 1;
    gint arg_int2_c = ipoint_x  - int_size_2 / 2 + 1;

    gboolean flag_valid;

/*
 * Check if Interrogation Areas are within the image boundaries.
 * Principally arg_int1_r,c don't have to be tested as 
 * int_size_2 >= int_size_1, but has been kept to maintain generallity with the
 * other assign_img2intarr* functions
 */
    if ((arg_int1_r) >= 0
         && (arg_int1_r + int_size_1 - 1) < nrows
         && (arg_int1_c) >= 0
         && (arg_int1_c + int_size_1 - 1) < ncolumns

         && (arg_int2_r) >= 0
         && (arg_int2_r + int_size_2 - 1) < nrows
         && (arg_int2_c) >= 0
         && (arg_int2_c + int_size_2 - 1) < ncolumns) {
        flag_valid = TRUE;
    } else {
        flag_valid = FALSE;
    }


    if (flag_valid == TRUE) {
/*
 * reset int_area_1, int_area_2 values
 */
        memset(int_area_1[0], 0.0, (sizeof(gfloat)) * int_size_0 * int_size_0);
        memset(int_area_2[0], 0.0, (sizeof(gfloat)) * int_size_0 * int_size_0);

        for (m = 0; m < int_size_1; m++) {
            for (n = 0; n < int_size_1; n++) {
                int_area_1[m][n] =
                    (float) img_1[m + arg_int1_r][n + arg_int1_c];
            }
        }

        for (m = 0; m < int_size_2; m++) {
            for (n = 0; n < int_size_2; n++) {
                int_area_2[m][n] =
                    (float) img_2[m + arg_int2_r][n + arg_int2_c];
            }
        }
    }

    return flag_valid;
}



static gboolean
assign_img2intarr_central(gint ipoint_x,
                          gint ipoint_y,
                          guint16 **img_1,
                          guint16 **img_2,
                          gint int_size_1,
                          gint int_size_2,
                          gfloat **int_area_1,
                          gfloat **int_area_2,
                          gint pre_shift_row,
                          gint pre_shift_col,
                          gint nrows,
                          gint ncolumns,
                          gint int_size_0
                          )
/*-----------------------------------------------------------------------------
 * Assigns image data to the interrogation area arrays for central differential
 * scheme
 */
{
    gint m, n;
    gint idum = gpiv_max((int_size_2 - int_size_1) / 2, 0);
    gint arg_int1_r = ipoint_y - int_size_1 / 2 + 1 - pre_shift_row / 2 -
        pre_shift_row % 2;
    gint arg_int1_c = ipoint_x - int_size_1 / 2 + 1 - pre_shift_col / 2 -
        pre_shift_col % 2;
    gint arg_int2_r = ipoint_y - int_size_2 / 2 + 1 + pre_shift_row / 2;
    gint arg_int2_c = ipoint_x - int_size_2 / 2 + 1 + pre_shift_col / 2;
    gboolean flag_valid;

/*
 * Check if Interrogation Areas are within the image boundaries.
 */
     if ((arg_int1_r) >= 0
         && (arg_int1_r + int_size_1 - 1) < nrows
         && (arg_int1_c) >= 0
         && (arg_int1_c + int_size_1 - 1) < ncolumns

         && (arg_int2_r) >= 0
         && (arg_int2_r + int_size_2 - 1) < nrows
         && (arg_int2_c) >= 0
         && (arg_int2_c + int_size_2 - 1) < ncolumns) {
         flag_valid = TRUE;
     } else {
         flag_valid = FALSE;
     }


    if (flag_valid == TRUE) {
/*
 * reset int_area_1, int_area_2 values
 */
        memset(int_area_1[0], 0.0, (sizeof(gfloat)) * int_size_0 * int_size_0);
        memset(int_area_2[0], 0.0, (sizeof(gfloat)) * int_size_0 * int_size_0);

        for (m = 0; m < int_size_1; m++) {
            for (n = 0; n < int_size_1; n++) {
                int_area_1[m + idum][n + idum] =
                    (float) img_1[m + arg_int1_r][n + arg_int1_c];
            }
        }


        for (m = 0; m < int_size_2; m++) {
            for (n = 0; n < int_size_2; n++) {
                int_area_2[m][n] =
                    (float) img_2[m + arg_int2_r][n + arg_int2_c];
            }
        }

    }

    return flag_valid;
}



static gboolean
assign_img2intarr_forward(gint ipoint_x,
                          gint  ipoint_y,
                          guint16 **img_1,
                          guint16 **img_2,
                          gint int_size_1,
                          gint int_size_2,
                          gfloat **int_area_1,
                          gfloat **int_area_2,
                          gint pre_shift_row,
                          gint pre_shift_col,
                          gint nrows,
                          gint ncolumns,
                          gint int_size_0
                          )
/*-----------------------------------------------------------------------------
 * Assigns image data to the interrogation area arrays for forward differential
 * scheme
 */
{
    gint m, n;
    gint idum = gpiv_max((int_size_2 - int_size_1) / 2, 0);
    gint arg_int1_r = ipoint_y - int_size_1 / 2 + 1 + pre_shift_row + idum;
    gint arg_int1_c = ipoint_x - int_size_1 / 2 + 1 + pre_shift_col + idum;
    gint arg_int2_r = ipoint_y - int_size_2 / 2 + 1 + pre_shift_row;
    gint arg_int2_c = ipoint_x - int_size_2 / 2 + 1 + pre_shift_col;
    gboolean flag_valid;

/*
 * Check if Interrogation Areas are within the image boundaries.
 */
    if ((arg_int1_r) >= 0
        && (arg_int1_r + int_size_1 - 1) < nrows
        && (arg_int1_c) >= 0
        && (arg_int1_c  + int_size_1 - 1) < ncolumns

        && (arg_int2_r) >= 0
        && (arg_int2_r + int_size_2 - 1) < nrows
        && (arg_int2_c) >= 0
        && (arg_int2_c  + int_size_2 - 1) < ncolumns) {
        flag_valid = TRUE;
    } else {
        flag_valid = FALSE;
    }


    if (flag_valid == TRUE) {
/*
 * reset int_area_1, int_area_2 values
 */
        memset(int_area_1[0], 0.0,
               (sizeof(float)) * int_size_0 * int_size_0);
        memset(int_area_2[0], 0.0,
               (sizeof(float)) * int_size_0 * int_size_0);

        for (m = 0; m < int_size_1; m++) {
            for (n = 0; n < int_size_1; n++) {
                int_area_1[m + idum][n + idum] =
                    (float) img_1[m + arg_int1_r][n + arg_int1_c];
            }
        }


        for (m = 0; m < int_size_2; m++) {
            for (n = 0; n < int_size_2; n++) {
                int_area_2[m][n] =
                    (float) img_2[m + arg_int2_r][n + arg_int2_c];
            }
        }

    }

    return flag_valid;
}



static float
int_mean_old(guint16 **img,
         int int_size,
         int int_size_2,
         int ipoint_x,
         int ipoint_y
         )
/* ----------------------------------------------------------------------------
 * calculates mean image value from which image data are taken
 */
{
    int m = 0, n = 0, idum = gpiv_max((int_size_2 - int_size) / 2, 0);
    int int_area_sum = 0;
    float mean;

    for (m = 0; m < int_size; m++) {
        for (n = 0; n < int_size; n++) {
            int_area_sum +=
                img[m + ipoint_y - int_size_2 / 2 + 1 + idum]
                [n + ipoint_x - int_size_2 / 2 + 1 + idum];
        }
    }
    mean = int_area_sum / (int_size * int_size);

    return mean;
}



static gfloat
int_mean(gfloat **int_area,
         gint int_size
         )
/* ----------------------------------------------------------------------------
 * calculates mean value from interrogation area
 */
{
    int m = 0, n = 0;
    gfloat int_area_sum = 0;
    gfloat mean = 0.0;

    for (m = 0; m < int_size; m++) {
        for (n = 0; n < int_size; n++) {
            int_area_sum += int_area[m][n];
        }
    }
    mean = int_area_sum / (int_size * int_size);

    return mean;
}



static gboolean
int_const(gfloat **int_area,
         gint int_size
         )
/* ----------------------------------------------------------------------------
 * Tests if all values with interrogation area are equal
 */
{
    int m = 0, n = 0;
    gboolean flag = TRUE;
    gfloat val;

    val = int_area[0][0];
    for (m = 1; m < int_size; m++) {
        for (n = 1; n < int_size; n++) {
            if (int_area[m][n] != val) flag = FALSE;
        }
    }

    return flag;
}



static void
cov_min_max(Covariance * cov
            )
/* ----------------------------------------------------------------------------
 * calculates minimum and maximum in cov
 */
{
    gfloat max = -10.0e+9, min = 10.0e+9;
    gint z_rl = cov->z_rl, z_rh = cov->z_rh, z_cl = cov->z_cl,
        z_ch = cov->z_ch;
    gint i, j;

    for (i = z_rl + 1; i < z_rh - 1; i++) {
        for (j = z_cl + 1; j < z_ch - 1; j++) {
	    if (cov->z[i][j] > max) max = cov->z[i][j];
	    if (cov->z[i][j] < min) min = cov->z[i][j];
        }
    }

    cov->min = min;
    cov->max = max;
}


void
search_top(Covariance * cov,
           gint peak_act,
           gint x_corr,
           gint sweep,
           gint i_skip_act,
           gint j_skip_act,
           float *z_max,
           long *i_max,
           long *j_max
           )
/* ----------------------------------------------------------------------------
 * searches top in cov. function
 */
{
    gint h, i, j;
    gint z_rl = cov->z_rl, z_rh = cov->z_rh, z_cl = cov->z_cl,
        z_ch = cov->z_ch;

    for (h = 1; h <= peak_act + 1; h++) {
	z_max[h] = -1.0;
	i_max[h] = -2;
	j_max[h] = -3;
    }

    for (h = 1; h <= peak_act; h++) {
	for (i = z_rl + 1; i < z_rh - 1; i++) {
	    for (j = z_cl + 1; j < z_ch - 1; j++) {
		if (x_corr == 1 || (sweep == 0 || (i != i_skip_act ||
						   j != j_skip_act))) {
		    if (h == 1
			|| (h == 2
			    && (i != i_max[1] || j != j_max[1]))
			|| (h == 3
			    && (i != i_max[1] || j != j_max[1])
			    && (i != i_max[2] || j != j_max[2]))) {
			if (cov->z[i][j] > z_max[h]) {
			    if ((cov->z[i][j] >= cov->z[i - 1][j]) &&
				(cov->z[i][j] >= cov->z[i + 1][j]) &&
				(cov->z[i][j] >= cov->z[i][j - 1]) &&
				(cov->z[i][j] >= cov->z[i][j + 1])) {
				z_max[h] = cov->z[i][j];
				i_max[h] = i;
				j_max[h] = j;
			    }
			}
		    }
		}
	    }
	}
    }
}



static char *
cov_subtop(float **z,
	   long *i_max,
           long *j_max,
           float *epsi_x,
           float *epsi_y,
           int ifit,
           int peak_act
           )
/*-----------------------------------------------------------------------------
 * Calculates particle displacements at sub-pixel level
 */
{
    char *err_msg = NULL;
    double A_log, B_log, C_log;
    double min = 10e-3;
    gboolean flag = TRUE;

    if (ifit == GPIV_GAUSS) {
/*
 * sub-pixel estimator by gaussian fit
 */
/*         g_message ("cov_subtop:: z = %f", z[i_max[peak_act]][j_max[peak_act]]); */
        if (z[i_max[peak_act]][j_max[peak_act]] > min) {
            C_log = log((double) z[i_max[peak_act]][j_max[peak_act]]);
/*         g_message ("cov_subtop:: C_log"); */
        } else {
            flag = FALSE;
        }

        if (flag && z[i_max[peak_act] - 1][j_max[peak_act]] > min) {
            A_log = log((double) z[i_max[peak_act] - 1][j_max[peak_act]]);
/*             g_message ("cov_subtop:: A_log"); */
        } else {
            flag = FALSE;
        }

        if (flag && z[i_max[peak_act] + 1][j_max[peak_act]] > min) {
            B_log = log((double) z[i_max[peak_act] + 1][j_max[peak_act]]);
/*             g_message ("cov_subtop:: B_log"); */
        } else {
            flag = FALSE;
        }

        if (flag && (2 * (A_log + B_log - 2 * C_log)) != 0.0) {
            *epsi_y = (A_log - B_log) / (2 * (A_log + B_log - 2 * C_log));
        } else {
            *epsi_y = 0.0;
            peak_act = -1;
            err_msg = "epsi_y = 0.0";
/*             g_message ("cov_subtop:: %s", err_msg); */
            flag = FALSE;
        }


        if (flag && z[i_max[peak_act]][j_max[peak_act] - 1] != 0.0) {
            A_log = log((double) z[i_max[peak_act]][j_max[peak_act] - 1]);
        } else {
            flag = FALSE;
        }

        if (flag && z[i_max[peak_act]][j_max[peak_act] + 1] != 0.0) {
            B_log = log((double) z[i_max[peak_act]][j_max[peak_act] + 1]);
        } else {
            flag = FALSE;
        }

        if (flag && (2 * (A_log + B_log - 2 * C_log)) != 0.0) {
            *epsi_x = (A_log - B_log) / (2 * (A_log + B_log - 2 * C_log));
        } else {
            *epsi_x = 0.0;
            peak_act = -1;
            err_msg = "epsi_x = 0.0";
/*             g_message ("cov_subtop:: %s", err_msg); */
            flag = FALSE;
        }


    } else if (ifit == GPIV_POWER) {
/*
 * sub-pixel estimator by quadratic fit
 */
	*epsi_y = (z[i_max[peak_act] - 1][j_max[peak_act]] -
		   z[i_max[peak_act] + 1][j_max[peak_act]]) /
	    (2 * (z[i_max[peak_act] - 1][j_max[peak_act]] +
		  z[i_max[peak_act] + 1][j_max[peak_act]] -
		  2 * z[i_max[peak_act]][j_max[peak_act]]));

	*epsi_x = (z[i_max[peak_act]][j_max[peak_act] - 1] -
		   z[i_max[peak_act]][j_max[peak_act] + 1]) /
	    (2 * (z[i_max[peak_act]][j_max[peak_act] - 1] +
		  z[i_max[peak_act]][j_max[peak_act] + 1] -
		  2 * z[i_max[peak_act]][j_max[peak_act]]));


    } else if (ifit == GPIV_GRAVITY) {
/*
 * sub-pixel estimator by centre of gravity
 */
	*epsi_y = (z[i_max[peak_act] - 1][j_max[peak_act]] -
		   z[i_max[peak_act] + 1][j_max[peak_act]]) /
	    (z[i_max[peak_act] - 1][j_max[peak_act]] +
	     z[i_max[peak_act] + 1][j_max[peak_act]] +
	     z[i_max[peak_act]][j_max[peak_act]]);

	*epsi_x = (z[i_max[peak_act]][j_max[peak_act] - 1] -
		   z[i_max[peak_act]][j_max[peak_act] + 1]) /
	    (z[i_max[peak_act]][j_max[peak_act] - 1] +
	     z[i_max[peak_act]][j_max[peak_act] + 1] +
	     z[i_max[peak_act]][j_max[peak_act]]);


    } else {
	err_msg = "COV_SUBTOP: cov_subtop: No valid fit parameter";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

/*     fprintf (stderr, "LIBGPIV cov_subtop:: epsi_y = %f epsi_x = %f\n", */
/*              *epsi_y, *epsi_x); */
        return err_msg;
}



static int
cov_top(GpivEvalPar piv_eval_par,
        GpivPivData * piv_data,
	int index_y,
        int index_x,
        Covariance * cov,
        int x_corr,
        int ifit,
        int sweep,
        int last_sweep,
        int peak,
        int peak_act,
        int pre_shift_row_act,
        int pre_shift_col_act,
        int i_skip_act,
        int j_skip_act,
        gboolean *flag_subtop
        )
/* ----------------------------------------------------------------------------
 * detects location of peak and snr in correlation function
 */
{
#define INITIAL_MIN 9999.9
    char *err_msg = NULL;
    float z_min, *z_max, *z_max_next;
    int h, i, j, i_min, j_min;
    long *i_max, *j_max, *i_max_next, *j_max_next;

    int z_rl = cov->z_rl, z_rh = cov->z_rh, z_cl = cov->z_cl, z_ch = cov->z_ch;

    int ipoint_x = (int) piv_data->point_x[index_y][index_x];
    int ipoint_y = (int) piv_data->point_y[index_y][index_x];
/*     float epsi_x = 0.0, epsi_y = 0.0; */
    gboolean flag_snr = TRUE;
    gint dim = peak_act;


    i_max = gpiv_nulvector_index(1, dim + 1);
    j_max = gpiv_nulvector_index(1, dim + 1);
    z_max = gpiv_vector_index(1, dim + 1);
    i_max_next = gpiv_nulvector_index(1, dim + 2);
    j_max_next = gpiv_nulvector_index(1, dim + 2);
    z_max_next = gpiv_vector_index(1, dim + 2);

/*
 * BUGFIX: CHECK!!
 * finding a local top within the interrogation region. In case of
 * autocorrelation, exclude the first max (normally at i=0,j=0 if no
 * pre-shifting has been used), by increasing peak_act with 1 during the first
 * iteration sweep, then call it skip_act
 */

    if (sweep == 0 && x_corr == 0) {
	peak_act = peak + 1;
    } else {
	if (x_corr == 0)
	    peak_act = peak;
    }


/*     g_message("cov_top:: finding a local top 1"); */
    search_top(cov, peak_act, x_corr, sweep, i_skip_act, j_skip_act,
               z_max, i_max, j_max);

    for (h = 1; h <= peak_act + 1; h++) {
        if (z_max_next[h] == -1.0) {
            ifit = GPIV_NONE;
            flag_snr = FALSE;
/*             g_warning("cov_top:: No first top found at i = %d j = %d", */
/*                       index_y, index_x); */
        }
    }

/*
 * Define first max to be skipped if autocorr, eventually shift this
 * point with new pre-shifting values
 */


    if (x_corr == 0 && sweep == 0) {
	i_skip_act = i_max[1];
	j_skip_act = j_max[1];
    }

/* BUGFIX: don't calculate snr for the Challenge project */
/*     flag_snr = FALSE;   */

/*
 * Search next higher peak for SNR calculation
 */
    if (flag_snr) {
        search_top(cov, peak_act + 1, x_corr, sweep, i_skip_act, j_skip_act,
                   z_max_next, i_max_next, j_max_next);
    }

/*
 * Check if second top has been found
 */
    for (h = 1; h <= peak_act + 1; h++) {
        if (z_max_next[h] == -1.0) {
            flag_snr = FALSE;
        }
    }


    if (flag_snr
        && cov->z[i_max_next[peak_act + 1]][j_max_next[peak_act + 1]] != 0.0) {
        cov->snr = cov->z[i_max[peak_act]][j_max[peak_act]] - cov->min /
	(cov->z[i_max_next[peak_act + 1]][j_max_next[peak_act + 1]] - cov->min);
/*         if (last_sweep == 1) g_message("cov_top:: snr = %f",cov->snr); */
    } else {
        cov->snr = 0.0;
        piv_data->snr[index_y][index_x] = cov->snr;
/*         piv_data->peak_no[index_y][index_x] = -1; */
    }

/*
 * Searching of minimum around cov. peak_act and remove 'pedestal'
 */
    z_min = INITIAL_MIN;
    i_min = INITIAL_MIN;
    j_min = INITIAL_MIN;
    for (i = i_max[peak_act] - 1; i <= i_max[peak_act] + 1; i++) {
	for (j = j_max[peak_act] - 1; j <= j_max[peak_act] + 1; j++) {
	    if ((i >= z_rl && i <= z_rh) && (j >= z_cl && j <= z_ch)) {
		if (cov->z[i][j] < z_min) {
		    z_min = cov->z[i][j];
		    i_min = i;
		    j_min = j;
		}
	    }
	}
    }

    if (z_min <= INITIAL_MIN) {
        for (i = i_max[peak_act] - 1; i <= i_max[peak_act] + 1; i++) {
            for (j = j_max[peak_act] - 1; j <= j_max[peak_act] + 1; j++) {
/*           cov->z[i][j] = cov->z[i][j]-z_min; */
                cov->z[i][j] = cov->z[i][j] - z_min + 0.1;
            }
        }
    } else {
        ifit = GPIV_NONE;
    }

/*
 * Calculate particle displacement at integer pixel numbers or at sub-pixel
 */
    if (ifit == GPIV_NONE) {
        cov->subtop_x = 0.0;
        cov->subtop_y = 0.0;

    } else {
            if ((err_msg = cov_subtop(cov->z, i_max, j_max, &cov->subtop_x,
                                      &cov->subtop_y, ifit, peak_act))
                != NULL) {
                g_message("%s", err_msg);
                *flag_subtop = TRUE;
            }
    }

/*
 * Writing maximuma  to cov
 */
    cov->top_y = i_max[peak_act];
    cov->top_x = j_max[peak_act];

/*
 * Free memeory
 */
    gpiv_free_nulvector_index(i_max, 1, dim + 1);
    gpiv_free_nulvector_index(j_max, 1, dim + 1);
    gpiv_free_vector_index(z_max, 1, dim + 1);
    gpiv_free_nulvector_index(i_max_next, 1, dim + 2);
    gpiv_free_nulvector_index(j_max_next, 1, dim + 2);
    gpiv_free_vector_index(z_max_next, 1, dim + 2);

    return 0;
}



static
void pack_cov(float **covariance,
              Covariance * cov,
              int int_size_0
              )
/*-----------------------------------------------------------------------------
 * Packs the unordered covariance data in an ordered sequence when returning
 * from cova_nr
 */
{
    int i, j;
    int z_rnl = cov->z_rnl, z_rnh = cov->z_rnh, z_rpl = cov->z_rpl, 
        z_rph = cov->z_rph;
    int z_cnl = cov->z_cnl, z_cnh = cov->z_cnh, z_cpl = cov->z_cpl, 
        z_cph = cov->z_rph;

    for (i = z_rnl; i < z_rnh; i++) {
	for (j = z_cnl; j < z_cnh; j++) {
	    cov->z[i - int_size_0][j - int_size_0] = covariance[i][j];
	}
	for (j = z_cpl; j < z_cph; j++) {
	    cov->z[i - int_size_0][j] = covariance[i][j];
	}
    }

    for (i = z_rpl; i < z_rph; i++) {
	for (j = z_cnl; j < z_cnh; j++) {
	    cov->z[i][j - int_size_0] = covariance[i][j];
	}
	for (j = z_cpl; j < z_cph; j++) {
	    cov->z[i][j] = covariance[i][j];
	}
    }


}



static
void weight_cov(Covariance * cov,
                Covariance * w_k
                )
/*-----------------------------------------------------------------------------
 * Corrects ordered covariance data with weighting kernel
 */
{
    int i, j;
    int z_rl = w_k->z_rl, z_rh = w_k->z_rh;
    int z_cl = w_k->z_cl, z_ch = w_k->z_ch;

    for (i = z_rl; i < z_rh; i++) {
	for (j = z_cl; j < z_ch; j++) {
	    cov->z[i][j] = cov->z[i][j] / w_k->z[i][j];
	}
    }

}




/*
 * BUGFIX: Output routine OBSOLETE!!
 */
void
write_cov(int ipoint_x,
          int ipoint_y,
          int int_size,
          float **cov_area,
          int weight);


static char *
cova (int int_size,
      float **int_area_1,
      float **int_area_2,
      Covariance * cov
      )
/*-----------------------------------------------------------------------------
 * Calculates the covariance function of int_area_1 and int_area_2 by
 * means of Fast Fourier Transforms.
 * Uses rfftw_nd (FFTW).
 */
{
    char *err_msg = NULL;
    int M = GPIV_ZEROPAD_FACT * int_size, N = GPIV_ZEROPAD_FACT * int_size;
    float covariance_max, covariance_min;
    int i, j;
    float **covariance;
/*     const char *function_name = "cova"; */


    fftw_real **a, **b, **c;
    fftw_complex  *A, *B, **C;

    rfftwnd_plan p, pinv;
    fftw_real scale = 1.0 / (M * N);

    assert(int_area_1[0] != NULL);
    assert(int_area_2[0] != NULL);
    assert(cov != NULL);

/*     gpiv_imgproc_correlate(int_size, int_size, (guint16) int_area_1,  */
/*                            (guint16) int_area_2,  */
/*                            (guint16) covariance); */

    a = gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1));
    A = (fftw_complex *) &a[0][0];
    b = gpiv_fftw_real_matrix(M, 2 * (N / 2 + 1));
    B = (fftw_complex *) &b[0][0];
    c =  gpiv_fftw_real_matrix(M, N);
    C = gpiv_fftw_complex_matrix(M, N /2 + 1);
    covariance = gpiv_matrix(M, N);
    memset(covariance[0], 0, (sizeof(float)) * M * N);


/*
 * import_wisdom from string and make plans
 */
    p = rfftw2d_create_plan (M, N, FFTW_REAL_TO_COMPLEX,
                            FFTW_MEASURE | FFTW_USE_WISDOM | FFTW_IN_PLACE);
    pinv = rfftw2d_create_plan (M, N, FFTW_COMPLEX_TO_REAL,
                               FFTW_MEASURE | FFTW_USE_WISDOM);


    for (i = 0; i < M; ++i) {
	for (j = 0; j < N; ++j) {
	    a[i][j] = (double) int_area_1[i][j];
	    b[i][j] = (double) int_area_2[i][j];
	}
    }

    rfftwnd_one_real_to_complex(p, &a[0][0], NULL);
    rfftwnd_one_real_to_complex(p, &b[0][0], NULL);


/*
 * B * conjugate(A) result in correct sign of displacements!
 */
    for (i = 0; i < M; ++i) {
	for (j = 0; j < N / 2 + 1; ++j) {
	    int ij = i * (N / 2 + 1) + j;
	    C[i][j].re = (B[ij].re * A[ij].re + B[ij].im * A[ij].im) * scale;
	    C[i][j].im = (B[ij].im * A[ij].re - B[ij].re * A[ij].im) * scale;
	}
    }

/*
 * inverse transform to get c, the covariance of a and b;
 * this has the side effect of overwriting C
 */
   rfftwnd_one_complex_to_real(pinv, &C[0][0], &c[0][0]);

   rfftwnd_destroy_plan(p);
   rfftwnd_destroy_plan(pinv);


/*
 * Put the data back in a 2-dim array covariance[][]
 */
    for (i = 0; i < M; i++) {
	for (j = 0; j < N; j++) {
	    covariance[i][j] = (float) c[i][j];
	}
    }

/*
 * normalisation => correlation function
 */
    covariance_max = -10.0e+9;
    covariance_min = 10.0e+9;
    for (i = 0; i < M; i++) {
	for (j = 0; j < N; j++) {
	    if (covariance[i][j] > covariance_max)
		covariance_max = covariance[i][j];
	    if (covariance[i][j] < covariance_min)
		covariance_min = covariance[i][j];
	}
    }

    for (i = 0; i < M; i++) {
	for (j = 0; j < N; j++) {
	    covariance[i][j] = covariance[i][j] / covariance_max;
	}
    }



/*
 * Packing the unordered covariance data into the ordered array of
 * Covariance structure
 */
    pack_cov(covariance, cov, M);
    cov_min_max(cov);


/*
 * free mems
 */
    gpiv_free_matrix(covariance);
    gpiv_free_fftw_complex_matrix(C);
    gpiv_free_fftw_real_matrix(c);
    gpiv_free_fftw_real_matrix(b);
    gpiv_free_fftw_real_matrix(a);
    A = NULL;
    B = NULL;

    return err_msg;
}



/*
 * Public functions
 */

char *
gpiv_piv_count_pivdata_fromimage(GpivPivData * piv_data,
				 GpivImagePar  image_par,
				 GpivEvalPar piv_eval_par
				 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates the number of interrogation areas from the image sizes,
 *     pre-shift and area of interest
 *
 * INPUTS:
 *     image_par:      structure of image parameters
 *     piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     out_data:       output piv data from image analysis
 *
 *  RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int row, column, row_1, column_1,
	pre_shift_row_max, pre_shift_col_max, count_x = 0, count_y = 0;
    int row_max, row_min, column_max, column_min;

    int ncolumns = image_par.ncolumns;
    int nrows = image_par.nrows;

    int int_geo = piv_eval_par.int_geo;
    int row_start = piv_eval_par.row_start;
    int row_end = piv_eval_par.row_end;
    int col_start = piv_eval_par.col_start;
    int col_end = piv_eval_par.col_end;
    int int_line_col = piv_eval_par.int_line_col;
    int int_line_col_start = piv_eval_par.int_line_col_start;
    int int_line_col_end = piv_eval_par.int_line_col_end;
    int int_line_row = piv_eval_par.int_line_row;
    int int_line_row_start = piv_eval_par.int_line_row_start;
    int int_line_row_end = piv_eval_par.int_line_row_end;
    int int_point_col = piv_eval_par.int_point_col;
    int int_point_row = piv_eval_par.int_point_row;
    int int_size_1 = piv_eval_par.int_size_1;
    int int_size_2 = piv_eval_par.int_size_2;
    int int_shift = piv_eval_par.int_shift;
    int pre_shift_row = piv_eval_par.pre_shift_row;
    int pre_shift_col = piv_eval_par.pre_shift_col;

    piv_data->nx = 0;
    piv_data->ny = 0;


    row_min = gpiv_min(-int_size_1 / 2 + 1,
                       pre_shift_row - int_size_2 / 2 + 1);
    column_min = gpiv_max(-int_size_1 / 2 + 1,
                          pre_shift_col - int_size_2 / 2 + 1);
    row_max = gpiv_max(int_size_1 / 2, pre_shift_row + int_size_2 / 2);
    column_max = gpiv_max(int_size_1 / 2, pre_shift_col + int_size_2 / 2);


    if (int_geo == GPIV_POINT) {
        piv_data->nx = 1;
        piv_data->ny = 1;


/*
 * Counts number of Interrrogation Area for a single row
 */
    } else if (int_geo == GPIV_LINE_R) {
        if ((int_size_1 - int_size_2) / 2 + pre_shift_col < 0) {
            column_1 = int_line_col_start -
                ((int_size_1 - int_size_2) / 2 +
                 pre_shift_col) + int_size_1 / 2 - 1;
        } else {
                column_1 = int_line_col_start + int_size_1 / 2 - 1;
        }

        for (column = column_1; column <= int_line_col_end - column_max;
             column += int_shift) {
            count_x++;
        }

        piv_data->nx = count_x;
        piv_data->ny = 1;

/*
 * Counts number of Interrrogation Area for a single column
 */
    } else if (int_geo == GPIV_LINE_C) {
        if ((int_size_1 - int_size_2) / 2 + pre_shift_row < 0) {
            row_1 = int_line_row_start -
                ((int_size_1 - int_size_2) / 2 +
                     pre_shift_row) + int_size_1 / 2 - 1;
        } else {
            row_1 = int_line_row_start + int_size_1 / 2 - 1;
        }

        for (row = row_1; row <= int_line_row_end - row_max;
             row += int_shift) {
            count_y++;
        }

        piv_data->ny = count_y;
        piv_data->nx = 1;


/*
 * Counts number of Interrrogation Area for a Area Of Interest
 */
    } else if (int_geo == GPIV_AOI) {
	if ((int_size_1 - int_size_2) / 2 + pre_shift_row < 0) {
	    row_1 =
		row_start - ((int_size_1 - int_size_2) / 2 +
			     pre_shift_row) + int_size_1 / 2 - 1;
	} else {
	    row_1 = row_start + int_size_1 / 2 - 1;
	}
	if ((int_size_1 - int_size_2) / 2 + pre_shift_col < 0) {
	    column_1 =
		col_start - ((int_size_1 - int_size_2) / 2 +
			     pre_shift_col) + int_size_1 / 2 - 1;
	} else {
	    column_1 = col_start + int_size_1 / 2 - 1;
	}


	pre_shift_col_max = gpiv_max(pre_shift_col, 0);
	column_max =
	    gpiv_max(int_size_1 / 2, pre_shift_col + int_size_2 / 2);
	pre_shift_row_max = gpiv_max(pre_shift_row, 0);
	row_max = gpiv_max(int_size_1 / 2, pre_shift_row + int_size_2 / 2);


	for (row = row_1; row + row_max <= row_end; row += int_shift) {
	    for (column = column_1; column + column_max <= col_end;
		 column += int_shift) {
		count_x++;
	    }
	    if (count_x > piv_data->nx)
		piv_data->nx = count_x;
	    count_x = 0;
	    count_y++;
	}
	if (count_y > piv_data->ny)
	  piv_data->ny = count_y;
    } else {
        err_msg = "GPIV_PIV_COUNT_PIVDATA_FROMIMAGE: should not arrive here";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (  piv_data->nx == 0 || piv_data->ny == 0)
        err_msg = "GPIV_PIV_COUNT_PIVDATA_FROMIMAGE: line or AOI too small: nx=0 ny=0";
/*         gpiv_warning("gpiv_piv_count_pivdata_fromimage:  */
/*                      " line or AOI too small: nx=%d ny=%d\n",  */
/*                      piv_data->nx, piv_data->ny); */

    return err_msg;
}



char *
gpiv_piv_select_int_point(GpivPivData * piv_data,
			  GpivImagePar  image_par,
			  GpivEvalPar piv_eval_par
			  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Old interface; substituted by gpiv_piv_gridgen
 *     Generates grid by Calculating the positions of interrogation areas.
 *
 * INPUTS:
 *     image_par:      structure of image parameters
 *     piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     out_data:       output piv data from image analysis
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    if ((err_msg = 
         gpiv_piv_gridgen(piv_data,image_par, piv_eval_par))
         != NULL) {
        return err_msg;
    }
    return err_msg;
}


char *
gpiv_piv_interr_reg(int index_y,
                    int index_x,
                    guint16 **img_1,
                    guint16 **img_2,
                    float **int_area_1,
                    float **int_area_2,
                    Covariance * cov,
                    Covariance * w_k,
                    GpivPivData * piv_data,
                    int sweep,
                    int last_sweep,
                    GpivImagePar image_par,
                     GpivEvalPar piv_eval_par
                    )
/**----------------------------------------------------------------------------
 * DESCRIPTION:
 *     gpiv_piv_interr_reg:
 *     Interrogates an individual region
 *
 * INPUTS:
 *     @index_y:        y-index of interrogation area position
 *     @index_x:        x-index of interrogation area position
 *     @img_1:          (binary) image data of first piv image
 *     @img_2:          (binary) image data of second piv image
 *     @int_area_1:     first interrogation area
 *     @int_area_2:     second interrogation area
 *     @cov:            structure containing covariance data
 *     @w_k:            structure containing weight kerel data
 *     @sweep:	       sweep number of iterative process
 *     @last_sweep:     flag for last sweep
 *     @image_par:      structure of image parameters
 *     @piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     @piv_data:       output piv data from image analysis
 *
 * RETURNS:
 *     %char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;

    int ncolumns = image_par.ncolumns;
    int nrows = image_par.nrows;
    int x_corr = image_par.x_corr;

    int ifit = piv_eval_par.ifit;
    int int_size_1 = piv_eval_par.int_size_1;
    int int_size_2 = piv_eval_par.int_size_2;
    int old_piv = piv_eval_par.old_piv;
    int peak = piv_eval_par.peak;
    int pre_shift_row = piv_eval_par.pre_shift_row;
    int pre_shift_col = piv_eval_par.pre_shift_col;
    int int_scheme = piv_eval_par.int_scheme;

    int return_val;
    int idum = gpiv_max((int_size_2 - int_size_1) / 2, 0);
    int m = 0, n = 0;
    float int_area_1_mean = 0.0, int_area_2_mean = 0.0;

    int ipoint_x = (int) piv_data->point_x[index_y][index_x];
    int ipoint_y = (int) piv_data->point_y[index_y][index_x];

    int pre_shift_row_act = 0, pre_shift_col_act = 0;
    int peak_act = 0, i_skip_act = 0, j_skip_act = 0;

    gboolean flag_subtop = FALSE, flag_intar0 = FALSE, flag_accept = FALSE;
/*
 * Interrogation area with zero padding
 */
    int int_size_0 = GPIV_ZEROPAD_FACT * int_size_2;


/*
 * Checking for memory allocation of input variables
 */
    assert(img_1[0] != NULL);
    assert(img_2[0] != NULL);
    assert(int_area_1[0] != NULL);
    assert(int_area_2[0] != NULL);
    assert(cov != NULL);
    assert(w_k != NULL);
    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);

/*
 * uses old piv values as pre-shifts searches closest Int. Area
 */
    if (int_scheme == GPIV_ZERO_OFF_FORWARD
        || int_scheme == GPIV_ZERO_OFF_CENTRAL
        || old_piv == 1) {
	pre_shift_col_act = piv_data->dx[index_y][index_x] + pre_shift_col;
	pre_shift_row_act = piv_data->dy[index_y][index_x] + pre_shift_row;
        piv_data->dx[index_y][index_x] = 0.0;
        piv_data->dy[index_y][index_x] = 0.0;
    } else {
	pre_shift_col_act = pre_shift_col;
	pre_shift_row_act = pre_shift_row;
    }

    peak_act = peak;


/*
 * Assigning image data to the interrogation area's
 */
    if (int_scheme == GPIV_ZERO_OFF_CENTRAL ) {
        flag_accept = assign_img2intarr_central(ipoint_x, ipoint_y,
                                                img_1, img_2,
                                                int_size_1, int_size_2,
                                                int_area_1, int_area_2,
                                                pre_shift_row_act,
                                                pre_shift_col_act,
                                                nrows,
                                                ncolumns,
                                                int_size_0);

    } else if (int_scheme == GPIV_ZERO_OFF_FORWARD) {
        flag_accept = assign_img2intarr_central(ipoint_x, ipoint_y,
                                                img_1, img_2,
                                                int_size_1, int_size_2,
                                                int_area_1, int_area_2,
                                                pre_shift_row_act,
                                                pre_shift_col_act,
                                                nrows,
                                                ncolumns,
                                                int_size_0);
    } else {
        flag_accept = assign_img2intarr(ipoint_x, ipoint_y,
                                                  img_1, img_2,
                                                  int_size_1, int_size_2,
                                                  int_area_1, int_area_2,
                                                  pre_shift_row_act,
                                                  pre_shift_col_act,
                                                  nrows,
                                                  ncolumns,
                                                  int_size_0);
    }

    if (flag_accept) {
/*
 * The mean value of the image data within an interrogation area will be
 * subtracted from the data
 * BUXFIX: test on differences in estimator!
 */
/*         int_area_1_meand = int_mean_old(img_1, int_size_1, int_size_2,  */
/*                                    ipoint_x, ipoint_y); */
/*         int_area_2_meand = int_mean_old(img_2, int_size_2, int_size_2,  */
/*                                    ipoint_x, ipoint_y); */

        int_area_1_mean = int_mean(int_area_1, int_size_1);
        int_area_2_mean = int_mean(int_area_2, int_size_2);
/*         g_message(":: sweep = %d int_area_1_mean[%d][%d] = %f int_area_2_mean[%d][%d] = %f", */
/*                   sweep, */
/*                   ipoint_y, ipoint_x, int_area_1_meand,  */
/*                   ipoint_y, ipoint_x, int_area_2_meand); */
/*         g_message(":: sweep = %d int_area_1_mean[%d][%d] = %f int_area_2_mean[%d][%d] = %f", */
/*                   sweep, */
/*                   ipoint_y, ipoint_x, int_area_1_mean,  */
/*                   ipoint_y, ipoint_x, int_area_2_mean); */


/*
 * BUGFIX: this might be substituted by counting number of particles within
 * Int. Area, as done in PTV
 */
        if (int_area_1_mean == 0.0 || int_area_2_mean == 0.0
            || int_const(int_area_1, int_size_1)
            || int_const(int_area_2, int_size_2)
            ) {
/*             err_msg = "gpiv_piv_interr_reg: int_area_1/2_mean = 0.0"; */
            flag_intar0 = TRUE;
/*             return err_msg; */
            }

        for (m = 0; m < int_size_1; m++) {
            for (n = 0; n < int_size_1; n++) {
                int_area_1[m + idum ][n + idum ] -= int_area_1_mean;
            }
        }

        for (m = 0; m < int_size_2; m++) {
            for (n = 0; n < int_size_2; n++) {
                int_area_2[m][n] -= int_area_2_mean;
            }
        }

/*
 * Calculate covariance function
 */
        if (!flag_intar0) {
            if ((err_msg = cova(int_size_2, int_area_1, int_area_2,
                                cov)) != NULL) {
                /* 	    err_msg = "%s error: Unable to call cova\n", LIBNAME); */
                gpiv_warning("%s", err_msg);
                return err_msg;
            }

/*
 * Weighting covariance data with weight kernel
 */
            weight_cov(cov, w_k);

/*
 * Searching maximum peak in covariance function
 */
            if ((return_val = cov_top(piv_eval_par, piv_data, index_y, index_x,
                                      cov, x_corr, ifit, sweep, last_sweep,
                                      peak, peak_act, pre_shift_row_act,
                                      pre_shift_col_act,
                                      i_skip_act, j_skip_act,
                                      &flag_subtop)) != 0) {
                err_msg = "GPIV_PIV_INTERR_REG: Unable to call cov_top";
                gpiv_warning("%s", err_msg);
                return err_msg;
            }

/*
 * Writing values to piv_data
 */
            piv_data->dx[index_y][index_x] =
                (double) pre_shift_col_act + 
                (double) cov->top_x + cov->subtop_x
                ;
            piv_data->dy[index_y][index_x] =
                (double) pre_shift_row_act + 
                (double) cov->top_y + cov->subtop_y
                ;

/*
 * Disabled as outliers are tested after each iteration
 */
/*             if (last_sweep == 1) { */
                piv_data->snr[index_y][index_x] = cov->snr;
                piv_data->peak_no[index_y][index_x] = peak_act;
/*             } */

        }
/*
 * Check on validity of data
 */
        if (isnan(piv_data->dx[index_y][index_x]) != 0
            || isnan(piv_data->dy[index_y][index_x]) != 0
            || isnan(piv_data->snr[index_y][index_x]) != 0
            || flag_subtop
            || flag_intar0
            ) {
            piv_data->dx[index_y][index_x] = 0.0;
            piv_data->dy[index_y][index_x] = 0.0;
            piv_data->snr[index_y][index_x] = GPIV_SNR_NAN;
            piv_data->peak_no[index_y][index_x] = -1;
        }

/*
 * Uses old piv data and sets flag peak_no to -1 if:
 * for zero offsetting: 2nd Interrogation Area is out of image boundaries
 * for zero offsetting with central diff: one of the Interrogation Area's
 * is out of image  boundaries
 */
    } else {
        piv_data->dx[index_y][index_x] = piv_data->dx[index_y][index_x];
        piv_data->dy[index_y][index_x] = piv_data->dy[index_y][index_x];
        piv_data->snr[index_y][index_x] = piv_data->snr[index_y][index_x];
        piv_data->peak_no[index_y][index_x] = -1;

    }

    return err_msg;
}



void
gpiv_piv_isizadapt(GpivEvalPar piv_eval_par_src,
                   GpivEvalPar *piv_eval_par_dest,
                   gboolean *isiz_last
                   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Adjusts interrogation area sizes. For each interrogation sweep,
 *     (dest) int_size2 is halved, until it reaches (src)
 *     int_size_1. Then, isiz_last is set TRUE, which will avoid
 *     changing the interrogation sizes in next calls.
 *
 * INPUTS:
 *     piv_eval_par_src:       original parameters
 *
 * OUTPUTS:
 *     piv_eval_par_dest:      actual parameters, to be modified during sweeps
 *     isiz_last:              flag for last interrogation sweep
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{

/*     if (piv_eval_par_dest->int_size_2 == piv_eval_par_src.int_size_2) */
/*         piv_eval_par_dest->ad_int = 0; */

/*     if (piv_eval_par_dest->ad_int == 1) { */
    if ((piv_eval_par_dest->int_size_2) / 2 <= piv_eval_par_src.int_size_1) {
        *isiz_last = TRUE;
        piv_eval_par_dest->int_size_1 = 
            (piv_eval_par_src.int_size_1 - GPIV_DIFF_ISI);
        piv_eval_par_dest->int_size_2 = 
            piv_eval_par_src.int_size_1;
    } else {
        piv_eval_par_dest->int_size_1 = piv_eval_par_dest->int_size_2 / 2;
        piv_eval_par_dest->int_size_2 = piv_eval_par_dest->int_size_2 / 2;
    }

/*     } else if (piv_eval_par_src.int_scheme == GPIV_ZERO_OFF_FORWARD */
/*                || piv_eval_par_src.int_scheme == GPIV_ZERO_OFF_CENTRAL */
/*                || piv_eval_par_src.int_scheme == GPIV_IMG_DEFORM */
/*                ) { */
/*         *isiz_last = TRUE; */
/*         piv_eval_par_dest->ifit = piv_eval_par_src.ifit; */
/*         piv_eval_par_dest->int_size_1 = (piv_eval_par_src.int_size_1 -  */
/*                                          GPIV_DIFF_ISI); */
/*         piv_eval_par_dest->int_size_2 = piv_eval_par_src.int_size_1; */
/*     } */

}



char *
gpiv_eval_write_deformed_image(guint16 **img1,
                               guint16 **img2,
                               GpivImagePar image_par
                               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Stores deformed image to file system with pre defined name to TMPDIR
 *     and prints message to stdout
 *
 * INPUTS:
 *     img1:                   first image of PIV image pair
 *     img2:                   second image of PIV image pair
 *     image_par:              image parameters to be stored in header
 *     vebose:                 prints the storing to stdout
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    const gchar *tmp_dir = g_get_tmp_dir ();
    char def_img[GPIV_MAX_CHARS], def_imgh[GPIV_MAX_CHARS];
    FILE *fp;

    g_message("Writing deformed image to: %s/%s",
              tmp_dir, GPIV_DEFORMED_IMG_NAME);
    snprintf(def_imgh, GPIV_MAX_CHARS, "%s/%s%s",
             tmp_dir, GPIV_DEFORMED_IMG_NAME, GPIV_EXT_HEADER);
    if ((fp = fopen(def_imgh, "wb")) == NULL) {
        err_msg = "gpiv_eval_write_deformed_image: Failure opening for input";
        return err_msg;
    }
    gpiv_img_fprint_header(fp, image_par);
    fclose(fp);
    snprintf(def_img, GPIV_MAX_CHARS, "%s/%s%s",
             tmp_dir, GPIV_DEFORMED_IMG_NAME, GPIV_EXT_RAW_IMAGE);
    gpiv_fwrite_image(def_img, img1, img2, image_par);

    return err_msg;
}



void
gpiv_piv_bounds_cov(Covariance * cov,
                    int int_size_0,
                    GpivImagePar image_par
                    )
/* ----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Defines boundaries of array to use for pos. and neg. displacements
 *     in covariance[][] and puts the data in z[][].
 *     Uses area of -2 until int_size/2 for auto-corr
 *
 * INPUTS:
 *     int_size_0:     size of zere-padded interrogation area
 *     image_par:      image parameters
 *
 * OUTPUTS:
 *     cov:            structure containing covariance boundaries
 *
 * SOME MNENOSYNTACTICS ON ARRAY BOUNDARIES:
 *    z:               name of array
 *    r:               row
 *    c:               column
 *    p:               positive displacements; from 0 to int_size_0/4 of array
 *                     covariance
 *    n:               negative displacements; from 3*int_size_0/4 to
 *                     int_size_0 of array covariance
 *    l:               lowest index
 *    h:               highest index without n or p indicates general array
 *                     boundaries (both auto and cross correlation)
 *---------------------------------------------------------------------------*/
{

    int x_corr = image_par.x_corr;

    assert(cov != NULL);
    if (x_corr == 1) {
	cov->z_rnl = 3.0 * (int_size_0) / 4 + 1;
	cov->z_rnh = int_size_0;
	cov->z_rpl = 0;
	cov->z_rph = int_size_0 / 4;
	cov->z_cnl = 3.0 * (int_size_0) / 4.0 + 1;
	cov->z_cnh = int_size_0;
	cov->z_cpl = 0;
	cov->z_cph = int_size_0 / 4;
    } else {
	cov->z_rnl = int_size_0 - 2;
	cov->z_rnh = int_size_0;
	cov->z_rpl = 0;
	cov->z_rph = int_size_0 / 4;
	cov->z_cnl = int_size_0 - 2;
	cov->z_cnh = int_size_0;
	cov->z_cpl = 0;
	cov->z_cph = int_size_0 / 4;
    }

    cov->z_rl = cov->z_rnl - int_size_0;
    cov->z_rh = cov->z_rph;
    cov->z_cl = cov->z_cnl - int_size_0;
    cov->z_ch = cov->z_cph;

}




void
gpiv_piv_weight_kernel_1(Covariance * w_k
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Sets weight values for covariance data to one
 *
 * INPUTS:
 *
 * OUTPUTS:
 *      w_k:	        zero-padded interrogation size
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    int i, j;
    int z_rl = w_k->z_rl, z_rh = w_k->z_rh;
    int z_cl = w_k->z_cl, z_ch = w_k->z_ch;

    assert(w_k != NULL);
    for (i = z_rl; i < z_rh; i++) {
	for (j = z_cl; j < z_ch; j++) {
	    w_k->z[i][j] = 1;
	}
    }


}



void
gpiv_piv_weight_kernel_lin(Covariance * w_k,
                           int int_size_0
                           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculate linear weight kernel values for covariance data
 *
 * INPUTS:
 *
 * OUTPUTS:
 *	w_k:            Structure containing weighting data
 *      int_size_0:     zero-padded interrogation size
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    int i, j;
    int z_rnl = w_k->z_rnl, z_rnh = w_k->z_rnh, z_rpl = w_k->z_rpl,
        z_rph = w_k->z_rph;
    int z_cnl = w_k->z_cnl, z_cnh = w_k->z_cnh, z_cpl = w_k->z_cpl,
        z_cph = w_k->z_rph;

    assert(w_k != NULL);
    for (i = z_rnl; i < z_rnh; i++) {
	for (j = z_cnl; j < z_cnh; j++) {
	    w_k->z[i - int_size_0][j - int_size_0] =
		(1 - abs(z_rnh - i) / (int_size_0 / 2.0)) * (1 -
							     abs(z_cnh -
								 j) /
							     (int_size_0 /
							      2.0));
	}

	for (j = z_cpl; j < z_cph; j++) {
	    w_k->z[i - int_size_0][j] =
		(1 - abs(z_rnh - i) / (int_size_0 / 2.0)) * (1 -
							     abs(z_cpl -
								 j) /
							     (int_size_0 /
							      2.0));
	}
    }


    for (i = z_rpl; i < z_rph; i++) {
	for (j = z_cnl; j < z_cnh; j++) {
	    w_k->z[i][j - int_size_0] =
		(1 - abs(z_rpl - i) / (int_size_0 / 2.0)) * (1 -
							     abs(z_cnh -
								 j) /
							     (int_size_0 /
							      2.0));
	}

	for (j = z_cpl; j < z_cph; j++) {
	    w_k->z[i][j] = (1 - abs(z_rpl - i) / (int_size_0 / 2.0)) *
		(1 - abs(z_cpl - j) / (int_size_0 / 2.0));
	}
    }


}



void
write_cov(int int_x,
          int int_y,
          int int_size,
          float **cov_area,
          int weight
          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *   Prints covariance data
 *
 * INPUTS:
 *   int_x:
 *   int_y
 *   cov_area:
 *   int_size,
 *   int_size_0
 *
 * OUTPUTS:
 *
 * SOME MNENOSYNTACTICS OF LOCAL VARIABLES:
 *   cov_area:         name of array
 *   r:                row
 *   c:                column
 *   n:                negative displacements ; from 3/4 to 1 of int_size_0
 *   p:                positive displacements; from 0 to 1/4 of int_size_0
 *   l:                lowest index
 *   h:                highest index
 *---------------------------------------------------------------------------*/
{
    int i, j;
    int cov_area_rnl, cov_area_rnh, cov_area_rpl, cov_area_rph,
	cov_area_cnl, cov_area_cnh, cov_area_cpl, cov_area_cph;
    float weight_kernel;
    int int_size_0 = GPIV_ZEROPAD_FACT * int_size;


    cov_area_rnl = 3.0 * (int_size_0) / 4 + 1;
    cov_area_rnh = int_size_0;
    cov_area_rpl = 0;
    cov_area_rph = int_size_0 / 4;

    cov_area_cnl = 3.0 * (int_size_0) / 4 + 1;
    cov_area_cnh = int_size_0;
    cov_area_cpl = 0;
    cov_area_cph = int_size_0 / 4;


    for (i = cov_area_rnl; i < cov_area_rnh; i++) {
	for (j = cov_area_cnl; j < cov_area_cnh; j++) {
	    if (weight == 1) {
		weight_kernel =
		    (1 - abs(cov_area_rnh - i) / (int_size_0 / 2.0)) * 
		  (1 - abs(cov_area_cnh - j) / (int_size_0 / 2.0));
	    }
	}
	for (j = cov_area_cpl; j < cov_area_cph; j++) {
	    if (weight == 1) {
		weight_kernel =
		    (1 - abs(cov_area_rnh - i) / (int_size_0 / 2.0)) * 
		  (1 - abs(cov_area_cpl - j) / (int_size_0 / 2.0));
	    }
	}
    }


    for (i = cov_area_rpl; i < cov_area_rph; i++) {
	for (j = cov_area_cnl; j < cov_area_cnh; j++) {
	    if (weight == 1) {
		weight_kernel =
		    (1 - abs(cov_area_rpl - i) / (int_size_0 / 2.0)) * 
		  (1 - abs(cov_area_cnh - j) / (int_size_0 / 2.0));
	    }
	}
	for (j = cov_area_cpl; j < cov_area_cph; j++) {
	    if (weight == 1) {
		weight_kernel =
		    (1 - abs(cov_area_rpl - i) / (int_size_0 / 2.0)) * 
		  (1 - abs(cov_area_cpl - j) / (int_size_0 / 2.0));
	    }
	}
    }

}



void
gpiv_fread_fftw_wisdom(int dir
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads fftw wisdoms from file and stores into a string
 *
 * INPUTS:
 *     dir:            direction of fft; forward (+1) or inverse (-1)
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     fftw_wisdom
 *---------------------------------------------------------------------------*/
{
    char fftw[GPIV_MAX_CHARS];
    FILE *fp;
    const gchar *tmp_dir = g_get_tmp_dir ();

    assert( dir == 1 || dir == -1);

/*
 * Forward FFT or inverse FFT
 */
    if (dir == 1) {
	snprintf(fftw, GPIV_MAX_CHARS, "%s/%s", tmp_dir, FFTWISDOM);
    } else {
	snprintf(fftw, GPIV_MAX_CHARS, "%s/%s", tmp_dir, FFTWISDOM_INV);
    }

    if ((fp = fopen(fftw, "r")) != NULL) {
	fftw_import_wisdom_from_file(fp);
        fclose(fp);
    }
}



void
gpiv_fwrite_fftw_wisdom(int dir
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes fftw wisdoms to a file
 *
 * INPUTS:
 *      dir:            direction of fft; forward (+1) or inverse (-1)
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    char fftw_file[GPIV_MAX_CHARS];
    FILE *fp;
    const gchar *tmp_dir = g_get_tmp_dir ();

    assert( dir == 1 || dir == -1);

/*
 * Forward FFT or inverse FFT
 */
    if (dir == 1) {
	snprintf(fftw_file, GPIV_MAX_CHARS, "%s/%s", tmp_dir, FFTWISDOM);
    } else {
	snprintf(fftw_file, GPIV_MAX_CHARS, "%s/%s", tmp_dir, FFTWISDOM_INV);
    }

    if ((fp = fopen(fftw_file, "w")) != NULL) {
	fftw_export_wisdom_to_file(fp);
	fclose(fp);
    }
    fftw_forget_wisdom();

}
