/*
 * Galago Image API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-context.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-image.h>
#include <sys/stat.h>
#include <string.h>
#include <stdio.h>

struct _GalagoImagePrivate
{
	guchar *data;
	gsize len;
};


/**************************************************************************
 * Object/Class support
 **************************************************************************/
static void galago_image_destroy(GalagoObject *object);
static void galago_image_dbus_message_append(DBusMessageIter *iter,
											 const GalagoObject *object);
static void *galago_image_dbus_message_get(DBusMessageIter *iter);
static gchar *galago_image_dbus_get_signature(void);
static void galago_image_set_data(GalagoImage *image, const guchar *data,
								  gsize len);

static GalagoObjectClass *parent_class = NULL;

G_DEFINE_TYPE(GalagoImage, galago_image, GALAGO_TYPE_OBJECT);

static void
galago_image_class_init(GalagoImageClass *klass)
{
#if 0
	GObjectClass      *gobject_class = G_OBJECT_CLASS(klass);
#endif
	GalagoObjectClass *object_class  = GALAGO_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->dbus_interface = GALAGO_DBUS_IMAGE_INTERFACE;

	object_class->destroy             = galago_image_destroy;
	object_class->dbus_message_append = galago_image_dbus_message_append;
	object_class->dbus_message_get    = galago_image_dbus_message_get;
	object_class->dbus_get_signature  = galago_image_dbus_get_signature;

#if 0
	g_object_class_install_property(gobject_class, PROP_IMAGE_DATA,
		g_param_spec_pointer("image-data", "Image Data",
							 "The image data",
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
#endif
}

static void
galago_image_init(GalagoImage *image)
{
	image->priv = g_new0(GalagoImagePrivate, 1);
}

static void
galago_image_destroy(GalagoObject *object)
{
	GalagoImage *image = GALAGO_IMAGE(object);

	if (image->priv != NULL)
	{
		if (image->priv->data != NULL)
			g_free(image->priv->data);

		g_free(image->priv);
		image->priv = NULL;
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->destroy != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->destroy(object);
}

static void
galago_image_dbus_message_append(DBusMessageIter *iter,
								 const GalagoObject *object)
{
	guchar *data;
	gsize image_len;
	DBusMessageIter array_iter;

	galago_image_get_data(GALAGO_IMAGE(object), &data, &image_len);

	dbus_message_iter_open_container(iter, DBUS_TYPE_ARRAY,
									 DBUS_TYPE_BYTE_AS_STRING, &array_iter);
	dbus_message_iter_append_fixed_array(&array_iter, DBUS_TYPE_BYTE,
										 &data, image_len);
	dbus_message_iter_close_container(iter, &array_iter);
}

void *
galago_image_dbus_message_get(DBusMessageIter *iter)
{
	DBusMessageIter array_iter;
	GalagoImage *image;
	guchar *image_data;
	int image_len;

	dbus_message_iter_recurse(iter, &array_iter);
	dbus_message_iter_get_fixed_array(&array_iter, &image_data, &image_len);

	image = galago_image_new_from_data(image_data, image_len);

	return image;
}

gchar *
galago_image_dbus_get_signature(void)
{
	return g_strdup(DBUS_TYPE_ARRAY_AS_STRING
					DBUS_TYPE_BYTE_AS_STRING);
}


/**************************************************************************
 * Image API
 **************************************************************************/

/**
 * galago_image_new_from_file
 * @filename: The image filename.
 *
 * Creates a new image from a filename.
 *
 * Returns: The image.
 */
GalagoImage *
galago_image_new_from_file(const gchar *filename)
{
	GalagoImage *image;
	struct stat sb;
	guchar *data;
	FILE *fp;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(filename  != NULL,   NULL);
	g_return_val_if_fail(*filename != '\0',   NULL);

	if (stat(filename, &sb) != 0)
		return NULL;

	fp = fopen(filename, "rb");

	if (fp == NULL)
		return NULL;

	data = g_malloc(sb.st_size);
	fread(data, 1, sb.st_size, fp);
	fclose(fp);

	image = galago_image_new_from_data(data, sb.st_size);

	g_free(data);

	return image;
}

/**
 * galago_image_new_from_data
 * @data: The image data.
 * @len:  The image data length.
 *
 * Creates a new image from in-memory image data.
 *
 * Returns: The image.
 */
GalagoImage *
galago_image_new_from_data(const guchar *data, gsize len)
{
	GalagoImage *image;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(data != NULL,        NULL);
	g_return_val_if_fail(len  > 0,            NULL);

	image = g_object_new(GALAGO_TYPE_IMAGE, NULL);
	galago_image_set_data(GALAGO_IMAGE(image), data, len);

	return image;
}

static void
galago_image_set_data(GalagoImage *image, const guchar *data, gsize len)
{
	g_return_if_fail(image != NULL);
	g_return_if_fail(GALAGO_IS_IMAGE(image));
	g_return_if_fail(data == NULL || len > 0);
	g_return_if_fail(image->priv->data == NULL);

	if (image->priv->data != NULL)
		g_free(image->priv->data);

	image->priv->data = (guchar *)g_malloc(len);
	memcpy(image->priv->data, data, len);

	image->priv->len = len;
}

/**
 * galago_image_get_data
 * @image:    The image.
 * @ret_data: The returned image data.
 * @ret_len:  The returned image data length.
 *
 * This function returns the image's data and the length of the data.
 */
void
galago_image_get_data(const GalagoImage *image, guchar **ret_data,
					  gsize *ret_len)
{
	g_return_if_fail(image != NULL);
	g_return_if_fail(GALAGO_IS_IMAGE(image));
	g_return_if_fail(ret_data != NULL || ret_len != NULL);

	if (ret_data != NULL)
		*ret_data = image->priv->data;

	if (ret_len != NULL)
		*ret_len = image->priv->len;
}
