/***************************************************************************
 *   Copyright (C)                                                         *
 *   2004 Kraschinski Arnold <arnold.kra67@gmx.de>                         *
 *   2005 Michael Goettsche <michael.goettsche@kdemail.net>                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.             *
 ***************************************************************************/

#include "kclock.h"

#include <qsize.h>
#include <qpainter.h>
#include <qtimer.h>
#include <qbrush.h>
#include <qcolor.h>
#include <qfont.h>
#include <qfontmetrics.h>

#include <kstandarddirs.h>
#include <kdebug.h>

KClock::KClock(QWidget *parent, const char *name)
    : QWidget(parent, name), m_timer(0), m_running(false), m_seconds(0)
{
    m_clockPic.load(locate("data", "kdeedu/pics/clock.png"));
    if (m_clockPic.isNull())
        kdDebug() << "The clock pixmap is missing." << endl;
    setBackgroundMode(NoBackground);
    setMinimumSize(QSize(TIMER_WIDTH, TIMER_HEIGHT));
}


KClock::~KClock() {
    if (m_timer != 0) {
        delete m_timer;
        m_timer = 0;
    }
}

void KClock::setTimeLeft(int seconds) {
    m_seconds = seconds;
    if (m_seconds > 0)
        m_up = false;
    else
        m_up = true;
    update();
}

void KClock::start() {
    if (m_timer == 0) {
        m_timer = new QTimer(this, "QTIMER");
        connect(m_timer, SIGNAL(timeout()), this, SLOT(slotTimeout()));
        m_timer->start(1000);
        m_running = true;
    }
    emit timerStarted();
}

void KClock::stop() {
    if (m_timer != 0) {
        m_timer->stop();
        delete m_timer;
        m_timer = 0;
        m_running = false;
    }
    emit timerStopped();
}

void KClock::setPaused(bool paused)
{
    m_running = paused;
    if (paused)
        m_timer->stop();
    else
        m_timer->start(1000);
}

void KClock::paintEvent(QPaintEvent* ) {
    QPixmap area(size());
    QPainter p;
    p.begin(&area, this);
    p.setBackgroundColor(QColor(0Xffffc5));
    p.eraseRect(0, 0, width(), height());

    int margin = 6;
    int offsetx = (width() - m_clockPic.width()) / 2;
    int offsetPie(6);
    // 5760 is a full circle in the qt-classes 96 = 5760 / 60
    int secArc = (96 * m_seconds) % 5760;
    int minArc = (96 * (m_seconds / 60)) % 5760;
    if (minArc < secArc) {
        QBrush brush1(yellow);
        p.setBrush(brush1); 
        p.drawPie(offsetx + offsetPie, margin + 5, m_clockPic.width() - 14, m_clockPic.height() - 14, 1440 - secArc, secArc);
        if (minArc > 0) {
            QBrush brush2(QColor(237, 162, 42));
            p.setBrush(brush2);
            p.drawPie(offsetx + offsetPie, margin + 5, m_clockPic.width() - 14, m_clockPic.height() - 14, 1440 - minArc, minArc);
        }
    } else {
        QBrush brush2(QColor(255, 0, 0));
        p.setBrush(brush2);
        p.drawPie(offsetx + offsetPie, margin + 5, m_clockPic.width() - 14, m_clockPic.height() - 14, 1440 - minArc, minArc);
        QBrush brush1(QColor(237, 162, 42));
        p.setBrush(brush1); 
        p.drawPie(offsetx + offsetPie, margin + 5, m_clockPic.width() - 14, m_clockPic.height() - 14, 1440 - secArc, secArc);
    }

    // offsetx + 2 because of the shahow of the clock. Without the 2 it looks as
    // if the clock isn't in the center.    
    p.drawPixmap(offsetx + 2, margin, m_clockPic);

    p.setPen( QColor(0x0000aa) );
    QFont font = p.font();
    font.setPixelSize(24);
    p.setFont(font);

    int s = m_seconds;
    QString st("");
    if (s >= 3600) {
      int h = s / 3600;
      s = s - h * 3600;
      st = "00" + st.setNum(h);
      st = st.right(2) + ":";
    }
    int m = s / 60;
    s = s - m * 60;
    QString n;
    n.setNum(m);
    n = "00" + n;
    st = st + n.right(2) + ":";
    n.setNum(s);
    n = "00" + n;
    st = st + n.right(2);
    int stWidth = p.fontMetrics().width(st);
    p.drawText((width() - stWidth) / 2, m_clockPic.height() + 6 * margin, st);
    p.end();
    bitBlt( this, 0, 0, &area);
}

void KClock::slotTimeout() {
    if (m_up)
        m_seconds++;
    else {
        m_seconds--;
        if (m_seconds == 0)
            emit timeout();
    }
    repaint();
}


#include "kclock.moc"
