/*--------------------------------------------------------------------------+
$Id: SimulinkUtilsTest.java 26285 2010-02-18 11:22:54Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.simulink.util;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import edu.tum.cs.commons.assertion.PreconditionException;
import edu.tum.cs.simulink.builder.SimulinkModelBuildingException;
import edu.tum.cs.simulink.model.SimulinkModel;
import edu.tum.cs.simulink.model.stateflow.StateflowChart;
import edu.tum.cs.simulink.model.stateflow.StateflowState;
import edu.tum.cs.simulink.testutils.SimulinkTestBase;

/**
 * Test class for {@link SimulinkUtils}.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 26285 $
 * @levd.rating GREEN Hash: 5C43D73221D2D8A158ACC23FFAE6C2EC
 */
public class SimulinkUtilsTest extends SimulinkTestBase {

	/** Test {@link SimulinkUtils#splitSimulinkId(String)} */
	public void testSplitSimulinkId() {
		assertSplit("hallo/ccsm", "hallo", "ccsm");
		assertSplit("hallo/ccsm/tum", "hallo", "ccsm", "tum");
		assertSplit("hallo/ccsm//tum", "hallo", "ccsm/tum");
		assertSplit("sl_subsys_fcncallerr7/f/Unit Delay",
				"sl_subsys_fcncallerr7", "f", "Unit Delay");
	}

	/** Test {@link SimulinkUtils#createSimulinkId(Iterable)}. */
	public void testCreateSimulinkId() {
		String[] nameArray = { "name1", "name2", "name3" };
		List<String> names = new ArrayList<String>();
		Collections.addAll(names, nameArray);
		assertEquals("name1/name2/name3", SimulinkUtils.createSimulinkId(names));

		names.add("test/name");
		assertEquals("name1/name2/name3/test//name", SimulinkUtils
				.createSimulinkId(names));

		try {
			names.add("name/");
			SimulinkUtils.createSimulinkId(names);
			fail();
		} catch (PreconditionException ex) {
			// expected
		}

		try {
			names.add("/name");
			SimulinkUtils.createSimulinkId(names);
			fail();
		} catch (PreconditionException ex) {
			// expected
		}
	}

	/**
	 * Test for {@link SimulinkUtils#getStateName(StateflowState)} and
	 * {@link SimulinkUtils#getFQStateName(StateflowState)}.
	 */
	public void testGetStateName() throws FileNotFoundException,
			SimulinkModelBuildingException {
		SimulinkModel model = loadModel("internal_sf_tictacflow.mdl");
		System.out.println(model.getStateflowMachine().getCharts());
		StateflowChart chart = model.getStateflowMachine().getChart(
				"internal_sf_tictacflow/simulink io/scheduled stuff");
		StateflowState state = (StateflowState) getStateflowNode("333", chart);
		assertEquals("sleep", SimulinkUtils.getStateName(state));
		assertEquals(
				"internal_sf_tictacflow/simulink io/scheduled stuff/sleep",
				SimulinkUtils.getFQStateName(state));
		state = (StateflowState) getStateflowNode("335", state);
		assertEquals("asleep", SimulinkUtils.getStateName(state));
		assertEquals(
				"internal_sf_tictacflow/simulink io/scheduled stuff/sleep.asleep",
				SimulinkUtils.getFQStateName(state));
	}

	/** Assert correct split. */
	private void assertSplit(String id, String... expectedNames) {
		List<String> actualNames = SimulinkUtils.splitSimulinkId(id);
		assertEquals(expectedNames.length, actualNames.size());
		for (int i = 0; i < actualNames.size(); i++) {
			assertEquals(expectedNames[i], actualNames.get(i));
		}
	}
}