// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.Collections;
using System.Collections.Generic;

namespace Anculus.Core
{
	public class SortedCollection<T> : ICollection<T> where T : IComparable<T>
	{
		private List<T> _list;
		private SortOrder _sortOrder;

		public SortedCollection ()
			: this (SortOrder.Ascending)
		{
		}
		
		public SortedCollection (int capacity)
			: this (SortOrder.Ascending, capacity)
		{
		}
		
		public SortedCollection (SortOrder sortOrder)
		{
			_list = new List<T> ();
			_sortOrder = sortOrder;
		}
		
		public SortedCollection (SortOrder sortOrder, int capacity)
		{
			if (capacity < 0)
				throw new ArgumentOutOfRangeException ("capacity");
			
			_sortOrder = sortOrder;
			_list = new List<T> (capacity);
		}
		
		public SortedCollection (IEnumerable<T> enumerable)
			: this ()
		{
			if (enumerable == null)
				throw new ArgumentNullException ("enumerable");
			
			_list = new List<T> ();
			foreach (T item in enumerable)
				Add (item);
		}
		
		public SortOrder SortOrder
		{
			get { return _sortOrder; }
			set {
				if (_sortOrder != value) {
					_sortOrder = value;
					Sort.QuickSort<T> (_list, _sortOrder);
				}
			}
		}
		
		public int Count
		{
			get { return _list.Count; }
		}
		
		public bool IsReadOnly
		{
			get { return false; }
		}
		
		public T this [int index]
		{
			get { return _list[index]; }
		}
		
		public void Add (T item)
		{
			if (item == null)
				throw new ArgumentNullException ("item");
			
			Insert.SortedInsert<T> (_list, _sortOrder, item);
		}
		
		public void AddRange (IEnumerable<T> items)
		{
			foreach (T item in items)
				Add (item);
		}
		
		public bool Remove (T item)
		{
			if (item == null)
				throw new ArgumentNullException ("item");
			
			return _list.Remove (item);
		}
		
		public bool Contains (T item)
		{
			if (item == null)
				throw new ArgumentNullException ("item");
			
			return _list.BinarySearch (item) >= 0;
		}
		
		public void Clear ()
		{
			_list.Clear ();
		}
		
		public void CopyTo (T[] array, int arrayIndex)
		{
			_list.CopyTo (array, arrayIndex);
		}
		
		public IEnumerator<T> GetEnumerator ()
		{
			return _list.GetEnumerator ();
		}
		
		IEnumerator IEnumerable.GetEnumerator ()
		{
			return ((IEnumerable)_list).GetEnumerator ();
		}
	}
}
