/*  Lziprecover - Data recovery tool for lzipped files
    Copyright (C) 2009, 2010, 2011 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _FILE_OFFSET_BITS 64

#include <cerrno>
#include <climits>
#include <cstdio>
#include <cstring>
#include <string>
#include <vector>
#include <stdint.h>
#include <unistd.h>
#include <sys/stat.h>

#include "lzip.h"


int repair_file( const std::string & input_filename,
                 const std::string & output_filename, const bool force )
  {
  struct stat in_stats;
  const int infd = open_instream( input_filename, &in_stats, true, true );
  if( infd < 0 ) return 1;
  const long long isize = lseek( infd, 0, SEEK_END );
  if( isize < 0 )
    { show_error( "Input file is not seekable", errno ); return 1; }
  if( isize < min_member_size )
    { show_error( "Input file is too short." ); return 2; }
  if( !verify_single_member( infd, isize ) ) return 2;
  if( lseek( infd, 0, SEEK_SET ) < 0 )
    { show_error( "Seek error in input file", errno ); return 1; }

  long long failure_pos = 0;
  if( try_decompress( infd, isize, &failure_pos ) )
    {
    if( verbosity >= 1 )
      std::printf( "Input file has no errors. Recovery is not needed.\n" );
    return 0;
    }
  if( failure_pos >= isize - 8 ) failure_pos = isize - 8 - 1;
  if( failure_pos < File_header::size )
    { show_error( "Can't repair error in input file." ); return 2; }

  if( lseek( infd, 0, SEEK_SET ) < 0 )
    { show_error( "Seek error in input file", errno ); return 1; }

  const int outfd = open_outstream_rw( output_filename, force );
  if( outfd < 0 ) { close( infd ); return 1; }
  if( !copy_file( infd, outfd ) )
    cleanup_and_fail( output_filename, outfd, 1 );

  const long long min_pos =
    std::max( (long long)File_header::size, failure_pos - 1000 );
  bool done = false;
  for( long long pos = failure_pos; pos >= min_pos; --pos )
    {
    if( verbosity >= 1 )
      {
      std::printf( "Trying position %lld \r", pos );
      std::fflush( stdout );
      }
    uint8_t byte;
    if( lseek( outfd, pos, SEEK_SET ) < 0 ||
        readblock( outfd, &byte, 1 ) != 1 )
      { show_error( "Error reading output file", errno );
        cleanup_and_fail( output_filename, outfd, 1 ); }
    for( int i = 0; i < 255; ++i )
      {
      ++byte;
      if( lseek( outfd, pos, SEEK_SET ) < 0 ||
          writeblock( outfd, &byte, 1 ) != 1 ||
          lseek( outfd, 0, SEEK_SET ) < 0 )
        { show_error( "Error writing output file", errno );
          cleanup_and_fail( output_filename, outfd, 1 ); }
      if( try_decompress( outfd, isize ) )
        { done = true; break; }
      }
    if( done ) break;
    ++byte;
    if( lseek( outfd, pos, SEEK_SET ) < 0 ||
        writeblock( outfd, &byte, 1 ) != 1 )
      { show_error( "Error writing output file", errno );
        cleanup_and_fail( output_filename, outfd, 1 ); }
    }
  if( verbosity >= 1 ) std::printf( "\n" );

  if( close( outfd ) != 0 )
    {
    show_error( "Error closing output file", errno );
    cleanup_and_fail( output_filename, -1, 1 );
    }
  if( !done )
    {
    show_error( "Error is larger than 1 byte. Can't repair input file." );
    cleanup_and_fail( output_filename, -1, 2 );
    }
  if( verbosity >= 1 )
    std::printf( "Copy of input file repaired successfully.\n" );
  return 0;
  }
