# Copyright (C) 2008-2010 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

import gtk

import logging
log = logging.getLogger(__name__)

from os.path import isfile

from lottanzb.config import ConfigSection
from lottanzb.hellaconfig import HellaConfig
from lottanzb.modes.base import Mode as _Mode
from lottanzb.util import gproperty, _

class Config(ConfigSection):
    """
    Config section class with litte performance optimization.
    
    Don't invoke HellaConfig.locate() in the declaration of the `config_file`
    property because it requires quite an amount of unnecessary hard drive
    accesses. Only invoke it if the property didn't exist in the configuration
    file before that.
    """

    config_file = gproperty(type=str)
    
    def __init__(self, *args, **kwargs):
        self._config_file_set = False
        
        ConfigSection.__init__(self, *args, **kwargs)
        
        if not self._config_file_set:
            self.config_file = HellaConfig.locate()
     
    def set_property(self, key, value):
        if key == "config_file":
            self._config_file_set = True
        
        ConfigSection.set_property(self, key, value)

class Mode(_Mode):
    title = _("Local front-end mode")
    short = _("Local front-end")
    description = _("Monitor a HellaNZB daemon on this computer, which is "
        "already configured and running.")
    icon = gtk.STOCK_CONNECT
    disconnect_message = "%s\n\n%s" % (
        _Mode.disconnect_message,
        _("Please make sure HellaNZB is running and the correct "
          "configuration file is selected.")
    )
    
    hella_config = None
    
    def __init__(self, config):
        _Mode.__init__(self, config)
        
        # LottaNZB 0.4 or older also proposed ~/.lottanzb/hellanzb.conf as a
        # possible HellaNZB configuration. However, this one is only meant to
        # be used in stand-alone mode. Due to the change of directory location
        # (now .config/lottanzb) this files is unlikely to exist any longer.
        # It doesn't make any sense to have it to leave it in the configuration
        # file.
        config_file = self.config.config_file
        
        if ".lottanzb" in config_file and not isfile(config_file):
            self.config.config_file = HellaConfig.locate()
    
    def enter(self):
        if not self.config.config_file:
            raise ValueError
        
        self.hella_config = HellaConfig(self.config.config_file, read_only=True)
        self.hella_config.load()
        
        _Mode.enter(self)
    
    def get_connection_args(self):
        return (
            self.hella_config.XMLRPC_SERVER,
            self.hella_config.XMLRPC_PORT,
            self.hella_config.XMLRPC_PASSWORD
        )
