/*
 * capture.c: capture data from stdout and stderr
 *
 *
 * Copyright (C) 2007 Michael Brunner <mibru@gmx.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * General Public License for more details.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <termios.h>

#include <sys/socket.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <pthread.h>

#include "logapp.h"
#include "configuration.h"
#include "logfile.h"
#include "capture.h"

static int (*handle_stream)(pipe_t* pipe);

int handle_cbstream(pipe_t* pipe);

void reset_console(void)
{
	if (!config.dumbterm) {
		write(app.pstdout->cfhno, app.pstdout->escreset,
		      strlen(app.pstdout->escreset));
		write(app.pstderr->cfhno, app.pstderr->escreset,
		      strlen(app.pstderr->escreset));
	}

	if (app.ptytermios_bak != NULL)
		tcsetattr(STDIN_FILENO, TCSADRAIN, app.ptytermios_bak);

	tcdrain(STDOUT_FILENO);
	tcdrain(STDERR_FILENO);
}

int print_stream(pipe_t *pipe)
{
	static pipe_t *pipe_old;
	static unsigned	offset;
	char* buf;
	unsigned ccount;
	unsigned esclength = 0;
	int regmatch = 0;

	if (pipe->bfill < 1)
		return 0;


	if (pipe->regexp) {
		if (!regexec(&pipe->preg, pipe->buf, strlen(pipe->buf),
			     NULL, 0)) {
			regmatch = 1;
		}
	}

	/* Change font style if we are switching the stream */
	if (((void*)pipe_old != (void*)pipe)||(regmatch)) {
		offset = 0;
		buf = pipe->esccolor;
		esclength = pipe->buf - pipe->esccolor;
		if (!regmatch) {
			*pipe->bgesccolor = pipe->bgcol + '0';
			pipe_old = pipe;
		} else {
			*pipe->bgesccolor = pipe->regbgcol + '0';
			pipe_old = NULL;
		}
	} else {
		buf = pipe->buf;
	}

	/* Clip output if configured */
	if (pipe->eclip) {
		if (offset >= pipe->clip)
			return 0;

		if ((pipe->bfill + offset) > pipe->clip) {
			ccount = pipe->clip - offset;
			pipe->buf[ccount] = '\n';
			ccount++;
			offset = 0;
		} else {
			ccount = pipe->bfill;
			if (pipe->buf[ccount - 1] == '\n') {
				offset = 0;
			} else {
				offset += ccount;
			}
		}
	} else {
		ccount = pipe->bfill;
	}

	if (regmatch) {
		/* Prevent colorizing the next line if regexp matches */
		if (pipe->buf[ccount-1] == '\n') {
			int rlen = strlen(pipe->escreset) + 1;
			strcpy(&pipe->buf[ccount-1],  pipe->escreset);
			strcpy(&pipe->buf[ccount-1 + rlen],  "\n");
			esclength+=rlen;

		}
	}

	write(pipe->cfhno, buf, ccount + esclength);

	pipe->bfill = 0;

	return 0;
}

int print_summary(void)
{
	time_t sec;
	unsigned int executiontime = 0;

	time(&sec); 
	
	if ((app.starttime > -1)&&(app.starttime <= sec)) {
		executiontime = (unsigned int)(sec - app.starttime);
	} else {
		error("Error calculating execution time\n");
	}
		
	fprintf(stderr, "\nLogapp exited after %us;"
		" %u stdout, %u stderr lines;",
		executiontime, app.pstdout->linecount, app.pstderr->linecount);

       if (WIFEXITED(app.exit_state)) {
			 fprintf(stderr, " exit state %u\n",
				WEXITSTATUS(app.exit_state));
	} else if (WIFSIGNALED(app.exit_state)) {
			 fprintf(stderr, " signal %d\n",
				WTERMSIG(app.exit_state));
	} else {
			 fprintf(stderr, " unknown exit state\n");
	}

       /* Show a message if we changed the logfile name */
	if ((logfile.appendnr && config.warnlogfilelock)
	    || config.printlogname)
		fprintf(stderr, "Logapp logfile is: %s\n", logfile.name);

	return 0;
}

int read_pipe(pipe_t *pipe)
{
	int i = 0;
	
	if (pipe->state > 0)
		pipe->bfill = 0;
	while(1) {
		i = read(pipe->fh, pipe->buf + pipe->bfill, 1);
		if (i < 0) {
			return i;
		}

		if (pipe->buf[pipe->bfill] == '\0'
		    || pipe->buf[pipe->bfill] == '\n'
		    || !(i < pipe->blen)
		    || ((pipe->buf[pipe->bfill] == 27) && config.detectescape)
		    || (i == 0)) {
			if ((pipe->buf[pipe->bfill] == '\n')
			    && (i > 0)) {
				pipe->bfill++;
				pipe->linecount++;
			} else if ((pipe->buf[pipe->bfill] == 27)
				   && (config.detectescape) && (i > 0)) {
				pipe->bfill++;
				/* Switch to charbased stream handling as we
				 * just detected an escape sequence */
				handle_stream = handle_cbstream;
			}
			pipe->buf[pipe->bfill] = '\0';
			pipe->buf[pipe->blen] = '\0';
			return(pipe->bfill);
		};
		if (pipe->bfill < pipe->blen)
			pipe->bfill++;
	}

	return 0;
}

int prepare_buffer(pipe_t* pipe)
{
	if (pipe->blen) {
		/* length = buffer length + escape prefix + escape postfix */
		int len = pipe->blen + strlen(pipe->esccolor)
			+ strlen(pipe->escreset) + 1;

		if (!(pipe->buf = (char*)malloc(len))) {
			error("out of memory trying to allocate %d byte "
			      "%s buffer\n", pipe->blen + 1, pipe->name);
			return -1;
		}

		/* We put the color escape sequence directly before the line
		 * buffer memory to be able to squeeze it all out at once.
		 * The reserved space at the end is for a console reset
		 * sequence to end the regexp match background color */
		strcpy(pipe->buf, pipe->esccolor);
		free(pipe->esccolor);
		pipe->bgesccolor = pipe->buf + (pipe->bgesccolor
			- pipe->esccolor); /* fixup bg color pointer */
		pipe->esccolor = pipe->buf;
		pipe->buf += strlen(pipe->esccolor);
	};

	return 0;
}

int handle_cbstream(pipe_t* pipe)
{
	if ((pipe->bfill = read(pipe->fh, pipe->buf, pipe->blen)) > 0) {
		write(pipe->cfhno, pipe->buf, pipe->bfill);

		if (logfile_write(app.logfile, pipe)) {
			error("error writing logfile\n");
			return -1;
		}
	} else {
		/* Pipe didn't provide data - exit if
		 * application isn't active anymore */
		if (!app.active)
			return 1;
		return 2;
	}

	return 0;
}

int handle_lbstream(pipe_t* pipe)
{
	if ((pipe->state = read_pipe(pipe)) > 0) {
		if (logfile_write(app.logfile, pipe)) {
			error("error writing logfile\n");
			return -1;
		}
		if (print_stream(pipe)) {
			error("error printing %s output\n", pipe->name);
			return -1;
		}
	} else {
		/* Pipe didn't provide data - exit if
		 * application isn't active anymore */
		if (!app.active)
			return 1;
		return 2;
	}

	return 0;
}

void* capture_thread(void* do_pipe)
{
	fd_set fd_pipe;
	pipe_t* pipe = (pipe_t*) do_pipe;


	FD_ZERO(&fd_pipe);
	FD_SET(pipe->fh, &fd_pipe);

	if (pipe->charbased) {
		handle_stream = handle_cbstream;
	} else {
		handle_stream = handle_lbstream;
	}

	while(1) {
		if (select(pipe->fh + 1, &fd_pipe, NULL, NULL, NULL) == 1) {
			switch (handle_stream(pipe)) {
				case -1:	reset_console();
					 	exit(EXIT_FAILURE);
					 	break;
				case 1:		if (CONFIG_USE_THREADS)
							pthread_exit(NULL);
						break;
				case 2:		usleep(10000);
						break;
				default:	break;
			}
		} else {
			if (CONFIG_USE_THREADS)
				pthread_exit(NULL);
		}
	}
}

int capture_loop(pipe_t* pipe_stdout, pipe_t* pipe_stderr)
{
	fd_set	rdfs;
	long	flags = 0;
	int	do_loop = 1;

	int (*handle_stdout_stream)(pipe_t* pipe);
	int (*handle_stderr_stream)(pipe_t* pipe);

	if (pipe_stdout->charbased) {
		handle_stdout_stream = handle_cbstream;
	} else {
		handle_stdout_stream = handle_lbstream;
	}

	if (pipe_stderr->charbased) {
		handle_stderr_stream = handle_cbstream;
	} else {
		handle_stderr_stream = handle_lbstream;
	}

	fcntl(pipe_stdout->fh, F_GETFL, flags);
	fcntl(pipe_stdout->fh, F_SETFL, flags | O_NONBLOCK);
	fcntl(pipe_stderr->fh, F_GETFL, flags);
	fcntl(pipe_stderr->fh, F_SETFL, flags | O_NONBLOCK);

	while (do_loop)
	{
		FD_ZERO(&rdfs);
		FD_SET(pipe_stdout->fh, &rdfs);
		FD_SET(pipe_stderr->fh, &rdfs);

		if (select(FD_SETSIZE, &rdfs, NULL, NULL, NULL) > 0)
		{
			if (FD_ISSET(pipe_stdout->fh, &rdfs))
			{
				switch (handle_stdout_stream(pipe_stdout)) {
					case -1:	reset_console();
							do_loop = 0;
							break;
					case 1:		do_loop = 0;
							break;
					case 2:		
					default:	break;
				}
			}
			if (FD_ISSET(pipe_stderr->fh, &rdfs))
			{
				switch (handle_stderr_stream(pipe_stderr)) {
					case -1:	reset_console();
							do_loop = 0;
							break;
					case 1:		do_loop = 0;
							break;
					case 2:		
					default:	break;
				}
			}
		}
		else
		{
			do_loop = 0;
		}
	}


	return 0;
}

int capture_start(void)
{
	if (prepare_buffer(app.pstdout)) {
		exit(EXIT_FAILURE);
	}
	if (prepare_buffer(app.pstderr)) {
		exit(EXIT_FAILURE);
	}

	if (CONFIG_USE_THREADS) {
		pthread_create(&app.pstderr->ct, NULL, capture_thread,
			       app.pstderr);
		pthread_create(&app.pstdout->ct, NULL, capture_thread,
			       app.pstdout);
	} else {
		return capture_loop(app.pstdout, app.pstderr);
	}

	return 0;
}

int capture_end(void)
{
	if (CONFIG_USE_THREADS) {
		/* wait for capture threads to exit */
		pthread_join (app.pstderr->ct, NULL);
		pthread_join (app.pstdout->ct, NULL);
	}

	return 0;
}
