/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  - Neither the name of Intel Corp. nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <errno.h>
#include "types.h"
#include "LMEConnection.h"
#include "LMS_if.h"

#ifdef _LINUX
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#define HECI_BUFF_SIZE 0x1000
#define HECI_IO_TIMEOUT 5000

const GUID LMEConnection::_guid = {0x3d98d9b7, 0x1ce8, 0x4252, 0xb3, 0x37, 0x2e, 0xff, 0x10, 0x6e, 0xf2, 0x9f};

LMEConnection::LMEConnection(bool verbose): _initState(INIT_STATE_DISCONNECTED), _reqID(0), _rxThread(NULL),
								_cb(NULL), _cbParam(NULL), _internalHECI(_guid, verbose), _heci(&_internalHECI)
{
}

LMEConnection::~LMEConnection()
{
}

bool LMEConnection::IsInitialized()
{
	bool res;

	_initLock.acquire();
	res = (_initState == INIT_STATE_CONNECTED) ? true : false;
	_initLock.release();

	return res;
}

bool LMEConnection::Init(HECICallback cb, void *param)
{
	_initLock.acquire();

	if (_initState == INIT_STATE_CONNECTED) {
		_heci->Deinit();
	} else if (_initState != INIT_STATE_DISCONNECTED) {
		_initLock.release();
		return false;
	}
	_initState = INIT_STATE_CONNECTING;

	_initLock.release();

	_cb = cb;
	_cbParam = param;
	if (!_heci->Init(LMS_PROCOL_VERSION)) {
		_initLock.acquire();
		_initState = INIT_STATE_DISCONNECTED;
		_initLock.release();
		return false;
	}

	_initLock.acquire();
	_initState = INIT_STATE_CONNECTED;
	_initLock.release();

	// Get LMS-LME Protocol Version
	unsigned char buff[HECI_BUFF_SIZE];
	int bytesRead, bytesWritten;

	LMS_PROTO_VERSION_MESSAGE msg;

	msg.MessageType = LMS_MESSAGE_TYPE_PROTO_VERSION;
	msg.ConnectionId = 0;
	msg.Protocol = 0;

	bytesWritten = _sendMessage((unsigned char*)&msg, sizeof(msg));
	if (bytesWritten != sizeof(msg)) {
		goto err;
	}

	bytesRead = _receiveMessage(buff, HECI_BUFF_SIZE);
	PRINT("Bytes read = %d\n", bytesRead);
	if (bytesRead != sizeof(LMS_PROTO_VERSION_REPLY_MESSAGE)) {
		goto err;
	}

	LMS_PROTO_VERSION_REPLY_MESSAGE *repMsg;
	repMsg = (LMS_PROTO_VERSION_REPLY_MESSAGE *)buff;
	if (repMsg->Status != LMS_PROTOCOL_STATUS_OK) {
		// TODO: handle new prototocl version proposal
	}

	// launch RX thread
	_rxThread = new Thread(_rxThreadFunc, this);
	_rxThread->start();

	return true;

err:
	Deinit();
	return false;
}

void LMEConnection::Deinit()
{
	_initLock.acquire();
	_heci->Deinit();
	_initState = INIT_STATE_DISCONNECTED;
	
	delete _rxThread;
	_rxThread = NULL;

	_initLock.release();
}

int LMEConnection::SendMessage(int connID, unsigned char *buffer, int len)
{
	unsigned char sendBuf[1024 + sizeof(LMS_SEND_DATA_MESSAGE)];
	LMS_SEND_DATA_MESSAGE *msg;

	if (len > 1024) {
		return -1;
	}

	msg = (LMS_SEND_DATA_MESSAGE *)sendBuf;

	msg->MessageType = LMS_MESSAGE_TYPE_SEND_DATA;
	msg->ConnectionId = connID;
	msg->DataLength = htons(len);
	memcpy(msg->Data, buffer, len);
	return _sendMessage(sendBuf, sizeof(LMS_SEND_DATA_MESSAGE) + len);
}

int LMEConnection::_receiveMessage(unsigned char *buffer, int len)
{
	int result;
	if (_initState != INIT_STATE_CONNECTED) {
		return -1;
	}

	result = _heci->ReceiveMessage(buffer, len, WAIT_INFINITE);

    if (result < 0 && errno == ENOENT)
    {
	    _initLock.acquire();
	    _initState = INIT_STATE_DISCONNECTED;
	    _initLock.release();
    }

    return result;
}

int LMEConnection::_sendMessage(unsigned char *buffer, int len)
{
	int result;

	if (_initState != INIT_STATE_CONNECTED) {
		return -1;
	}
	
	//_sendMessageLock.acquire();
	result = _heci->SendMessage(buffer, len, HECI_IO_TIMEOUT);
	//_sendMessageLock.release();

    if (result < 0 && errno == ENOENT)
    {
	    _initLock.acquire();
	    _initState = INIT_STATE_DISCONNECTED;
	    _initLock.release();
    }

	return result;
}


void LMEConnection::RequestIPFQDN()
{
	LMS_IP_FQDN_REQUEST_MESSAGE msg;

	msg.MessageType = LMS_MESSAGE_TYPE_IP_FQDN_REQUEST;
	msg.ConnectionId = 0;
	_sendMessage((unsigned char*)&msg, sizeof(msg));
}

#ifdef _REMOTE_SUPPORT
void LMEConnection::RequestEntDNSSuffixList()
{
	LMS_ENTERPRISE_DNS_SUFFIX_LIST_REQUEST_MESSAGE msg;

	msg.MessageType = LMS_MESSAGE_TYPE_ENTERPRISE_DNS_SUFFIX_LIST_REQUEST;
	msg.ConnectionId = 0;

	_sendMessage((UCHAR *)&msg, sizeof(msg));
}

void LMEConnection::SendEnterpriseAccessStatus(bool access, unsigned long localIp)
{
	LMS_ENTERPRISE_NETWORK_ACCESS_MESSAGE msg;
	msg.MessageType = LMS_MESSAGE_TYPE_ENTERPRISE_NETWORK_ACCESS;
	msg.ConnectionId = 0;
	msg.Flags = 0;
	memcpy(msg.HostIPAddress, &localIp, sizeof(unsigned long));
	msg.EnterpiseAccess = (access) ? 1 : 0;

	_sendMessage((UCHAR *)&msg, sizeof(msg));
}

#endif

bool LMEConnection::OpenConnection(struct sockaddr_in addr, unsigned short mePort, int *connID, bool remote)
{
	unsigned char currReqID = _reqID++;
	bool ret = false;

	_initLock.acquire();
	if (_initState != INIT_STATE_CONNECTED) {
		_initLock.release();
		return false;
	}
	_initLock.release();

	LMS_OPEN_CONNECTION_EX_MESSAGE openConnectionExMsg;
	LMS_OPEN_CONNECTION_MESSAGE openConnectionMsg;
	unsigned char *msg = NULL;
	int msgLen = 0;

	if (_heci->GetProtocolVersion() == LMS_PROCOL_VERSION) {

		openConnectionExMsg.MessageType = LMS_MESSAGE_TYPE_OPEN_CONNECTION_EX;
		openConnectionExMsg.ConnectionId = 0;
		openConnectionExMsg.Protocol = LMS_PROTOCOL_TYPE_TCP_IPV4;
		openConnectionExMsg.Flags = 0;
		openConnectionExMsg.Flags |= (remote)? REMOTE_BIT : 0;
		openConnectionExMsg.OpenRequestId = currReqID;
		memcpy(openConnectionExMsg.Host, &addr.sin_addr, sizeof(addr.sin_addr));
		openConnectionExMsg.HostPort = addr.sin_port;
		openConnectionExMsg.MEPort = htons(mePort);

		msg = (unsigned char*)&openConnectionExMsg;
		msgLen = sizeof(openConnectionExMsg);
	}
	else {

		openConnectionMsg.MessageType = LMS_MESSAGE_TYPE_OPEN_CONNECTION;
		openConnectionMsg.ConnectionId = 0;
		openConnectionMsg.Protocol = LMS_PROTOCOL_TYPE_TCP_IPV4;
		openConnectionMsg.OpenRequestId = currReqID;
		memcpy(openConnectionMsg.HostIPAddress, &addr.sin_addr, sizeof(addr.sin_addr));
		openConnectionMsg.HostPort = addr.sin_port;
		openConnectionMsg.MEPort = htons(mePort);

		msg = (unsigned char*)&openConnectionMsg;
		msgLen = sizeof(openConnectionMsg);
	}

	// save as pending request
	Connection conn;
	conn.event = new Event();
	conn.status = LMS_CONNECTION_STATUS_FAILED;

	_mapLock.acquire();
	_pendingConnections[currReqID] = conn;
	_mapLock.release();

	int bytesWritten;
	bytesWritten = _sendMessage(msg, msgLen);
	if (bytesWritten != msgLen) {
		goto out;
	}

	
	if (conn.event->wait(10000) == false){
		// no response from FW
		goto out;
	}

	ret = true;

out:
	_mapLock.acquire();
	if (_pendingConnections[currReqID].status != LMS_CONNECTION_STATUS_OK) {
		ret = false;
	} else {
		*connID = _pendingConnections[currReqID].connID;
	}
	_pendingConnections.erase(currReqID);
	_mapLock.release();

	delete conn.event;
	conn.event = NULL;

	return ret;
}

void LMEConnection::CloseConnection(int connID, int status)
{
	LMS_CLOSE_CONNECTION_MESSAGE msg;

	msg.MessageType = LMS_MESSAGE_TYPE_CLOSE_CONNECTION;
	msg.ConnectionId = connID;
	msg.ClosingReason = status;

	_sendMessage((unsigned char*)&msg, sizeof(msg));
}

void LMEConnection::_rxThreadFunc(void *param)
{
	LMEConnection *connection = (LMEConnection*)param;

	try {
		connection->_doRX();
	}
	
	catch (...) {
		PRINT("LMEConnection do RX exception\n");
	}
    pthread_exit(NULL);
}

void LMEConnection::_doRX()
{
	unsigned char buff[HECI_BUFF_SIZE];
	int bytesRead;

	while (true) {
		int status = 1;

		bytesRead = _receiveMessage(buff, HECI_BUFF_SIZE);
		if (bytesRead < 0) {
			// TODO: signal main thread to de-init and restart
			PRINT("Error receiving data from HECI\n");
			Deinit();
			break;
		}

		if (bytesRead == 0) {
			// ERROR
			continue;
		}

		if (bytesRead < 2) {
			// ERROR
			continue;
		}

		switch (buff[0]) {
			case LMS_MESSAGE_TYPE_OPEN_CONNECTION_REPLY:
				{
					LMS_OPEN_CONNECTION_REPLY_MESSAGE *repMsg;
					repMsg = (LMS_OPEN_CONNECTION_REPLY_MESSAGE *) buff;

					_mapLock.acquire();
					ConnMap::iterator itr;
					itr = _pendingConnections.find(repMsg->OpenRequestId);
					if (itr != _pendingConnections.end()) {
						(*itr).second.connID = repMsg->ConnectionId;
						(*itr).second.status = repMsg->Status;
						(*itr).second.event->set();
					}
					_mapLock.release();
				}
				break;

			case LMS_MESSAGE_TYPE_OPEN_CONNECTION_EX:
				{
					// report incoming connection request
					_cb(_cbParam, buff, bytesRead, &status);

					LMS_OPEN_CONNECTION_EX_MESSAGE *msg;
					msg = (LMS_OPEN_CONNECTION_EX_MESSAGE *)buff;

					if ((msg->Flags & HOSTNAME_BIT) != 0) {
						PRINT("Got client connection request %d for host %s, port %d\n",
							msg->ConnectionId, msg->Host, msg->HostPort);
					}
					else {

						PRINT("Got client connection request %d for IP %s, port %d\n",
						msg->ConnectionId, inet_ntoa(*((struct in_addr*)msg->Host)), msg->HostPort);
					}

					LMS_OPEN_CONNECTION_REPLY_MESSAGE repMsg;
					memset(&repMsg, 0, sizeof(repMsg));

					repMsg.MessageType = LMS_MESSAGE_TYPE_OPEN_CONNECTION_REPLY;
					repMsg.ConnectionId = msg->ConnectionId;
					if (status == 0) {
						repMsg.Status = LMS_CONNECTION_STATUS_OK;
					} else {
						repMsg.Status = LMS_CONNECTION_STATUS_FAILED;
					}

					DWORD bytesWritten;
					bytesWritten = _sendMessage((UCHAR *)&repMsg, sizeof(repMsg));
					if (bytesWritten != sizeof(repMsg)) {
						PRINT("Send failed: bytesWritten: %lu\n", bytesWritten);
					}
				}
				break;

			// for the following cases, we just pass the message upstream
			case LMS_MESSAGE_TYPE_CLOSE_CONNECTION:
			case LMS_MESSAGE_TYPE_SEND_DATA:
			case LMS_MESSAGE_TYPE_IP_FQDN:
#ifdef _REMOTE_SUPPORT
			case LMS_MESSAGE_TYPE_ENTERPRISE_DNS_SUFFIX_LIST:
			case LMS_MESSAGE_TYPE_REMOTE_ACCESS_STATUS:
#endif
				_cb(_cbParam, buff, bytesRead, &status);
				break;

			case LMS_MESSAGE_TYPE_PROTO_VERSION_REPLY:
				// TODO: handle LMS-LME version handshake
				break;

			default:
				// ERROR
				break;
		}
	}
}

HECI& LMEConnection::GetHECI()
{
	return *_heci;
}
