/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  - Neither the name of Intel Corp. nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdint.h>
#include <aio.h>
#include "HECILinux.h"
//#include "HECI_if.h"
#pragma pack(1)
typedef struct heci_ioctl_data{
	uint32_t size;
	char* data;
}heci_ioctl_data_t;
/* IOCTL commands */
#define HECI_IOCTL_LETTER 'H'
#define IOCTL_HECI_GET_VERSION \
    _IOWR(HECI_IOCTL_LETTER ,0x800,heci_ioctl_data_t)
#define IOCTL_HECI_CONNECT_CLIENT \
    _IOWR(HECI_IOCTL_LETTER ,0x801, heci_ioctl_data_t)
#define IOCTL_HECI_WD \
    _IOWR(HECI_IOCTL_LETTER ,0x802, heci_ioctl_data_t)
#define IAMT_HECI_GET_RECEIVED_MESSAGE_DATA \
	 _IOW(HECI_IOCTL_LETTER, 0x803, heci_ioctl_data_t) 
typedef struct heci_client
{
    uint32_t	max_message_length;
    uint8_t   	protocol_version;
} heci_client_t;
typedef struct heci_driver_version
{
    uint8_t     major;
    uint8_t     minor;
    uint8_t     hotfix;
    uint16_t    build;
} heci_driver_version_t;
#pragma pack(0)
/***************************** public functions *****************************/

HECILinux::HECILinux(const GUID guid, bool verbose): HECI(guid, verbose)
{
}

HECILinux::~HECILinux()
{
    if (_fd != -1) {
        close(_fd);		
	}
}

bool HECILinux::Init(UINT8 LMSVersion)
{
    int result;
	heci_driver_version_t *version;
    heci_client_t  *heci_client;
    bool return_result = true;
   	heci_ioctl_data_t version_response;
    heci_ioctl_data_t client_connect;
	if (_initialized) {
		Deinit();
	}

	_fd = open("/dev/heci", O_RDWR);

	if (_fd == -1 ) {
		if (_verbose) {			
			fprintf(stderr, "Error: Cannot establish a handle to the HECI driver\n");
		}
		return false;
	}
	_initialized = true;
    version_response.size = sizeof(heci_driver_version_t);
    version_response.data = (char*)malloc(version_response.size);
    if (!version_response.data) {
        printf("malloc failure.\n");
        return_result = false;
        Deinit();
       	goto heci_free;
    }
    	  
    result = ioctl(_fd, IOCTL_HECI_GET_VERSION, &version_response);
    if (result) {
        if (_verbose) {
			fprintf(stderr, "error in IOCTL_HECI_GET_VERSION recieve message. err=%d\n",result);
		}   
        return_result = false;
        Deinit();
        goto heci_free;
    }
    version =(heci_driver_version_t*)version_response.data;
    if (_verbose) {
		printf("Connected to HECI driver, version: %d.%d.%d.%d\n",
			version->major, version->minor, version->hotfix, version->build);
		printf("Size of guid = %d\n",sizeof(_guid));

	}
    client_connect.size = sizeof(_guid);
    client_connect.data = (char*)malloc(client_connect.size);
    if (!client_connect.data) {
        printf("malloc failure.\n");
        return_result = false;
        Deinit();
       	goto heci_free;
    }
    memcpy(client_connect.data,&_guid,sizeof(_guid)); 
	result = ioctl(_fd, IOCTL_HECI_CONNECT_CLIENT, &client_connect);
	if (result) {
         if (_verbose) {
			fprintf(stderr, "error in IOCTL_HECI_CONNECT_CLIENT recieve message. err=%d\n",result);
		}
        return_result = false;
        Deinit();
        goto heci_free;
	}
	heci_client = (heci_client_t*) client_connect.data;
	 if (_verbose) {
		printf("max_message_length %d \n",(heci_client->max_message_length));
	    printf("protocol_version %d \n",(heci_client->protocol_version));
	}
	if (heci_client->protocol_version > LMSVersion) {
		if (_verbose) {
			fprintf(stderr, "Error: AMT version not supported\n");
		}
		return_result = false;
        	Deinit();
        	goto heci_free;
	}

	_protocolVersion = heci_client->protocol_version;
	_bufSize =heci_client->max_message_length;
heci_free:
    if(NULL!=version_response.data)
        free(version_response.data);
    if(NULL!=client_connect.data)
        free(client_connect.data);
	return return_result;
}

void HECILinux::Deinit()
{
	if (_fd != -1) {
		close(_fd);
		_fd = -1;
	}

	_bufSize = 0;
	_initialized = false;
}

int HECILinux::ReceiveMessage(unsigned char *buffer, int len, unsigned long timeout)
{
   	int rv = 0;
   
    rv = read(_fd,(void*)buffer,len);
	if (rv < 0) {
		if (_verbose) {
			fprintf(stderr,"read failed with status %d\n", rv);
		}
		Deinit();
	}
	return rv;
}

int HECILinux::SendMessage(unsigned char *buffer, int len, unsigned long timeout)
{
   	int rv = 0;
	int return_length =0;
	fd_set set;
	struct timeval tv;
    tv.tv_sec =  timeout / 1000;
	tv.tv_usec =(timeout % 1000) * 1000000;
    if (_verbose)
	{
		printf("call write length = %d\n",len);
	}
	return_length = write( _fd,(void *)buffer,len);
	if ( return_length < 0)
	{
		if (_verbose) {
			 fprintf(stderr,"write failed with status %d\n", rv);
		}
		goto out;
	} 
    
    FD_ZERO(&set);
    FD_SET(_fd, &set);
    rv = select(_fd+1 ,&set,NULL, NULL, &tv);
    if (rv > 0 && FD_ISSET(_fd, &set)) {
       if (_verbose) {
			 fprintf(stderr,"write success\n");
		}
    }
    else if (rv == 0) {
		if (_verbose)
		{
			 fprintf(stderr,"write failed on timeout with status\n");
			 goto out;
		}
    }
    else { //rv<0
        if (_verbose)
		{
			 fprintf(stderr,"write failed on select with status %d\n", rv);
			 goto out;
		}
    }
	rv = return_length; 
out:
	if (rv < 0) {
		printf("Send failed!\n");
		Deinit();
	}

	return rv;
}

