/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/board.h"

static GtkTargetEntry target_table[] = {
  { (gchar *) "application/lwizard", GTK_TARGET_SAME_APP, 0 }
};

static guint n_targets = sizeof(target_table) / sizeof(target_table[0]);

static void clear_mark (LWPiece **marked_piece)
{
	*marked_piece = NULL;
}

static void zoom_inout (LWBoard *board)
{
	if (board->getPieceSize() < 40)
		board->setPieceSize(40);
	else
		board->setPieceSize(20);
}

static void shrink_w (LWBoard *board)
{
	guint h = board->getHeight();
	g_return_if_fail (h > 0);
	
	guint w = board->getRowNth(0)->getWidth();
	
	if (w > 3)
		board->setSize (w-1, h);
}

static void expand_w (LWBoard *board)
{
	guint h = board->getHeight();
	g_return_if_fail (h > 0);
	
	guint w = board->getRowNth(0)->getWidth();
	
	if (w < 30)
		board->setSize (w+1, h);
}

static void shrink_h (LWBoard *board)
{
	guint h = board->getHeight();
	g_return_if_fail (h > 0);
	
	guint w = board->getRowNth(0)->getWidth();
	
	if (h > 3)
		board->setSize (w, h-1);
}

static void expand_h (LWBoard *board)
{
	guint h = board->getHeight();
	g_return_if_fail (h > 0);
	
	guint w = board->getRowNth(0)->getWidth();
	
	if (h < 30)
		board->setSize (w, h+1);
}

/*! \brief Obsługa zdarzenia otrzymania danych za pomocą mechanizmu DND
*/
static void  
target_drag_data_received  (GtkWidget          *widget,
							GdkDragContext     *context,
							gint                x,
							gint                y,
							GtkSelectionData   *data,
							guint               info,
							guint               time,
							gpointer           userdata)

{
	/* Lista zmiennych nieużywanych */
	(void) widget;
	(void) x;
	(void) y;
	(void) info;
	
	if ((data->length == 0) || (data->format != 8)) {
    	gtk_drag_finish (context, FALSE, FALSE, time);
 
    	return;
    }
	
	LWPiece *sourcepiece = * ((LWPiece **) data->data);
	
	LWType type = sourcepiece->getRow()->getBoard()->getType();
	
	if (type == LW_TYPE_PROGRAM) {
		if (TRUE == sourcepiece->getRow()->isPieceDummy(sourcepiece)) {
	
			/* Usuwanie pustych wierszy */
			if (NULL == sourcepiece->getRow()->getPieceNth(0)) 
				if (FALSE == sourcepiece->getRow()->getBoard()->isRowDummy (sourcepiece->row)) {
					sourcepiece->getRow()->getBoard()->removeRow (sourcepiece->row);
					gtk_drag_finish (context, TRUE, FALSE, time);
					return;
				} 
			
			/* Nie ruszać dodatkowych klocków na końcach wierszy,
			 * w trybie programowania */
			gtk_drag_finish (context, TRUE, FALSE, time);
			return;
		}
		
		sourcepiece->getRow()->removePiece (sourcepiece);
	}
	
	if (type == LW_TYPE_WORLD)
			sourcepiece->clear();
	
	gtk_drag_finish (context, TRUE, FALSE, time);
}

void LWBoard::buildResizeBar ()
{
	GtkWidget *bbox1 = gtk_vbutton_box_new ();
	gtk_button_box_set_child_size (GTK_BUTTON_BOX(bbox1), 20, 20);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox1), GTK_BUTTONBOX_START);
	gtk_table_attach (GTK_TABLE (widget), bbox1, 1, 2, 0, 1, 
		(GtkAttachOptions) 0, (GtkAttachOptions) GTK_FILL, 0, 0);
	gtk_widget_show (bbox1);
	
	GtkWidget *button1 = gtk_button_new ();
	zoominout = GTK_IMAGE (gtk_image_new());
	gtk_widget_show (GTK_WIDGET (zoominout));
	gtk_container_add (GTK_CONTAINER (button1), GTK_WIDGET(zoominout));
	gtk_widget_show (button1);
	gtk_container_add (GTK_CONTAINER(bbox1), button1);	

	GtkWidget *arrow1 = gtk_arrow_new (GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_widget_show (arrow1);
	GtkWidget *button2 = gtk_button_new ();	
	gtk_container_add (	GTK_CONTAINER (button2), arrow1);
	gtk_widget_show (button2);
	gtk_container_add (GTK_CONTAINER (bbox1), button2);
	
	GtkWidget *arrow2 = gtk_arrow_new (GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_widget_show (arrow2);
	GtkWidget *button3 = gtk_button_new ();	
	gtk_container_add (	GTK_CONTAINER (button3), arrow2);
	gtk_widget_show (button3);
	gtk_container_add (	GTK_CONTAINER (bbox1), button3);

	GtkWidget *bbox2 = gtk_vbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox2), GTK_BUTTONBOX_END);
	gtk_button_box_set_child_size (GTK_BUTTON_BOX(bbox2), 20, 20);
	gtk_table_attach (GTK_TABLE (widget), bbox2, 1, 2, 1, 2, 
		(GtkAttachOptions) 0, (GtkAttachOptions) GTK_FILL, 0, 0);

	gtk_widget_show (bbox2);
	
	GtkWidget *arrow3 = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
	gtk_widget_show (arrow3);
	GtkWidget *button4 = gtk_button_new ();	
	gtk_container_add (	GTK_CONTAINER (button4), arrow3);
	gtk_widget_show (button4);
	gtk_container_add (	GTK_CONTAINER (bbox2), button4);
	
	GtkWidget *arrow4 = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_widget_show (arrow4);
	GtkWidget *button5 = gtk_button_new ();	
	gtk_container_add (	GTK_CONTAINER (button5), arrow4);
	gtk_widget_show (button5);
	gtk_container_add (	GTK_CONTAINER (bbox2), button5);
	
	g_signal_connect_swapped (G_OBJECT (button1), "clicked", 
			G_CALLBACK(zoom_inout), (gpointer) this);
	
	g_signal_connect_swapped (G_OBJECT (button2), "clicked", 
			G_CALLBACK(expand_w), (gpointer) this);
			
	g_signal_connect_swapped (G_OBJECT (button3), "clicked", 
			G_CALLBACK(shrink_w), (gpointer) this);

	g_signal_connect_swapped (G_OBJECT (button4), "clicked", 
			G_CALLBACK(shrink_h), (gpointer) this);

	g_signal_connect_swapped (G_OBJECT (button5), "clicked", 
			G_CALLBACK(expand_h), (gpointer) this);
}

/*!	\brief Inicjalizacja planszy

	Inicjalizuje obiekt, metoda prywatna używana wewnętrznie przez konstruktor

	\param type Typ planszy
*/
void LWBoard::init (LWType a_type)
{
	gint dest_actions = 0;
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox);
	
	type = a_type;

	switch (type) {
		case LW_TYPE_WORLD:
			widget = gtk_table_new (2, 2, FALSE);
			gtk_table_attach (GTK_TABLE (widget), vbox, 0, 1, 0, 2,
				(GtkAttachOptions) 0, (GtkAttachOptions) 0, 0, 0);
			buildResizeBar ();
			gtk_widget_show (widget);
			break;			
		
		case LW_TYPE_ICONS:
		case LW_TYPE_COMMANDS:
			widget = gtk_event_box_new ();
			gtk_container_add (GTK_CONTAINER(widget), vbox);
			gtk_widget_show (widget);

			dest_actions = GDK_ACTION_COPY | GDK_ACTION_MOVE;
		
 			gtk_drag_dest_set (widget,
			    GTK_DEST_DEFAULT_ALL,
			   	target_table, n_targets,
		   		(GdkDragAction) dest_actions); 		
		
	    	g_signal_connect ( G_OBJECT(widget), "drag_data_received",
	 			G_CALLBACK(target_drag_data_received), NULL);

			break;
		
		case LW_TYPE_PROGRAM:
			dummy_row = new LWRow(this);
			gtk_box_pack_end (GTK_BOX (vbox), dummy_row->getWidget(), TRUE, TRUE, 0);
		
		default:
			widget = vbox;
	}
}

/*! 
 	\brief Konstruktor klasy
	
	Konstruktor klasy, na podstawie typu. W zależności od zastosowania
	planszy będą używane różne typy.
	
	\param type Typ tworzonej planszy
*/
LWBoard::LWBoard (LWType a_type):
list_row(NULL), dummy_row(NULL), marked_piece(NULL), 
enable_grid(FALSE), boardset(NULL)
				
{
	init (a_type);
	setPieceSize(40);
}

/*! 
	\brief Konstruktor klasy
 
	Konstruktor klasy, na podstawie zestawu plansz do którego będzie należała
	plasza. Typ tworzonej planszy będzie ustalony na taki sam jakiego używa
	zestaw. 
*/
LWBoard::LWBoard (LWBoardSet *a_boardset): 
list_row(NULL), dummy_row(NULL), marked_piece(NULL),
enable_grid(FALSE)
{
	init (a_boardset->getType());
	setPieceSize(40);
	
	boardset = a_boardset;
}

/*! \brief Destruktor klasy

	Destruktor klasy, podczas destrukcji planszy usunięte również wszystkie 
	należące do niej wiersze.
*/
LWBoard::~LWBoard ()
{
	clear();

	if (dummy_row != NULL)
		delete dummy_row;
	
	gtk_widget_destroy (widget);
}

/*!	\brief Konstruktor kopiujący
 
	Konstruje planszę oraz kopiuje wszystkie należące do niej 
	wiersze. Nowa plansza przyjmuje typ według argumentu typ

	\param board Kopiowana plansza
	\param type Typ tworzonej planszy 
*/
LWBoard::LWBoard (const LWBoard *board, LWType a_type):
list_row(NULL), dummy_row(NULL), marked_piece(NULL),
enable_grid(FALSE), boardset(NULL)
{
	init (a_type);
	setPieceSize(board->piecesize);
	
	for (GSList *l=board->list_row; l != NULL; l = l->next) {
		LWRow *row = (LWRow *) l->data;
		LWRow *newrow = new LWRow (row, this);
		addRow (newrow);
	}
}

/*!	\brief Rekonstrukcja elementów planszy na podstawie opisu XML

	Rekonstrukcja planszy na podstawie opisu XML zawartego w węźle 
	drzewa XML. Poprzednia zawartość planszy zostaje usunięta.

	\param node Węzeł drzewa XML na postawie którego dokonana zostanie
				rekonstrukcja
*/
void LWBoard::restoreFromXML (xmlNode *node)
{
	g_return_if_fail (!xmlStrcasecmp (node->name, (xmlChar *) "Board"));

	gchar *title = (gchar *) xmlGetProp (node, (xmlChar *) "title");	
	gchar *piecesizestr = (gchar *) xmlGetProp (node, (xmlChar *) "piecesize");
	
	clear();

	if (piecesizestr != NULL) {
		gint piecesize = atoi (piecesizestr);
		g_return_if_fail (piecesize > 0);
		setPieceSize (piecesize);
	}
	
	if (boardset != NULL && title != NULL)
		boardset->changeBoardTitle (this, title);
	
	for (xmlNode *n=node->children; n != NULL; n=n->next) {
		LWRow *row = new LWRow (this);
		row->restoreFromXML (n);
		addRow (row);
	}
}

/*! \brief Zapis elementów planszy w pastaci opisu XML
 
	Zapis planszy do postaci węzła drewa XML, utworzony węzeł zostanie dodany
	jako struktury rodzica.

	Nie jest dokonywane sprawdzenie nazwy węzła rodzica, najczęściej
	będzie to korzeń.

	\param node Węzeł drzewa XML na postawie którego dokonanie zostanie
				rekonstrukcja
*/
void LWBoard::storeToXML (xmlNode *parent_node)
{
	xmlNode *node = xmlNewChild(parent_node, NULL, (xmlChar *) "Board", NULL);

	gchar *s = g_strdup_printf ("%u", piecesize);
	xmlNewProp (node, (xmlChar *) "piecesize", (xmlChar *) s);
	g_free (s);
	
	for (GSList *l = list_row; l != NULL; l = l->next) {
		LWRow *row = (LWRow *) l->data;
		row->storeToXML (node);
	}
	
}

/*! \brief Dodanie wiersza do planszy
 
	Dodanie wiersza do planszy, dadany wiersz będzie znajdował się na dole 
	planszy. 

	\param row Dodawany wiersz
*/
void LWBoard::addRow (LWRow *row)
{
	g_return_if_fail (row != NULL);
	g_return_if_fail (row->getBoard() == this);
	
	gtk_box_pack_start (GTK_BOX (vbox), row->getWidget(), FALSE, FALSE, 0);
	list_row = g_slist_append (list_row, (gpointer) row);
}

/*! \brief Wstawienie wiersza do planszy
 
	Wstawia wiersz do planszy, dadany wiersz będzie znajdował przed wierszem
	sibiling.

	\param row Wstawiany wiersz
	\param sibiling Wiersz, przed którym ma być wstawiony wiersz
*/
void LWBoard::insertRowBefore (LWRow *row, LWRow *sibiling)
{
	g_return_if_fail (row != NULL);
	g_return_if_fail (row->getBoard() == this);
	g_return_if_fail (sibiling != NULL);
	g_return_if_fail (sibiling->getBoard() == this);
	gint a=0;
	
	list_row = g_slist_insert_before (list_row,
			g_slist_find (list_row, sibiling),	(gpointer) row);

	gtk_box_pack_start (GTK_BOX (vbox), row->getWidget(), FALSE, FALSE, 0);

	/* 
	Funckja gtk_box_reorder_child wstawia kontrolke na danej pozycji,
	jednak pozycja na ktorej jest wstawiana kontrolka zalezy
	od pozycji w wewętrznej liście już wstawionych kontrolek,
	lista jest posortowana w kolejnosci wywoływania przez gtk_box_pack_start,
	i gtk_box_pack_end. Co oznacza, że zawrtość może sie różnić od faktycznej
	kolejności kontrolek w przypadku mieszania tych funkcji. Aby obejść
	to ograniczenie sprawdzana jest obecność dodatkowego wiersza, który
	był dodany przez gtk_box_pack_end. Ponieważ jest to kontrolka, która
	zawsze jest dodawana jako perwsza więc w liście widnieje też jako pierwsza,
	choć fizycznie na planszy znajduje się jako ostatnia, z tego powodu
	w funkcji gtk_box_reorder_child pozycja jest zwiększana o 1 w
	przypadku istnienia dodatkowego klocka.	
	*/
	
	if (dummy_row != NULL) a = 1;	

	gtk_box_reorder_child (GTK_BOX (vbox), row->getWidget(), 
		g_slist_index (list_row, (gpointer) row)+a);
	
}

/*! \brief Usuwanie wiersza z planszy
 
	Usuwanie wiersza z planszy, usuwany wiersz musi się zawierać w danej instancji
	klasy, w przeciwnym wypadku nie zostanie usunięty.

	\param row Usuwany wiersz
*/
void LWBoard::removeRow (LWRow *row)
{
	g_return_if_fail (row != NULL);
	g_return_if_fail (row->getBoard() == this);
	
	list_row = g_slist_remove (list_row, (gpointer) row);
	delete row;
}

/*! \brief Pozycja wiersza w planszy

	Zwraca pozycję weiersza w strukturze planszy.

	\param row Pozycja wiersza na planszy liczona od 0. Jeśli wiersz nie należy
	do planszy zwrócone zostanie -1.
*/
gint LWBoard::getRowIndex (LWRow *row)
{
	g_return_val_if_fail (row != NULL, -1);
	g_return_val_if_fail (row->getBoard() == this, -1);
	
	return g_slist_index (list_row, (gpointer) row);
}

/*! \brief Wiersz na danej pozycji

	Zwraca wiersz na planszy który znajduje się na pozycji przekazanej w 
	argumencie. Pozycja na planszy liczona jest od 0.

	\param n Pozycja wiersza, na której się znajduje poszukiwany wiersz.
	\return Zwrócony wiersz, lub NULL w przypadku, gdy nie ma wiersza
	na poszukiwanej pozycji.
*/
LWRow *LWBoard::getRowNth (guint n)
{
	return (LWRow *) g_slist_nth_data (list_row, n);
}

/*! \brief Lista klocków na planszy

	Zwraca listę wszystkich klocków na planszy. 

	\return Zwrócona kopia listy klocków. Listę należy zwolnić
	przez g_slist_free.
*/
GSList *LWBoard::getPieceList ()
{
	GSList *list=NULL;
	
	for (GSList *l = list_row; l != NULL; l = l->next)
		list = g_slist_concat (list, ((LWRow *) l->data)->getPieceList());
	
	return list;
}

/*! \brief Określenie rozmiaru planszy

	Określa ilość klocków w pionie i poziomie
	\param width Szerokość planszy
	\param height Wysokość planszy
*/
void LWBoard::setSize (guint width, guint height)
{
	GSList *l;

	/* Zmniejsz liczbe wierszy jeśli potrzeba */
	while (NULL != (l = g_slist_nth (list_row, height))) {
		LWRow *row = (LWRow *) l->data;
		g_slist_delete_link (list_row, l);
		delete row;
	}
	
	/* Zwiększ liczbę wierszy jeśli potrzeba */
	for (guint i = g_slist_length (list_row); i < height; i++)
		addRow (new LWRow (this));
	
	/* Ustaw poprawną ilość elementów w wierszach */
	for (l = list_row; l != NULL; l = l->next) {
		LWRow *row = (LWRow *) l->data;
		row->setWidth (width);
	}
}

/*! \brief Pobranie wysokości planszy

	Pobiera ilość wierszy na planszy.
	\return Obliczona wielkość planszy
*/
guint LWBoard::getHeight ()
{
	return g_slist_length (list_row);
}

/*! \brief Ustawia wielkość klocków na planszy
	
	Ustawia wielkość ikony na klocku.
	\param size Wielkość w pikselach
*/
void LWBoard::setPieceSize (guint size)
{
	g_return_if_fail (size != 0);
	
	piecesize = size;
	
	for (GSList *l = list_row; l != NULL; l = l->next) {
		LWRow *row = (LWRow *) l->data;
		row->updatePieceSize();
	}
	
	if (type == LW_TYPE_WORLD)
		gtk_image_set_from_stock (zoominout, 
			(piecesize < 40) ? GTK_STOCK_ZOOM_IN : GTK_STOCK_ZOOM_OUT,
				GTK_ICON_SIZE_SMALL_TOOLBAR);
	
	if (type == LW_TYPE_PROGRAM)
		gtk_widget_set_size_request (dummy_row->getWidget(), piecesize, piecesize); /* Minimum size */
	
}

/*! \brief Pobranie wielkości klocków na planszy
	
	Ustawia wielkość ikon na klockach, znajdujących się na planszy.
	\return Wielkość ikon w pikselach
*/
guint LWBoard::getPieceSize ()
{
	return piecesize;
}

/*! \brief Włączanie siatki

	Włącza siatkę na planszy
	\param enable Jeśli wartość TRUE siatka zostanie włączona,
	w przeciwnym razie wyłączona
*/
void LWBoard::enableGrid (gboolean enable)
{
	if (boardset != NULL)
		g_return_if_fail (boardset->isGridEnabled() == enable);
		
	enable_grid = enable;
	
	for (GSList *l = list_row; l != NULL; l = l->next) {
		LWRow *row = (LWRow *) l->data;
		row->updateGrid();
	}
	
	gtk_box_set_spacing (GTK_BOX (vbox), (enable == TRUE) ? 2 : 0);
}

/*! \brief Sprawdzanie czy siatka jest włączona

	Sprawdza czy siatka jest włączona.
	\return Zwracana jest prawda jeśli siatka jest włączona, w przeciwnym
	razie fałsz.
*/
gboolean LWBoard::isGridEnabled ()
{
	return boardset != NULL ? boardset->isGridEnabled() : enable_grid;
}

/*! \brief Czyszczenie zawarości planszy

	Usuwa wszystkie wiersze na planszy.
	W przypadku gdy typem planszy jest LW_TYPE_PROGRAM
	pozostaje tylko dodatkowy wiersz u dołu planszy przeznaczony
	do upuszczania nowych klocków przy pomocy przeciągnij i upuść.
	Metoda nie przyjmuje argumentów i nie zwraca wyniku
*/
void LWBoard::clear ()
{
	for (GSList *l=list_row; l != NULL; l = l->next) {
		LWRow *row = (LWRow *) l->data;
		delete row;		
	}

	g_slist_free (list_row);
	list_row = NULL;
}

/*! \brief Wyróżnianie klocka

	Wyróżnia klocek, który leży w wierszu, który leży na planszy.
	Wyróżninie klocków jest użyteczne przy syganalizowaniu komunikatów,
	komunikat wyróznia klocek, którego komunikat dotyczy.

	\param piece Klocek do wyróżnienia
*/
void LWBoard::markPiece (LWPiece *piece)
{
const GdkColor markcolor = {0, 0xe000, 0x4000, 0x0000};
	
	g_return_if_fail (piece != NULL);
	g_return_if_fail (piece->getRow() != NULL);
	g_return_if_fail (piece->getRow()->getBoard() == this);
	
	if (marked_piece != NULL)
		unmarkPiece();
		
	gtk_widget_modify_bg (piece->getWidget(), GTK_STATE_NORMAL, &markcolor);
	
	g_object_set_data_full ( G_OBJECT (piece->getWidget()), "mark", 
								(gpointer) &marked_piece, (GDestroyNotify) clear_mark);
	
	marked_piece = piece;
}

/*! \brief Usuwanie wyróżnienia klocka

	Usuwa wyróznionienie kloceka, wyróznionego przez metodę markPiece.
	Jeśli nie było zaznaczonego klocka, metoda nic nie robi.
	Metoda nie przyjmuje parametrów.
*/
void LWBoard::unmarkPiece ()
{
	if (marked_piece == NULL) return;
		
	gtk_widget_modify_bg (marked_piece->getWidget(), GTK_STATE_NORMAL, NULL);
	marked_piece = NULL;
}

/*! \brief Pobieranie kontrolki
 
	Pobiera kontrolkę (ang. widget) zawierająca planszę

	\return Zwrócona kontrolka
*/
GtkWidget *LWBoard::getWidget ()
{
	return widget;
}

/*! \brief Pobieranie typu planszy
 
	Pobiera typ planszy

	\return Zwracany typ
*/
LWType LWBoard::getType ()
{
	return type;
}

/*! \brief Sprawdzanie czy element jest dodatkowy

	Sprawdza czy element jest dodatkowym elemntem, 
	Metoda przeznaczona wyłącznie do obsługi mechanizmu 
	Przeciągnij i Upuść.

	\param row Sprawdzany element	
	\return Rezultat testu. TRUE jeśli wiersz jest dodatkowy,
	w przeciwnym wypdaku FALSE
*/
gboolean LWBoard::isRowDummy (LWRow *row)
{
	return (row == dummy_row) ? TRUE : FALSE;
}
