/* $Id: util.c,v 1.15 2005/03/31 15:54:51 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../li18n.h"
#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "gtkconfig.h"
#include "util.h"

#define INFO_STRING "<span weight=\"bold\" size=\"larger\">%s</span>\n\n" \
                     "%s"

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

void
util_open_link (const gchar *link)
{
     gchar *cmd;
     GError *err = NULL;
     if (settings->prefs->browser)
     {
          cmd = g_strdup_printf ("%s %s", settings->prefs->browser, link);
          debug ("Running \"%s\"\n", cmd);
          
          if (!g_spawn_command_line_async (cmd, &err))
          {
               error_warning (_("Link could not be opened!"),
                              err->message, ERR_NOT_AVAILABLE);
               g_error_free (err);
          }
          g_free (cmd);
     }
     return;
}

void
util_play_sound (const gchar *file)
{
     gchar *cmd;
     GError *err = NULL;
     if (settings->prefs->player)
     {
          cmd = g_strdup_printf ("%s %s", settings->prefs->player, file);
          debug ("Running \"%s\"\n", cmd);
          
          if (!g_spawn_command_line_async (cmd, &err))
          {
               error_warning (_("Soundfile could not be played!"),
                              err->message, ERR_NOT_AVAILABLE);
               g_error_free (err);
          }
          g_free (cmd);
     }
     return;
}

void
util_new_status (GtkStatusbar *status, const gchar *msg)
{
     gtk_statusbar_pop (status, 1);
     gtk_statusbar_push (status, 1, msg);
     return;
}

void
util_get_folderselection (GtkWidget *widget)
{
     gchar *text =
          util_run_fileselection (_("Select Folder"),
                                  GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

     if (text)
     {
          if (GTK_IS_ENTRY (widget))
               gtk_entry_set_text (GTK_ENTRY (widget), text);
          else if (GTK_IS_LABEL (widget))
               gtk_label_set_text (GTK_LABEL (widget), text);
          g_free (text);
     }
     return;
}

void
util_get_fileselection (GtkWidget *widget)
{
     gchar *text = util_run_fileselection (_("Open File"),
                                           GTK_FILE_CHOOSER_ACTION_OPEN);

     if (text)
     {
          if (GTK_IS_ENTRY (widget))
               gtk_entry_set_text (GTK_ENTRY (widget), text);
          else if (GTK_IS_LABEL (widget))
               gtk_label_set_text (GTK_LABEL (widget), text);
          g_free (text);
     }
     return;
}

gchar*
util_run_fileselection (const gchar *title, GtkFileChooserAction action)
{
     GtkWidget *dialog;
     gchar *text = NULL;

     dialog = gtk_file_chooser_dialog_new (title,
                                           GTK_WINDOW (main_app->window),
                                           action,
                                           GTK_STOCK_CANCEL,
                                           GTK_RESPONSE_CANCEL,
                                           GTK_STOCK_OPEN,
                                           GTK_RESPONSE_ACCEPT,
                                           NULL);

     if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
          text = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

     gtk_widget_destroy (dialog);
     return text;
}

GtkWidget*
util_create_button_with_image (const gchar *label, const gchar *icon,
                               gboolean right)
{
     GtkWidget *button;
     GtkWidget *buttonlabel;
     GtkWidget *align;
     GtkWidget *box;
     GtkWidget *image;

     button = gtk_button_new ();

     align = gtk_alignment_new (0.5, 0.5, 0, 0);
     gtk_container_add (GTK_CONTAINER (button), align);

     box = gtk_hbox_new (FALSE, 2);
     gtk_container_add (GTK_CONTAINER (align), box);

     image = gtk_image_new_from_stock (icon, GTK_ICON_SIZE_BUTTON);
     if (right)
          gtk_box_pack_start (GTK_BOX (box), image, FALSE, FALSE, 0);
     else
          gtk_box_pack_end (GTK_BOX (box), image, FALSE, FALSE, 0);
     
     buttonlabel = gtk_label_new_with_mnemonic (label);
     gtk_box_pack_start (GTK_BOX (box), buttonlabel, FALSE, FALSE, 0);
     
     return button;
}

void
util_copy_int (GtkWidget *widget, int *data)
{
     if  (GTK_IS_SPIN_BUTTON (widget))
          *data = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
     else if (GTK_IS_COMBO_BOX (widget))
          *data = gtk_combo_box_get_active (GTK_COMBO_BOX (widget));
     return;
}

void 
util_copy_string (GtkWidget *widget, gchar **data)
{
     if (GTK_IS_ENTRY (widget))
          *data = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
     return;
}

void
util_copy_boolean (GtkWidget *widget, gboolean *data)
{
     if (GTK_IS_TOGGLE_BUTTON (widget))
          *data = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
     return;
}


gboolean
util_ask_for_confirmation (const gchar *title, const gchar *text)
{
     GtkWidget *dlg_ask;
     GtkWidget *box_main;
     GtkWidget *lbl_text;
     GtkWidget *image;
     gchar *info;
     gint result = 0;

     dlg_ask = gtk_dialog_new_with_buttons (title,
                                            /*GTK_WINDOW (main_app->window)*/
                                            NULL,
                                            GTK_DIALOG_MODAL |
                                            GTK_DIALOG_DESTROY_WITH_PARENT,
                                            GTK_STOCK_CANCEL,
                                            GTK_RESPONSE_CANCEL,
                                            GTK_STOCK_OK,
                                            GTK_RESPONSE_ACCEPT,
                                            NULL);
     gtk_dialog_set_has_separator (GTK_DIALOG (dlg_ask), FALSE);
     gtk_container_set_border_width (GTK_CONTAINER (dlg_ask), WIDGET_BORDER);
     gtk_window_set_resizable (GTK_WINDOW (dlg_ask), FALSE);

     box_main = hig_hbox_new ();
     gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg_ask)->vbox), box_main,
                         TRUE, TRUE, 0);

     image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION,
                                       GTK_ICON_SIZE_DIALOG);
     gtk_box_pack_start (GTK_BOX (box_main), image, FALSE, FALSE, 0);
                 
     info = g_strdup_printf (INFO_STRING, title, text);
     lbl_text = gtk_label_new (NULL);
     gtk_label_set_line_wrap (GTK_LABEL (lbl_text), TRUE);
     gtk_label_set_markup (GTK_LABEL (lbl_text), info);
     g_free (info);
     gtk_box_pack_start (GTK_BOX (box_main), lbl_text, TRUE, TRUE, 0);

     gtk_widget_show_all (box_main);
     
     result = gtk_dialog_run (GTK_DIALOG (dlg_ask));
     switch (result)
     {
     case GTK_RESPONSE_ACCEPT:
          gtk_widget_destroy (dlg_ask);
          return TRUE;
          break;
     case GTK_RESPONSE_CANCEL:
          gtk_widget_destroy (dlg_ask);
          return FALSE;
          break;
     default:
          break;
     }
     return FALSE;
}

void 
util_info (const gchar *text)
{
     GtkWidget *dialog;

     dialog = gtk_message_dialog_new (GTK_WINDOW (main_app->window),
                                      GTK_DIALOG_DESTROY_WITH_PARENT,
                                      GTK_MESSAGE_INFO,
                                      GTK_BUTTONS_CLOSE,
                                      text);
     gtk_container_set_border_width (GTK_CONTAINER (dialog), WIDGET_BORDER);
     gtk_window_set_position (GTK_WINDOW (dialog),
                              GTK_WIN_POS_CENTER_ON_PARENT);
     gtk_dialog_run (GTK_DIALOG (dialog));
     gtk_widget_destroy (dialog);
}

void
util_event_unregister (void *object)
{
     main_app->objects = event_remove_listener_by_object (main_app->objects,
                                                          object);
     return;
}

/*
 * the code idea behind that was taken from the XFCE project, package
 * libxfce4util, i18n.c
 * Copyright (c) 2003-2004 Benedikt Meurer <benny@xfce.org>
 * Copyright (c) 2004 Jasper Huijsman <jasper@xfce.org>
 *
 * The code looks for a localized _path_, e.g.
 *
 * /usr/local/share/doc/C/WELCOME  - C locale
 * /usr/local/share/doc/en_US/WELCOME  - en_US locale
 *
 * the caller has to free the result using g_free ()
 */
#define BASEPATH "%s/%s"

gchar*
util_get_file_with_locale (gchar *path, gchar *file)
{
     gchar *name = NULL;
     gchar *filename = NULL;
     char **sub = NULL;
     guint i = 0;
     char *seps[] = { ".", "@", "_" };
     char *lang = getenv ("LANG");
     
     if (lang && !g_strrstr (lang, "/"))
     {
          /* check full locale first */
          name = g_strdup_printf (BASEPATH, lang, file);
          filename = g_build_filename (path, name, NULL);
          g_free (name);

          debug ("looking in: %s\n", filename);
          if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
               return filename;
          g_free (filename);

          for (i = 0; i < G_N_ELEMENTS (seps); i++)
          {
               sub = g_strsplit (lang, seps[i], 2); /* max. two tokens */
               if (sub[0])
               {
                    filename = g_build_filename (path, sub[0], file, NULL);

                    debug ("looking in: %s\n", filename);

                    if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
                    {
                         g_strfreev (sub);
                         return filename;
                    }
               }
               g_strfreev (sub);
               g_free (filename);
          }
     }

     /* none found, so use the fallback C locale */
     debug ("no locale dir found, falling back to C\n");
     filename = g_build_filename (path, "C", file, NULL);
     return filename;
}
