/* $Id: gtkimport.c,v 1.7 2005/03/31 15:54:51 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../li18n.h"
#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "util.h"
#include "gtkedit.h"
#include "gtkconfig.h"
#include "gtkimport.h"

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

enum {
     IMP_NAME,
     IMP_VERSION,
     IMP_OPT_CALLBACK, /* hidden */
     IMP_CALLBACK,     /* hidden */
     IMP_COLUMNS
};

typedef struct
{
     const gchar     *name;
     const gchar     *version;
     OptionsFunction options;
     ImportFunction  callback;
} ImportFilter;

GSList *list_import = NULL;

static GtkListStore* create_filter_list (void);
static void import_filter_changed (GtkWidget *expand, GParamSpec *param_spec,
                                   GtkTreeView *view);
static void unexpand_options (GtkWidget *expand);
static void import_file (const gchar *file, ImportFunction func);

static GtkListStore*
create_filter_list (void)
{
     GtkListStore *list;
     GtkTreeIter iter;
     GSList *item = list_import;
     ImportFilter *tmp;

     list = gtk_list_store_new (IMP_COLUMNS, G_TYPE_STRING, G_TYPE_STRING,
                                G_TYPE_POINTER, G_TYPE_POINTER);
     while (item)
     {
          tmp = (ImportFilter *) item->data;

          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              IMP_NAME, tmp->name,
                              IMP_VERSION, tmp->version,
                              IMP_OPT_CALLBACK, tmp->options,
                              IMP_CALLBACK, tmp->callback,
                              -1);

          item = item->next;
     }
     return list;
}

static void
import_filter_changed (GtkWidget *expand, GParamSpec *param_spec,
                       GtkTreeView *view)
{
     GtkTreeSelection *select;
     GtkTreeIter iter;
     GtkTreeModel *model;
     GtkWidget *options;
     OptionsFunction func;

     if (gtk_expander_get_expanded (GTK_EXPANDER (expand)))
     {
          select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
          if (gtk_tree_selection_get_selected (select, &model, &iter))
          {
               gtk_tree_model_get (model, &iter, IMP_OPT_CALLBACK, &func, -1);
               options = func ();
               gtk_container_add (GTK_CONTAINER (expand), options);
               gtk_widget_show_all (options);
          }
     }
     else /* expander closed, clean up */
     {
          options = gtk_bin_get_child (GTK_BIN (expand));
          if (options)
              gtk_widget_destroy (options);
     }
     return;
}

static void
unexpand_options (GtkWidget *expand)
{
     gtk_expander_set_expanded (GTK_EXPANDER (expand), FALSE);
     return;
}

static void
import_file (const gchar *file, ImportFunction func)
{
     lingLesson *ls = func (file);

     if (!ls)
     {
          error_warning (_("Error on file access!"),
                         _("TODO"), _("TODO"));
          return;
     }
     
     settings->prefs->lessons = ling_lesson_add (settings->prefs->lessons, ls);

     /* inform the other widgets */
     event_emit_signal (main_app->objects, "lessons-changed");

     util_new_status (GTK_STATUSBAR (main_app->status), _("Lesson imported."));

     /* start the editor, so the user can save the lesson */
     util_info (_("The editor will be started now, so you can edit the lesson "
                  "and save it afterwards.\n"
                  "ATTENTION: You need to save it, else LingoTeach could "
                  "behave in an undefined manner!"));

     main_app->editor = gtkedit_init_editor ();
     gtk_widget_show_all (main_app->editor->window);

     return;
}


void
gtkimport_activate (void)
{
     GtkWidget *dlg_import;
     GtkWidget *frm_import;
     GtkWidget *align;

     GtkWidget *tbl_import;

     GtkWidget *lbl_file;
     GtkWidget *txt_file;
     GtkWidget *btn_file;

     GtkWidget *lbl_filter;
     GtkListStore *list;
     GtkWidget *view_import;
     GtkWidget *scr_win;
     GtkCellRenderer *renderer;
     GtkTreeSelection *select;
     GtkTreeIter iter;

     GtkWidget *exp_options;

     ImportFunction func = NULL;
     const gchar *file;

     gint result = 0;

     /* dialog */
     dlg_import = gtk_dialog_new_with_buttons (_("Import meanings"),
                                               GTK_WINDOW (main_app->window),
                                               GTK_DIALOG_MODAL |
                                               GTK_DIALOG_DESTROY_WITH_PARENT,
                                               GTK_STOCK_CANCEL,
                                               GTK_RESPONSE_CANCEL,
                                               GTK_STOCK_OK,
                                               GTK_RESPONSE_ACCEPT,
                                               NULL);
     gtk_dialog_set_has_separator (GTK_DIALOG (dlg_import), FALSE);
     gtk_container_set_border_width (GTK_CONTAINER (dlg_import),
                                     WIDGET_BORDER);
     gtk_window_set_resizable (GTK_WINDOW (dlg_import), FALSE);

     /* frame */
     frm_import = hig_frame_new (_("Import"));
     gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg_import)->vbox), frm_import,
                         TRUE, TRUE, 0);
     align = gtk_bin_get_child (GTK_BIN (frm_import));

     /* main table */
     tbl_import = hig_table_new (3, 3);
     gtk_container_set_border_width (GTK_CONTAINER (tbl_import), 0);
     gtk_container_add (GTK_CONTAINER (align), tbl_import);

     /* file selection */
     lbl_file = gtk_label_new_with_mnemonic (_("_File:"));
     gtk_table_attach (GTK_TABLE (tbl_import), lbl_file, 0, 1, 0, 1,
                       GTK_FILL, GTK_FILL, 0, 0);
     gtk_misc_set_alignment (GTK_MISC (lbl_file), 0.0, 0.5);

     txt_file = gtk_entry_new ();
     gtk_table_attach (GTK_TABLE (tbl_import), txt_file, 1, 2, 0, 1,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     gtk_tooltips_set_tip (main_app->tips, txt_file, _("The file to import"),
                           NULL);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_file), txt_file);

     btn_file = gtk_button_new_with_mnemonic (_("_Browse..."));
     g_signal_connect_swapped (G_OBJECT (btn_file), "clicked",
                               G_CALLBACK (util_get_fileselection),
                               txt_file);
     gtk_table_attach (GTK_TABLE (tbl_import), btn_file, 2, 3, 0, 1,
                       GTK_FILL, GTK_FILL, 0, 0);

     /* filters */
     lbl_filter = gtk_label_new_with_mnemonic (_("F_ilter:"));
     gtk_table_attach (GTK_TABLE (tbl_import), lbl_filter, 0, 1, 1, 2,
                       GTK_FILL, GTK_FILL, 0, 0);
     gtk_misc_set_alignment (GTK_MISC (lbl_filter), 0.0, 0.0);

     list = create_filter_list ();
     view_import = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_unref (list);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_filter), view_import);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_import),
                                                  IMP_NAME, _("Name"),
                                                  renderer, "text",
                                                  IMP_NAME, NULL);
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_import),
                                                  IMP_VERSION, _("Version"),
                                                  renderer, "text",
                                                  IMP_VERSION, NULL);
     select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view_import));
     gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

     /* scrolled window */
     scr_win = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_container_add (GTK_CONTAINER (scr_win), view_import);
     gtk_table_attach (GTK_TABLE (tbl_import), scr_win, 1, 3, 1, 2,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     
     /* filter options */
     exp_options = gtk_expander_new_with_mnemonic (_("Filter O_ptions"));
     gtk_table_attach (GTK_TABLE (tbl_import), exp_options, 0, 3, 2, 3,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     /* filter option callback */
     g_signal_connect (G_OBJECT (exp_options), "notify::expanded",
                       G_CALLBACK (import_filter_changed), view_import);
     g_signal_connect_swapped (G_OBJECT (view_import), "cursor-changed",
                               G_CALLBACK (unexpand_options), exp_options);

     gtk_widget_show_all (frm_import);

     result = gtk_dialog_run (GTK_DIALOG (dlg_import));
     switch (result)
     {
     case GTK_RESPONSE_ACCEPT:
          /* get the file and current filter */
          file = gtk_entry_get_text (GTK_ENTRY (txt_file));
          select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view_import));
          if (gtk_tree_selection_get_selected (select, NULL, &iter))
          {
               gtk_tree_model_get (GTK_TREE_MODEL (list), &iter,
                                   IMP_CALLBACK, &func, -1);
               import_file (file, func);
          }
          break;
          
     case GTK_RESPONSE_CANCEL:
     default:
          break;
     }
     gtk_widget_destroy (dlg_import);
     return;
}

void
gtkimport_register_filter (const gchar *name, const gchar *version,
                           OptionsFunction option, ImportFunction callback)
{
     ImportFilter *filter = g_new (ImportFilter, 1);

     filter->name = name;
     filter->version = version;
     filter->options = option;
     filter->callback = callback;

     list_import = g_slist_append (list_import, filter);
     return;
}

void
gtkimport_unregister_filter (const gchar *name, const gchar *version)
{
     ImportFilter *tmp;
     GSList *item = list_import;

     while (item)
     {
          tmp = (ImportFilter *) item->data;

          if (g_utf8_collate (tmp->name, name) == 0
              && g_utf8_collate (tmp->version, version) == 0)
          {
               list_import = g_slist_remove (list_import, item->data);
               return;
          }
          item = item->next;
     }
     /* TODO: add information about bad luck here */
     return;
}
