/* $Id: cfg.c,v 1.16 2005/05/10 17:18:38 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2001-2005 The LingoTeach Team. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <libxml/xpath.h>
#include <unistd.h>
#include <fcntl.h>

#include "defs.h"
#include "errors.h"
#include "lesson.h"
#include "cfg.h"

static char *val_types[] = {
     "soundpath", "browser", "player", NULL
};

#define QUERY "/lingoteach/general/value[@type='%s']"

static lingchar* get_setting (char *setting, xmlXPathContextPtr context);
static void write_val (xmlNodePtr node, lingoteachSettings *config,
                       lingConfValue which);


/*
 * extracts a specific value out of an xml context pointer and returns it
 * to the caller
 * the return value has to be freed.
 */
static lingchar*
get_setting (char *setting, xmlXPathContextPtr context)
{
     xmlXPathObjectPtr object = NULL;
     lingchar *query = NULL;
     lingchar *value = NULL;
     
     query = malloc (strlen (QUERY) + strlen (setting));
     if (!query)
     {
          error_warning (ERR_MEM_INFO,
                         _("A configuration value will not be set to the "
                           "correct value."), ERR_NOT_AVAILABLE);
          return NULL;
     }
     
     /* extract the value from the context */
     snprintf (query, strlen (QUERY) + strlen (setting), QUERY, setting);
     object = xmlXPathEval (query, context);
     free (query);
     if (!object)
     {
          error_warning (_("Value could not be read from configuration."),
                         _("Querying the configuration file did not bring "
                           "any result, thus a value will not be set."),
                         CONF_SOLUTION);
          return NULL;
     }
     value = xmlXPathCastToString (object);
     if (!value)
     {
          xmlXPathFreeObject (object);
          error_warning (_("Value could not be converted!"),
                         _("A read value could not be converted correctly "
                           "and thus will not be set."), CONF_SOLUTION);
          return NULL;
     }
     
     xmlXPathFreeObject (object);
     return value;
}

/* 
 * adds the content of a config entry to a node in order to write this
 * later into a configuration file
 */
static
void write_val (xmlNodePtr node, lingoteachSettings *config,
                lingConfValue which)
{
     switch (which)
     {
     case CONF_VAL_SOUND:
          xmlNodeSetContent (node, config->soundpath);
          break;
     case CONF_VAL_BROWSER:
          xmlNodeSetContent (node, config->browser);
          break;
     case CONF_VAL_PLAYER:
          xmlNodeSetContent (node, config->player);
          break;
     default: /* do nothing */
          break;
     }
     return;
}

/*
 * returns a new lingoteachSettings struct or NULL on error
 */ 
lingoteachSettings*
conf_init_new (void)
{
     lingoteachSettings *config = malloc (sizeof (lingoteachSettings));
     if (!config)
          return NULL;
     
     config->conf_dir = NULL;
     config->conf_file = NULL;
     config->soundpath = NULL;
     config->player = NULL;
     config->browser = NULL;
     config->languages = NULL;
     config->lessons = NULL;
     config->config = NULL;
     config->predef = FALSE;
     config->skip = FALSE;
     config->old_conf = FALSE;

     return config;
}

/*
 * loads the configuration data into the lingoteachSettings
 */
void
conf_load_config (lingoteachSettings *config)
{
     xmlDocPtr doc = NULL;
     xmlXPathContextPtr context = NULL;
     lingchar *value = NULL;
     int i = 0;

     debug ("Loading user configuration...\n");

     if (!config->config && !CONF_LOAD_DEFAULT_LANGUAGES (config))
          exit (EXIT_FAILURE);

     doc = xmlParseFile (config->conf_file);
     if (!doc)
     {
          error_warning (_("Configuration file could not be loaded!"),
                         _("The configuration file could not loaded, which "
                           "means, that it is possibly broken."),
                         CONF_SOLUTION);
          conf_load_defaults (config);
          return;
     }

     xmlXPathInit ();
     context = xmlXPathNewContext (doc);

     /* traverse the settings */
     while (val_types[i])
     {
          value = get_setting (val_types[i], context);
          if (!value || strlen (value) == 0)
               error_warning (_("A configuration value will not be set!"),
                              _("A configuration value will not be set to "
                                "the correct value."), CONF_SOLUTION);
          else
          {
               conf_set_preference_val (config, i, value);
               xmlFree (value);
          }
          i++;
          
     }
     xmlXPathFreeContext (context);
     xmlFreeDoc (doc);
     return;
}

/*
 * loads the languages from a file to the passed lingoteachSettings
 */
lingbool
conf_load_languages (lingoteachSettings *config, char *file)
{
     debug ("Loading languages from file...\n");

     config->config = ling_conf_init_config (XMLNAME, file,
                                             DATADIR"/"LANGUAGEDTD,
                                             DATADIR"/"LESSONDTD);

     if (!config->config)
     {
          error_warning (_("Incorrect language file!"),
                         _("The language file does not seem to be correct "
                           "and will not be loaded."),
                         _("Make sure, that the language file respects the "
                           "DTD definitions of LingoTeach."));
          return FALSE;
     }

     config->languages = ling_lang_get_languages (config->config);
     if (!config->languages)
     {
          error_warning (_("Languages could not be loaded!"),
                         _("The languages could not be loaded out of the "
                           "give language file."),
                         _("Make sure, that the language file respects the "
                           "DTD definitions of LingoTeach."));
          return FALSE;
     }

     return TRUE;
}

/*
 * saves the data of the lingoteachSettings to the conf_file 
 */
lingbool
conf_save_settings (lingoteachSettings *config)
{
     xmlDocPtr doc = NULL;
     xmlNodePtr root = NULL;
     xmlNodePtr node = NULL;
     lingchar *prop = NULL;
     lingbool found = FALSE;
     int i = 0;

     debug ("Saving configuration...\n");
     
     doc = xmlParseFile (config->conf_file);
     if (!doc)
     {
          error_warning (_("Configuration file error!"),
                         _("The configuration file seems to be errornous. "
                           "Your current settings will probably be lost."),
                         CONF_SOLUTION);
          return FALSE;
     }

     root = xmlDocGetRootElement (doc); /* skip <general> */
     root = root->children;
     while (root)
     {
          if (xmlStrcasecmp (root->name, "general") == 0)
               break;
          root = root->next;
     }
     if (!root)
          return FALSE;

     while (val_types[i])
     {
          found = FALSE;
          node = root->children;
          if (node)
          {
               while (node) /* walk through the nodes */
               {
                    if (xmlStrcasecmp (node->name, "value") == 0
                        && xmlHasProp (node, "type"))
                    {
                         /* found one, check the property */
                         prop = xmlGetProp (node, "type");
                         if (xmlStrcasecmp (prop, val_types[i]) == 0)
                         {
                              write_val (node, config, i);
                              found = TRUE;
                         }
                         xmlFree (prop);
                    }
                    node = node->next;
               }
          }

          if (!found) /* node does not exist, so create it */
          {
               node = xmlNewTextChild (root, NULL, "value", NULL);
               xmlNewProp (node, "type", val_types[i]);
               write_val (node, config, i);
          }
          i++;
     }
     
     xmlKeepBlanksDefault (0);
     if (!xmlSaveFormatFile (config->conf_file, doc, 0))
     {
          error_warning (_("Configuration file could not be saved!"),
                         _("The configuration file could not be saved. "
                           "Your current settings will probably be lost."),
                         CONF_SOLUTION);
          xmlFreeDoc (doc);
          return FALSE;
     }
     xmlFreeDoc (doc);
     return TRUE;
}

/*
 * sets the content of "what" to the passed config entry
 */
void
conf_set_preference_val (lingoteachSettings *config, lingConfValue which,
                         char *what)
{
     debug ("Setting val %s\n", what);
     char *buf = malloc (strlen (what) + 1);
     strncpy (buf, what, strlen (what) + 1);
     
     switch (which)
     {
     case CONF_VAL_SOUND:
          if (config->soundpath)
               free (config->soundpath);
          config->soundpath = buf;
          break;
     case CONF_VAL_BROWSER:
          if (config->browser)
               free (config->browser);
          config->browser = buf;
          break;
     case CONF_VAL_PLAYER:
          if (config->player)
               free (config->player);
          config->player = buf;
          break;
     default: /* do nothing */
          break;
     }

     return;
}

/*
 * frees a lingoteachSettings struct
 */
void 
conf_free_config (lingoteachSettings *config)
{
     lingLesson *tmp;

     debug ("Freeing settings...\n");

     if (config->conf_dir)
          free (config->conf_dir);
     if (config->conf_file)
          free (config->conf_file);
     if (config->soundpath)
          free (config->soundpath);
     if (config->browser)
          free (config->browser);
     if (config->languages)
          ling_strings_free (config->languages);
     
     while (config->lessons)
     {
          tmp = config->lessons;
          config->lessons = config->lessons->next;
          ling_lesson_free (tmp);
     }
     if (config->config)
          ling_conf_free_config (config->config);
     return;
}

void
conf_load_defaults (lingoteachSettings *config)
{
     lingLesson *tmp = NULL;
     char *tmp_file = NULL;
     debug ("Loading default configuration...\n");

     if (config->soundpath)
     {
          free (config->soundpath);
          config->soundpath = NULL;
     }
     if (config->browser)
     {
          free (config->browser);
          config->browser = NULL;
     }
     if (config->languages)
     {
          ling_strings_free (config->languages);
          config->languages = NULL;
     }
     
     /* set 'predefined' flag */
     config->predef = TRUE;
     config->old_conf = FALSE;

     if (config->lessons)
     {
          tmp = config->lessons;
          config->lessons = NULL;
     }
     
     if (!CONF_LOAD_DEFAULT_LANGUAGES (config))
          exit (EXIT_FAILURE);

     while (tmp)
     {
          lesson_load_lesson_to_prefs (config, ling_lesson_get_path (tmp));
          tmp = ling_lesson_remove (tmp, tmp);
     }

     /* load default config */
     tmp_file = config->conf_file;
     config->conf_file = malloc (strlen (DATADIR) + strlen (CONFFILE) + 2);
     snprintf (config->conf_file, strlen (DATADIR) + strlen (CONFFILE) + 2,
               "%s/%s", DATADIR, CONFFILE);
     conf_load_config (config);
     free (config->conf_file);
     config->conf_file = tmp_file;

     conf_create_config_file (config);
     conf_save_settings (config);

     return;
}


void
conf_create_config_file (lingoteachSettings *config)
{
     FILE *conf = NULL;

     /* create the user config */
     conf = fopen (config->conf_file, "w+");
     if (!conf)
          error_critical (_("Configuration file could not be created!"),
                          _("The configuration file could not be created "
                            "by the application."),
                          _("Please make sure, that you have proper "
                            "access rights in your home direcory and "
                            "enough disk space and/or quota left to "
                            "create the file."));

     fprintf (conf, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
     fprintf (conf, "<!DOCTYPE lingoteach SYSTEM \"config.dtd\">\n\n");
     fprintf (conf, "<lingoteach version=\"%s\">\n", VERSION);
     fprintf (conf, "  <general>\n");
     fprintf (conf, "  </general>\n");
     fprintf (conf, "</lingoteach>\n");
     fclose (conf);
     return;
}

char*
file_read_line (FILE *fp, int *ok)
{
     unsigned int size = 2;
     unsigned int n_size = 0;
     char ch = 0;
     char *line = malloc (size);

     if (!line)
     {
          *ok = -1;
          return NULL;
     }

     do
     {
          ch = fgetc (fp);
          /* bigger buffer needed? */
          if (n_size == size)
               size += 2; /* we allocate a bit more every time */
          line = realloc (line, size);
          if (!line)
          {
               *ok = -1;
               return NULL;
          }
          
          if (ch == '\n' || ch == EOF)
          {
               *ok = 0;
               line[n_size] = '\0';
               return line;
          }
          line[n_size++] = ch;
     }
     while (1);
}

/* stuff for backward compatibility */
#define OLD_CONF "config"

void 
conf_load_old_conf (lingoteachSettings *config)
{
     char *file = NULL; 
     char *line = NULL;
     char *val = NULL;
     FILE *fp = NULL;
     lingConfValue value = CONF_VAL_NONE;
     int l_ok = 0;

     debug ("Checking for old configuration...\n");

     /* we need to convert an old version */
     conf_create_config_file (config);

     file = malloc (strlen (config->conf_dir) + strlen (OLD_CONF) + 2);
     if (!file)
     {
          error_warning (ERR_MEM_INFO,
                         _("The old configuration file will not be read."),
                         ERR_NOT_AVAILABLE);

          conf_load_defaults (config);
          return;
     }

     snprintf (file, strlen (config->conf_dir) + strlen (OLD_CONF) + 2,
               "%s/%s", config->conf_dir, OLD_CONF);
     
     /* the file does not exist, check for the old config */
     fp = fopen (file, "r");
     if (!fp)
     {
          /* it does not exist, so run defaults */
          free (file);
          conf_load_defaults (config);
          return;
     }

     /* read and convert the content */
     line = file_read_line (fp, &l_ok);
     while (line && l_ok == 0)
     {
          val = line;
          while (*val++ != '=')
               ;
          
          /* now check the line and the value and put them into the
             correct preference slot... */
          if (strstr (line, "soundpath"))
               value = CONF_VAL_SOUND;
          else if (strstr (line, "browser"))
               value = CONF_VAL_BROWSER;
          else if (strstr (line, "player"))
               value = CONF_VAL_PLAYER;
          else 
               value = CONF_VAL_NONE;
          if (value != CONF_VAL_NONE)
               conf_set_preference_val (config, value, val);
          free (line);
          
          if (feof (fp) != 0)
               break;

          line = file_read_line (fp, &l_ok);
     }

     if (!config->config && !CONF_LOAD_DEFAULT_LANGUAGES (config))
          exit (EXIT_FAILURE);
   
     /* save the changed settings to the new file */
     conf_save_settings (config);
     config->old_conf = TRUE;

     return;
}

/* -- stuff for backward compatibility end -- */
