#include <stdio.h>
#include <string.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <linux/isdn.h>
#include <syslog.h>
#include <stdlib.h>

#include "global_vars.h"
#include "srv_def.h"

#ifndef ISDN_MAX_CHANNELS
#define ISDN_MAX_CHANNELS	8
#endif

int iface_up_normal(int, const char *);
int iface_up_isdn(const char *, const char *, int);
int iface_up_file(char *);

int iface_up(int socket_fd, struct line_t *line)
/* get active net-interface information;
   return 0 if up, if down -1, on error -2 */
{
    int ret;
    if ( !line )
    {
	syslog(LOG_ERR, "iface_up(): no line specified!!");
	return(-2);
    }
    switch ( line->con_type )
    {
	case CT_NETDEV:
	    ret = iface_up_normal(socket_fd, line->interface);
	    break;
	case CT_ISDN:
	    if ( (ret = iface_up_isdn(line->isdn_id, line->isdninfo, line->isdn_channel)) == -2 )
		syslog(LOG_ERR, "couldn't get devicestatus (con_type isdn) for line '%s'!", line->linename);
	    break;
	case CT_FILE:
	    ret = iface_up_file(line->con_status_file);
	    break;
	default:
	    syslog(LOG_ERR, "invalid con_type '%d' for line '%s'!", line->con_type, line->linename);
	    return(-2);
    }
    if ( ret == -2 ) ret++;
    return(ret);
}

int iface_up_normal(int socket_fd, const char *ifname)
/* copied from interface.c of fetchmail 4.7.6 and changed */
{
    struct ifreq request;

    /* see if the interface is up */
    strcpy(request.ifr_name, ifname);
    if (ioctl(socket_fd, SIOCGIFFLAGS, &request) < 0) return(-2);
    if (!(request.ifr_flags & IFF_RUNNING)) return(-1);
    /* OK, iface is up */
    return(0);
}

int iface_up_isdn(const char *id, const char *isdninfofile, int channel)
/* I've had a look at imon.c ... */
{
#define LC_ISDN_LINE_LEN	2048
    FILE *isdninfo;
    char *ii_idmap;
    char *idptr;
    char *ii_chmap;
    char *chptr;
    char *ii_usage;
    char *usptr;
    char *ii_flags;
    char *flptr;
    char drvid[15];
    int usage, chan, flags;
    int len, nmbr;
    static char  fbuf[LC_ISDN_LINE_LEN];

    /* maybe we have to malloc/fopen global vars. But then we have to lock */
    if ( !(isdninfo = fopen(isdninfofile, "r")) )
	{
		syslog(LOG_INFO, "unable to open isdninfo-file '%s'. Please check your config (keyword isdninfo).",
				isdninfofile);
		return(-2); // error!
	}

    setvbuf(isdninfo, fbuf, _IOFBF, sizeof(fbuf));

    
    if ( !(ii_idmap = malloc(LC_ISDN_LINE_LEN)) )
	{ fclose(isdninfo); return(-2); }
    if ( !(ii_chmap = malloc(LC_ISDN_LINE_LEN)) )
	{ free(ii_idmap); fclose(isdninfo); return(-2); }
    if ( !(ii_usage = malloc(LC_ISDN_LINE_LEN)) )
	{ free(ii_idmap); free(ii_chmap); fclose(isdninfo); return(-2); }
    if ( !(ii_flags = malloc(LC_ISDN_LINE_LEN)) )
	{ free(ii_idmap); free(ii_chmap); free(ii_usage); fclose(isdninfo); return(-2); }
    /* init finished, now we parse the file isdninfo */
    fgets(ii_idmap, LC_ISDN_LINE_LEN, isdninfo);
    fgets(ii_chmap, LC_ISDN_LINE_LEN, isdninfo);
    fgets(ii_usage, LC_ISDN_LINE_LEN, isdninfo); // drmap!!
    fgets(ii_usage, LC_ISDN_LINE_LEN, isdninfo); // the real 'usage' line...
    fgets(ii_flags, LC_ISDN_LINE_LEN, isdninfo);
    fclose(isdninfo);
    /* we got the information, now parse it */
    idptr = ii_idmap + 7;
    chptr = ii_chmap + 7;
    usptr = ii_usage + 7;
    flptr = ii_flags + 7;
    for (nmbr=0;nmbr<ISDN_MAX_CHANNELS;nmbr++)
    {
	sscanf(idptr, "%s %n", drvid, &len);
	idptr += len;
	sscanf(chptr, "%d %n", &chan, &len);
	chptr += len;
	sscanf(usptr, "%d %n", &usage, &len);
	usptr += len;
	sscanf(flptr, "%d %n", &flags, &len);
	flptr += len;
	// is this the right id/ch or do we have to go on?
//if (chan != -1 ) fprintf(stderr, "\ndrvid: '%s'         chan: %d\n", drvid, chan);
	if ( (!(strcmp(drvid, id))) && (chan == channel) )
	{ // ok, we got the right id/channel pair
	    free(ii_idmap);
	    free(ii_chmap);
	    free(ii_usage);
	    free(ii_flags);
	    if ( (usage & (~7)) && (flags & (1<<chan)) )
	    { // connection is up
			return(0);
	    }
	    else
	    { // down or dialing/closing
			return(-1);
	    }
	}
    }
//fprintf(stderr, "\nno such id/channel\n");
    free(ii_idmap);
    free(ii_chmap);
    free(ii_usage);
    free(ii_flags);
    return(-2); // we would have returned already if we found the id !
}

int iface_up_file(char *filename)
{
    int ret = -1;
    FILE *fhndl;
    if ( (!filename) || (!filename[0]) ) return(-2);
    if ( (fhndl = fopen(filename, "r")) )
    {
	fclose(fhndl);
	ret = 0;
    }
    return(ret);
}

/*
unsigned int iface_get_kb_from_string(char * str)
{
	unsigned int kb, len = strlen(str);
	char *blah;
	double fac = 1.024;
	len -= 3;
	str[len] = 0;
	kb = strtoul(str, &blah, 10);
	kb = (unsigned int)( kb / fac );
}
*/

/* TPUT_DEBUG: defined in srv_def.h, but for
   testing purposes we have it here too:
*/
//#define TPUT_DEBUG

char *iface_compare(char *statline, const char *iface)
{
	register char *start = statline;
	register char *end;
	while ( *start && (*start == ' ' || *start == '\t') ) start++;
	end = start;
	while ( *end && *end != ':' && *end != ' ' && *end != '\t' ) end++;
	if ( *end )
	{
		*end = 0;
		end++;
		if ( !strcmp(start, iface) )
		{
			#ifdef TPUT_DEBUG
				syslog(LOG_DEBUG, "iface found: name from statline: '%s', searched: '%s'",
							start, iface);
			#endif
			return end;
		}
	}
	return NULL;	
}

int iface_bcounts(const char *iface, unsigned int *recvd, unsigned int *sent)
{
	char *bufp;
    char sbuf[100]; //, sup[30], sdn[30];
	const char *statfilename = "/proc/net/dev";
    FILE *stats_file = fopen(statfilename, "r");

    if ( !stats_file )
	{ // can't read /proc/net/dev, log that only once...
		static char logged = 0;
		if ( !logged )
		{
			syslog(LOG_WARNING, "'%s' not readable. Throughput doesn't work.",
					statfilename);
			logged++;
		}
		*recvd = 0;
		*sent = 0;
		return(-1);
	}
	
	// strip first line (field explanation):
	fgets(sbuf, sizeof(sbuf), stats_file);

	// read other lines, search interface iface:
    while ( fgets(sbuf, sizeof(sbuf), stats_file) )
    {
		if ( (bufp = iface_compare(sbuf, iface)) )
		{ // ok, we got it
			#ifdef TPUT_DEBUG
				syslog(LOG_DEBUG, "bufp: '%s'\n", bufp);
			#endif

			sscanf(bufp, "%u %*d %*d %*d %*d %*d %*d %*d %u", recvd, sent);

			#ifdef TPUT_DEBUG
				syslog(LOG_DEBUG, "values: recvd = %d, sent = %d\n", *recvd, *sent);
			#endif

		    break;
		}
    }
    fclose(stats_file);
    return(0);
}
