/***********************************************************************************

	Copyright (C) 2007-2008 Ahmet Öztürk (aoz_2@yahoo.com)

	Parts of this file are loosely based on an example gcrypt program
	on http://punkroy.drque.net/

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/

#include <gtkmm/stock.h>
#include "helpers.hpp"

using namespace LIFEO;

// STRING PROCESSING
long
LIFEO::convert_string( const std::string &str )
{
	//TODO: add negative number support
	long f_result=0;	// result
	for (unsigned int i=0;
		 i<str.size() && i<10 && int (str[i])>='0' && int (str[i])<='9';
		 i++)
	{
		f_result = (f_result * 10) + int (str[i]) - '0';
	}
	return (f_result);
}

// COLOR PROCESSING
//TODO: this needs lots of love...
Gdk::Color
LIFEO::contrast( const Gdk::Color &bg, const Gdk::Color &fg )
{
	Gdk::Color contrast( fg );
	int dom_index = 0;	// 1,2,3 = R,G,B
	int existingcontrast =	abs( fg.get_red() - bg.get_red() ) +
							abs( fg.get_green() - bg.get_green() ) +
							abs( fg.get_blue() - bg.get_blue() );
	if( existingcontrast > 32767 )
		return contrast;

	if( fg.get_red() > fg.get_green() )
	{
		if( fg.get_red() > fg.get_blue() )
			dom_index = 1;
		else
			dom_index = 3;
	}
	else
	{
		if( fg.get_green() > fg.get_blue() )
			dom_index = 2;
		else
			dom_index = 3;
	}

	switch ( dom_index )
	{
		case 0:
		break;
		case 1:
			if( bg.get_red() > 32767 )
			{
				contrast.set_red( bg.get_red() - 32767 );// + existingcontrast );
//				contrast.set_green( 0 );
//				contrast.set_blue( 0 );
			}
			else
			{
				contrast.set_red( bg.get_red() + 32767 );//- existingcontrast );
//				contrast.set_green( bg.get_green() > fg.get_green() ?
//											fg.get_green()*0.5 :  );
//				contrast.set_blue( 0 );
			}
		break;
		case 2:
			contrast.set_green( bg.get_green() > 32767 ?
					fg.get_green() - 32767 :// + existingcontrast :
					fg.get_green() + 32767 );// - existingcontrast );
		break;
		case 3:
			contrast.set_blue( bg.get_blue() > 32767 ?
					bg.get_blue() - 32767 :// + existingcontrast :
					bg.get_blue() + 32767 );// - existingcontrast );
		break;
	}
	return contrast;
}

Gdk::Color
LIFEO::midtone( const Gdk::Color &c1, const Gdk::Color &c2 )
{
	Gdk::Color midtone;
	midtone.set_red( ( c1.get_red() + c2.get_red() )>>1 );
	midtone.set_green( ( c1.get_green() + c2.get_green() )>>1 );
	midtone.set_blue( ( c1.get_blue() + c2.get_blue() )>>1 );
	return midtone;
}

// FILE FUNCTIONS
std::ios::pos_type
LIFEO::get_file_size( std::ifstream &file )
{
   std::ios::pos_type size;
   const std::ios::pos_type currentPosition = file.tellg();

   file.seekg( 0, std::ios_base::end );
   size = file.tellg();
   file.seekg( currentPosition );

   return size;
}

// URL HANDLERS
void
LIFEO::open_url( Gtk::AboutDialog&, const Glib::ustring &link )
{
	std::string command( "x-www-browser " );
	command += link;
	//TODO: what else can be made with the return value?
	std::cout << system( command.c_str() ) << std::endl;
}

void
LIFEO::mail_to( Gtk::AboutDialog&, const Glib::ustring &link )
{
}

// ENCRYPTION
void
Cipher::create_iv( unsigned char **iv )
{
	// (Allocate memory for and fill with strong random data)
	*iv = (unsigned char *) gcry_random_bytes( Cipher::cIV_SIZE, GCRY_STRONG_RANDOM );

	if( ! *iv )
		throw Error( "Unable to create IV" );
}

void
Cipher::expand_key(	const char *passphrase,
					const unsigned char *salt,
					unsigned char **key )
{
	gcry_md_hd_t hash;
	gcry_error_t error = 0;
	int hashdigestsize;
	unsigned char *hashresult;

	// OPEN MESSAGE DIGEST ALGORITHM
	error = gcry_md_open( &hash, cHASH_ALGORITHM, 0 );
	if( error )
		throw Error( "Unable to open message digest algorithm: %s" ); //, gpg_strerror( Error ) );

	// RETRIVE DIGEST SIZE
	hashdigestsize = gcry_md_get_algo_dlen( cHASH_ALGORITHM );

	// ADD SALT TO HASH
	gcry_md_write( hash, salt, cSALT_SIZE );

	// ADD PASSPHRASE TO HASH
	gcry_md_write( hash , passphrase , strlen( passphrase ) );

	// FETCH DIGEST (THE EXPANDED KEY)
	hashresult = gcry_md_read( hash , cHASH_ALGORITHM );

	if( ! hashresult )
	{
		gcry_md_close( hash );
		throw Error( "Unable to finalize key" );
	}

	// ALLOCATE MEMORY FOR KEY
	// can't use the 'HashResult' because those resources are freed after the
	// hash is closed
	*key = new unsigned char[ cKEY_SIZE ];
	if( ! key )
	{
		gcry_md_close( hash );
		throw Error( "Unable to allocate memory for key" );
	}

	// DIGEST SIZE SMALLER THEN KEY SIZE?
	if( hashdigestsize < cKEY_SIZE )
	{
		// PAD KEY WITH '0' AT THE END
		memset( *key , 0 , cKEY_SIZE );

		// COPY EVERYTHING WE HAVE
		memcpy( *key , hashresult , hashdigestsize );
	}
	else
		// COPY ALL THE BYTES WE'RE USING
		memcpy( *key , hashresult , hashdigestsize );

	// FINISHED WITH HASH
	gcry_md_close( hash );
}

// create new expanded key
void
Cipher::create_new_key(	char const *passphrase,
						unsigned char **salt,
						unsigned char **key )
{
	// ALLOCATE MEMORY FOR AND FILL WITH STRONG RANDOM DATA
	*salt = (unsigned char *) gcry_random_bytes( cSALT_SIZE, GCRY_STRONG_RANDOM );

	if( ! *salt )
		throw Error( "Unable to create salt value" );

	expand_key( passphrase, *salt, key );
}

void
Cipher::encrypt_buffer (	unsigned char *buffer,
							size_t& size,
							const unsigned char *key,
							const unsigned char *iv )
{
	gcry_cipher_hd_t	cipher;
	gcry_error_t		error = 0;

	error = gcry_cipher_open( &cipher, cCIPHER_ALGORITHM, cCIPHER_MODE, 0 );

	if( error )
		throw Error( "Unable to initilize cipher: " ); // + gpg_strerror( Error ) );

	// GET KEY LENGTH
	int cipherKeyLength = gcry_cipher_get_algo_keylen( cCIPHER_ALGORITHM );
	if( ! cipherKeyLength )
		throw Error( "gcry_cipher_get_algo_keylen failed" );

	// SET KEY
	error = gcry_cipher_setkey( cipher, key, cipherKeyLength );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Cipher key setup failed: %s" ); //, gpg_strerror( Error ) );
	}

	// SET INITILIZING VECTOR (IV)
	error = gcry_cipher_setiv( cipher, iv, cIV_SIZE );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Unable to setup cipher IV: %s" );// , gpg_strerror( Error ) );
	}

	// ENCRYPT BUFFER TO SELF
	error = gcry_cipher_encrypt( cipher, buffer, size, NULL, 0 );

	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Encrption failed: %s" ); // , gpg_strerror( Error ) );
	}

	gcry_cipher_close( cipher );
}

void
Cipher::decrypt_buffer (	unsigned char * buffer,
							size_t& size,
							const unsigned char * key,
							const unsigned char * iv )
{
	gcry_cipher_hd_t	cipher;
	gcry_error_t		error = 0;

	error = gcry_cipher_open( &cipher, cCIPHER_ALGORITHM, cCIPHER_MODE, 0 );

	if( error )
		throw Error( "Unable to initilize cipher: " ); // + gpg_strerror( Error ) );

	// GET KEY LENGTH
	int cipherKeyLength = gcry_cipher_get_algo_keylen( cCIPHER_ALGORITHM );
	if( ! cipherKeyLength )
		throw Error( "gcry_cipher_get_algo_keylen failed" );

	// SET KEY
	error = gcry_cipher_setkey( cipher, key, cipherKeyLength );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Cipher key setup failed: %s" ); //, gpg_strerror( Error ) );
	}

	// SET IV
	error = gcry_cipher_setiv( cipher, iv, cIV_SIZE );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Unable to setup cipher IV: %s" );// , gpg_strerror( Error ) );
	}

	// DECRYPT BUFFER TO SELF
	error = gcry_cipher_decrypt( cipher, buffer, size, NULL, 0 );

	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Encrption failed: %s" ); // , gpg_strerror( Error ) );
	}

	gcry_cipher_close( cipher );
}

// ENTRYIDLETEXT
EntryIdletext::EntryIdletext( const Glib::ustring& idletext )
: Gtk::Entry(), m_idletext( idletext ), m_flag_empty( true )
{
}

void
EntryIdletext::on_map( void )
{
	Gdk::Color color = get_style()->get_text( Gtk::STATE_INSENSITIVE );

	m_idletext.insert( 0, "\">" );
	m_idletext.insert( 0, color.to_string() );
	m_idletext.insert( 0, "<span foreground=\"" );
	m_idletext.append( "</span>" );

	Gtk::Entry::on_map();
}

void
EntryIdletext::on_changed( void )
{
	m_flag_empty = get_text().empty();

	Gtk::Entry::on_changed();
}

bool
EntryIdletext::on_expose_event( GdkEventExpose *event )
{
	if( m_flag_empty && has_focus() == false)
		get_layout()->set_markup( m_idletext );

	return( Gtk::Entry::on_expose_event( event ) );
}

bool
EntryIdletext::on_focus_in_event( GdkEventFocus *event )
{
	if( m_flag_empty )
		get_layout()->set_markup( "" );

	return( Gtk::Entry::on_focus_in_event( event ) );
}

// MENUBUTTON
Menubutton::Menubutton(	const Gtk::StockID &stockid,
						const Glib::ustring &label,
						Gtk::ReliefStyle style,
						Gtk::IconSize iconsize )
	:	Gtk::ToggleButton()
{
	set_relief( style );
	m_menu = new Menu2;
	m_hbox = Gtk::manage( new Gtk::HBox );
	m_vbox = Gtk::manage( new Gtk::VBox );
	m_icon = Gtk::manage( new Gtk::Image( stockid, iconsize ) );
	m_label = Gtk::manage( new Gtk::Label( label ) );
	Gtk::Arrow *arrow = Gtk::manage(
			new Gtk::Arrow(Gtk::ARROW_DOWN, Gtk::SHADOW_IN) );

	m_hbox->pack_start(*m_vbox, Gtk::PACK_SHRINK);
	// the only purpose of Vbox is supporting Gtk::TOOLBAR_BOTH toolbar style
	m_vbox->pack_start( *m_icon, Gtk::PACK_SHRINK );
	m_hbox->pack_start( *m_label, Gtk::PACK_EXPAND_WIDGET );
	m_hbox->pack_start( *arrow, Gtk::PACK_SHRINK );
	add( *m_hbox );
	m_menu->attach_to_widget( *this );
	m_menu->signal_deactivate().connect( sigc::mem_fun( *this, &Menubutton::release ) );
	add_events( Gdk::BUTTON_PRESS_MASK );
}

//FIXME: code duplication
Menubutton::Menubutton(	const std::string& path,
						const Glib::ustring& label,
						Gtk::ReliefStyle style )
	:	Gtk::ToggleButton()
{
	set_relief( style );
	m_menu = new Menu2;
	m_hbox = Gtk::manage( new Gtk::HBox );
	m_vbox = Gtk::manage( new Gtk::VBox );
	m_icon = Gtk::manage( new Gtk::Image( path ) );
	m_label = Gtk::manage( new Gtk::Label( label ) );
	Gtk::Arrow *arrow = Gtk::manage(
			new Gtk::Arrow( Gtk::ARROW_DOWN, Gtk::SHADOW_IN ) );

	m_hbox->pack_start(*m_vbox, Gtk::PACK_SHRINK);
	// the only purpose of Vbox is supporting Gtk::TOOLBAR_BOTH toolbar style
	m_vbox->pack_start( *m_icon, Gtk::PACK_SHRINK );
	m_hbox->pack_start( *m_label, Gtk::PACK_EXPAND_WIDGET );
	m_hbox->pack_start( *arrow, Gtk::PACK_SHRINK );
	add( *m_hbox );
	m_menu->attach_to_widget( *this );
	m_menu->signal_deactivate().connect( sigc::mem_fun( *this, &Menubutton::release ) );
	add_events( Gdk::BUTTON_PRESS_MASK );
}

Menubutton::~Menubutton( void )
{
	delete m_menu;
}

Menu2*&
Menubutton::get_menu( void )
{
	return m_menu;
}

void
Menubutton::release( void )
{
	set_active( false );
}

void
Menubutton::set_label( const Glib::ustring& string )
{
	m_label->set_label( string );
}

void
Menubutton::get_menu_position (int& x, int& y, bool& push_in)
{
	get_window()->get_origin( x, y );
	x += get_allocation().get_x();
	y += get_allocation().get_y() + get_allocation().get_height();
	push_in = true;
}

bool
Menubutton::on_button_press_event (GdkEventButton *l_event)
{
	m_menu->popup( sigc::mem_fun(
			*this, &Menubutton::get_menu_position), l_event->button, l_event->time);

	set_active( true );

	return true;
}

// MENUTOOLBUTTON
Menutoolbutton::Menutoolbutton (const Gtk::StockID& stockid,
								const std::string& string)
	:	m_button( stockid, string, Gtk::RELIEF_NONE, Gtk::ICON_SIZE_LARGE_TOOLBAR )
{
	add( m_button );
}

Menu2*&
Menutoolbutton::get_menu (void)
{
	return m_button.m_menu;
}

void Menutoolbutton::on_toolbar_reconfigured (void)
{
	switch ( get_toolbar_style() )
	{
		case Gtk::TOOLBAR_BOTH:
			if( m_button.m_label->get_parent() == m_button.m_hbox )
			{
				m_button.m_hbox->remove( *m_button.m_label );
				m_button.m_vbox->pack_start ( *m_button.m_label );
			}
			m_button.m_label->show();
			m_button.m_icon->show();
		break;
		case Gtk::TOOLBAR_BOTH_HORIZ:
			if( m_button.m_label->get_parent() == m_button.m_vbox ) {
				m_button.m_vbox->remove( *m_button.m_label );
				m_button.m_hbox->pack_start ( *m_button.m_label );
				m_button.m_hbox->reorder_child ( *m_button.m_label, 1 );
			}
			m_button.m_label->show();
			m_button.m_icon->show();
		break;
		case Gtk::TOOLBAR_ICONS:
			m_button.m_label->hide();
			m_button.m_icon->show();
		break;
		case Gtk::TOOLBAR_TEXT:
			m_button.m_icon->hide();
			m_button.m_label->show();
		break;
		default:
		break;
	}
}

// MENUITEMRECENT
MenuitemRecent::MenuitemRecent( const std::string &path )
	:	Gtk::MenuItem(), m_icon_remove( Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU ),
	m_path( path )
{
	Gtk::Label		*label = Gtk::manage( new Gtk::Label(
											Glib::filename_display_basename( path ) ) );
	Gtk::HBox		*hbox = Gtk::manage( new Gtk::HBox );

	label->set_justify( Gtk::JUSTIFY_LEFT );
	label->set_alignment( Gtk::ALIGN_LEFT );
	label->set_ellipsize( Pango::ELLIPSIZE_START );
	label->set_tooltip_text( path );
	m_icon_remove.set_tooltip_text( _("Remove from list") );
	hbox->set_spacing( 5 );

	hbox->pack_start( *label );
	hbox->pack_start( m_icon_remove, Gtk::PACK_SHRINK );
	this->add( *hbox );

	hbox->show();
	label->show();
}

bool
MenuitemRecent::on_motion_notify_event( GdkEventMotion *event )
{
	m_icon_remove.show();

	return Gtk::MenuItem::on_motion_notify_event( event );
}

bool
MenuitemRecent::on_leave_notify_event( GdkEventCrossing *event )
{
	m_icon_remove.hide();

	return Gtk::MenuItem::on_leave_notify_event( event );
}

bool
MenuitemRecent::on_button_release_event( GdkEventButton *event )
{
	if(	event->x >= m_icon_remove.get_allocation().get_x() &&
		event->x < m_icon_remove.get_allocation().get_x() +
					m_icon_remove.get_allocation().get_width() )
	{
		m_signal_removerecent.emit( m_path );
		return true;
	}

	return Gtk::MenuItem::on_button_press_event( event );
}



// FILEBUTTONRECENT
FilebuttonRecent::FilebuttonRecent( const Glib::ustring& label_fallback )
	:	Menubutton( ICONDIR "/diary-16.png", label_fallback, Gtk::RELIEF_NORMAL ),
	m_icon_new( Gtk::Stock::NEW, Gtk::ICON_SIZE_MENU ),
	m_icon_browse( Gtk::Stock::OPEN, Gtk::ICON_SIZE_MENU ),
	m_recentcount( 0 )
{
	m_hbox->set_spacing( 4 );

	m_label->set_ellipsize( Pango::ELLIPSIZE_START );
	m_label->set_alignment( Gtk::ALIGN_LEFT );

	m_menu->items().push_back( Gtk::Menu_Helpers::SeparatorElem() );

	m_menu->items().push_back(
			Gtk::Menu_Helpers::ImageMenuElem( _("_Browse For a Diary..."), m_icon_browse,
					sigc::mem_fun( *this,
							&FilebuttonRecent::show_filechooser) ) );
	m_menu->items().push_back(
			Gtk::Menu_Helpers::ImageMenuElem( _("_Create A New Diary..."), m_icon_new,
					sigc::mem_fun( *this,
							&FilebuttonRecent::on_create_file) ) );

	m_menu->show_all_children();

	// ACCEPT DROPPED FILES
	//~ std::list< Gtk::TargetEntry > list_targets;
	//~ list_targets.push_back( Gtk::TargetEntry( "STRING" ) );
	//~ list_targets.push_back( Gtk::TargetEntry( "text/plain" ) );
	//~ this->drag_dest_set( list_targets );
	// TODO: i could not figure out how this is supposed to work and replicated
	// almost blindly bmpx' approach.
	drag_dest_set( Gtk::DEST_DEFAULT_ALL, ( Gdk::ACTION_COPY | Gdk::ACTION_MOVE ) );
	drag_dest_add_uri_targets();
}

std::string
FilebuttonRecent::get_filename( void ) const
{
	return( *m_list_recent.begin() );
}

//std::string
//FilebuttonRecent::get_filename_recent (int index) const {
//	if( index >= RecentCount )
//		return "";
//	else {
//		std::list< std::string >::iterator l_iter;
//		for ( int i = 0; i < index; i++ ) l_iter++;
//		return (*l_iter);
//	}
//}

void
FilebuttonRecent::set_filename( const std::string &path )
{
	if( m_recentcount > 0 )
		if( path == ( *m_list_recent.begin() ) )
			return;

	m_label->set_label( Glib::filename_display_basename( path ) );
	set_tooltip_text( path );
	add_recent( path );
	m_signal_selectionchanged.emit();
}

void
FilebuttonRecent::update_filename()
{
	if( m_recentcount > 0 )
	{
		m_label->set_label( Glib::filename_display_basename( *m_list_recent.begin() ) );
		set_tooltip_text( *m_list_recent.begin() );
	}
}

bool
FilebuttonRecent::add_recent( const std::string &path )
{
	// do not add a file twice:
	Gtk::Menu_Helpers::MenuList::iterator iter_menu = m_menu->items().begin();
	std::list< std::string >::iterator iter_recent = m_list_recent.begin();
	int i;
	for( i = 0; i < m_recentcount; i++ )
	{
		if( ( *iter_recent ) == path )
		{
			m_menu->items().erase( iter_menu );
			m_list_recent.erase( iter_recent );
			break;
		}
		else
		{
			iter_recent++;
			iter_menu++;
		}
	}

	MenuitemRecent *menuitem = Gtk::manage( new MenuitemRecent( path ) );

	menuitem->signal_activate().connect(
			sigc::bind(	sigc::mem_fun( *this, &FilebuttonRecent::set_filename ),
						path ) );
	menuitem->signal_removerecent().connect(
			sigc::mem_fun( *this, &FilebuttonRecent::remove_recent ) );

	m_menu->items().push_front( Gtk::Menu_Helpers::Element( *menuitem ) );

	menuitem->show();

	m_list_recent.push_front( path );

	// if nothing is deleted:
	if( i == m_recentcount )
	{
		if( m_recentcount == MAX_RECENT_FILE_COUNT )
			m_menu->items().erase( --iter_menu );
		else
			m_recentcount++;
	}

	return true;	// reserved
}

void
FilebuttonRecent::remove_recent( const std::string &path )
{
	Gtk::Menu_Helpers::MenuList::iterator	iter_menu = m_menu->items().begin();
	std::list< std::string >::iterator		iter_recent = m_list_recent.begin();
	int i;
	for( i = 0; i < m_recentcount; i++ )
	{
		if( ( *iter_recent ) == path )
		{
			m_menu->items().erase( iter_menu );
			m_list_recent.erase( iter_recent );
			m_recentcount--;
			break;
		}
		else
		{
			iter_recent++;
			iter_menu++;
		}
	}
	// update current path if first item is deleted
	if( i == 0 && m_recentcount > 0 )
	{
		m_label->set_label( Glib::filename_display_basename( *m_list_recent.begin() ) );
		set_tooltip_text( *m_list_recent.begin() );
		m_signal_selectionchanged.emit();
	}
}

void
FilebuttonRecent::show_filechooser( void )
{
	m_filechooserdialog = new Gtk::FileChooserDialog(
	// TRANSLATORS: this is the title of filechooser dialog for selecting a new diary
			_("Select a Diary"), Gtk::FILE_CHOOSER_ACTION_OPEN );
	m_filechooserdialog->set_filename( m_list_recent.size() > 0 ?
										*m_list_recent.begin() : Glib::get_home_dir() );
	m_filechooserdialog->add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	m_filechooserdialog->add_button( Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT );
	FilefilterAny		filter_any;
	FilefilterDiary		filter_diary;
	m_filechooserdialog->add_filter( filter_any );
	m_filechooserdialog->add_filter( filter_diary );
	m_filechooserdialog->set_filter( filter_diary );
	if( m_filechooserdialog->run() == Gtk::RESPONSE_ACCEPT )
	{
		set_filename( m_filechooserdialog->get_filename() );
	}

	delete m_filechooserdialog;
	m_filechooserdialog= NULL;
}

void
FilebuttonRecent::on_create_file( void )
{
	m_signal_createfile.emit();
}

void
FilebuttonRecent::on_drag_data_received(	const Glib::RefPtr<Gdk::DragContext>& context,
											int, int,
											const Gtk::SelectionData& seldata,
											guint info,
											guint time )
{
	if( seldata.get_length() < 0 )
		return;

	Glib::ustring uri = seldata.get_data_as_string();
	std::string filename = uri.substr( 0, uri.find('\n') - 1 );
	filename = Glib::filename_from_uri( filename );

	if( Glib::file_test( filename, Glib::FILE_TEST_IS_DIR ) )
		return;

	set_filename( filename );
	std::cout << "dropped: " << filename << std::endl;

	context->drag_finish( true, false, time );
}

void
FilebuttonRecent::on_size_allocate( Gtk::Allocation &allocation )
{
	Menubutton::on_size_allocate( allocation );
	m_menu->set_size_request( allocation.get_width(), -1 );
}

SignalVoid
FilebuttonRecent::signal_selection_changed( void )
{
	return m_signal_selectionchanged;
}

SignalVoid
FilebuttonRecent::signal_create_file( void )
{
	return m_signal_createfile;
}
