/***********************************************************************************

	Copyright (C) 2009 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <gtkmm/table.h>

#include "helpers.hpp"
#include "dialogtageditor.hpp"


namespace LIFEO
{
	ColrecTag *colrec_tag;
}


using namespace LIFEO;


Tagwidget::Tagwidget( OperationType type )
	:	m_entry( _( "Tag name" ) ),
	m_entrycompletion( Gtk::EntryCompletion::create() ),
	m_flag_activatable( false ), m_flag_empty( true ), m_operation( type )
{
	Gtk::Bin::remove();
	// new entry widget is added in on_remove() to prevent warnings

	m_entrycompletion->set_popup_completion( true );
	m_entrycompletion->set_popup_single_match( true );

	m_entry.set_completion( m_entrycompletion );
	m_entry.set_icon_from_stock(	Gtk::Stock::NO,
									Gtk::ENTRY_ICON_SECONDARY );

	m_entry.signal_activate().connect(
			sigc::mem_fun( *this, &Tagwidget::on_activate ) );

	m_entry.signal_icon_release().connect(
			sigc::mem_fun( *this, &Tagwidget::handle_icon_release ) );
}

void
Tagwidget::on_remove( Gtk::Widget* widget )
{
	// this is the only solution I could find to prevent Gtk from...
	// ...complaining when a ComboBoxText's Entry is removed from it
	if( widget != &m_entry )
		Gtk::Container::add( m_entry );
	else
		Gtk::Container::on_remove( widget );
}

void
Tagwidget::set_model( const Glib::RefPtr< Gtk::TreeModel > &model )
{
	ComboBoxEntry::set_model( model );
	m_entrycompletion->set_model( model );
	m_model = model;
}

void
Tagwidget::set_text_column( const Gtk::TreeModelColumnBase& column )
{
	ComboBoxEntry::set_text_column( column );
	m_entrycompletion->set_text_column( column );
}

void
Tagwidget::on_activate( void )
{
	if( m_flag_activatable )
		m_signal_activate.emit();
}

void
Tagwidget::on_changed( void )
{
	if( ! m_model )
		return;

	Glib::ustring				tagname = m_entry.get_text();
	Glib::ustring::size_type	size = tagname.size();

	// disable '<' and '>' to prevent markup tags in user strings
	// (markup is meant to be used by program to indicate states etc.)
	for( Glib::ustring::size_type i = 0; i < size; i++ )
		if( tagname[ i ] == '<' || tagname[ i ] == '>' )
			m_entry.delete_text( i, i + 1 );
	// get the updated text:
	tagname = m_entry.get_text();

	m_flag_activatable = ( tagname.size() > 0 );
	m_flag_empty = ! m_flag_activatable;

	if( m_flag_empty )
	{
		m_entry.set_icon_from_stock(	Gtk::Stock::NO,
										Gtk::ENTRY_ICON_SECONDARY );
	}
	else
	{
		Tag *tag = Tag::get_tag( tagname );
		if( tag == NULL )
		{
			m_entry.set_icon_from_stock(
					m_operation == OT_EDIT ? Gtk::Stock::OK : Gtk::Stock::NEW,
					Gtk::ENTRY_ICON_SECONDARY );
		}
		else
		if( m_operation == OT_EDIT )
		{
			m_entry.set_icon_from_stock(	Gtk::Stock::NO,
											Gtk::ENTRY_ICON_SECONDARY );
			m_flag_activatable = false;
		}
		else
		{
			bool has_tag = true;
			for(	Gtk::TreeIter iter_tags = get_model()->children().begin();
					iter_tags != get_model()->children().end();
					iter_tags++ )
			{
				if( ( *iter_tags )[ colrec_tag->m_ptr2tag ] == tag )
				{
					// if it is in the available tags list, entry does not have it
					has_tag = false;
					break;
				}
			}

			if( has_tag )
				m_entry.set_icon_from_stock(	Gtk::Stock::REMOVE,
												Gtk::ENTRY_ICON_SECONDARY );
			else
				m_entry.set_icon_from_stock(	Gtk::Stock::ADD,
												Gtk::ENTRY_ICON_SECONDARY );
		}
	}

	m_entry.set_icon_activatable( m_flag_activatable, Gtk::ENTRY_ICON_SECONDARY );

	Gtk::ComboBoxEntry::on_changed();
}

void
Tagwidget::handle_icon_release( Gtk::EntryIconPosition, const GdkEventButton* event )
{
	if( event->button == 1 )	// left click
		on_activate();
}

// TAG EDITOR DIALOG
DialogTageditor::DialogTageditor( Tag *ptr2tag )
	:	Gtk::Dialog( ptr2tag ? _("Edit Tag") : _("Create A New Tag") ),
	m_ptr2tag( ptr2tag )
{
	Gtk::Table *table;
	Gtk::Label *label_name;

	try
	{
		table = Gtk::manage( new Gtk::Table );
		label_name = Gtk::manage( new Gtk::Label( _("Tag Name:") ) );
		m_tagwidget = Gtk::manage( new Tagwidget( Tagwidget::OT_EDIT ) );
	}
	catch ( std::exception &e )
	{
		throw e;
	}

	m_liststore_tag = Gtk::ListStore::create( *colrec_tag );

	Gtk::TreeRow row;
	for(	Tag::Tagpool::const_iterator iter = Tag::get_pool().begin();
			iter != Tag::get_pool().end();
			iter++ )
	{
		row = *( m_liststore_tag->append() );
		row[ colrec_tag->m_name ] = ( *iter )->get_name();
		row[ colrec_tag->m_ptr2tag ] = ( *iter );
	}

	m_tagwidget->set_model( m_liststore_tag );
	m_tagwidget->set_text_column( colrec_tag->m_name );

	table->set_spacings( 5 );
	table->set_border_width( 5 );
	table->attach( *label_name, 0, 1, 0, 1, Gtk::SHRINK, Gtk::SHRINK );
	table->attach( *m_tagwidget, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	get_vbox()->pack_start( *table );
	get_vbox()->set_spacing( 5 );

	add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );

	if( ptr2tag )
	{
		m_tagwidget->get_entry()->set_text( ptr2tag->get_name() );
		m_button_create = add_button( Gtk::Stock::APPLY, Gtk::RESPONSE_OK  );
	}
//	else
//	{
//		m_button_create = add_button( _("C_reate Tag"), Gtk::RESPONSE_OK );
//	}

	set_default( *m_button_create );
	m_button_create->set_sensitive( false );

	m_tagwidget->signal_changed().connect( sigc::mem_fun(
			*this, &DialogTageditor::handle_tagname_changed ) );

	m_tagwidget->signal_activate().connect( sigc::bind(
			sigc::mem_fun( *this, &DialogTageditor::response ),
			Gtk::RESPONSE_OK ) );

	show_all();
}

Glib::ustring
DialogTageditor::get_tagname( void )
{
	return m_tagname;
}

void
DialogTageditor::handle_tagname_changed( void )
{
	m_tagname = m_tagwidget->get_active_text();
	m_button_create->set_sensitive( m_tagwidget->is_activatable() );
}

