/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <cstdlib>
#include <cassert>

#include "lifeograph.hpp"
#include "dialog_preferences.hpp"


using namespace LIFEO;


// CONSTRUCTOR
Lifeograph::Lifeograph( int argc, char *argv[] )
:	m_about_dialog( NULL ), m_hpaned_main( NULL ), m_view_login( NULL ),
	m_theme_view( NULL ),
	m_programpath( argv[ 0 ] ),
	m_flag_force_welcome( false )
{
	Cipher::init();

	UndoManager::m = new UndoManager;
	Diary::d = new Diary;
	ListData::colrec = new ListData::Colrec;

	// CONFIGURATION
	m_flag_firsttimeuser = ! settings.read();
	set_default_size( settings.width, settings.height );
	if( settings.position_x != POSITION_NOTSET && settings.position_y != POSITION_NOTSET )
		move( settings.position_x, settings.position_y );
    if( settings.state_maximized )
        maximize();

	// COMMANDLINE OPTIONS
	for( int i = 1; i < argc; i++ )
	{
		if( ! strcmp( argv[ i ], "--open" ) || ! strcmp( argv[ i ], "-o" ) )
		{
			if( i < argc )
			{
				if( Glib::file_test( argv[ ++i ], Glib::FILE_TEST_EXISTS ) )
				{
					if( ! Glib::file_test( argv[ i ], Glib::FILE_TEST_IS_DIR ) )
					{
						ViewLogin::m_path_cur = argv[ i ];
						m_flag_open_directly = true;
					}
				}
				if( ! m_flag_open_directly )
				    Error( "File cannot be opened" );
			}
			else
				Error( "No path provided" );
		}
		else
		if( ! strcmp( argv[ i ], "--force-welcome" ) )
		{
			m_flag_force_welcome = true;
		}
		else
		if( ! strcmp( argv[ i ], "--ignore-locks" ) )
		{
			Diary::d->m_flag_ignore_locks = true;
		}
        else
        if( ! strcmp( argv[ i ], "--read-only" ) || ! strcmp( argv[ i ], "-r" ) )
        {
            m_flag_read_only = true;
        }
		else
		if( Glib::file_test( argv[ i ], Glib::FILE_TEST_EXISTS ) &&
		    Glib::file_test( argv[ i ], Glib::FILE_TEST_IS_DIR ) == false )
		{
			    ViewLogin::m_path_cur = argv[ i ];
				m_flag_open_directly = true;
		}
        else
        {
            print_info( std::string( "Unrecognized argument: " ) + argv[ i ] );
        }
	}

	// GTKBUILDER
	load_gui( UIDIR "/lifeograph.ui" );

    // ENCHANT
    s_enchant_broker = enchant_broker_init();
    enchant_broker_list_dicts( s_enchant_broker, fill_lang_list_cb, NULL );

    // DRAW UI
    // TODO: make sure that this is execute after Gtk::Main::run()
    if( m_flag_firsttimeuser || m_flag_force_welcome )
        draw_welcomescreen();
    else
        draw_loginscreen();
}

Lifeograph::~Lifeograph( void )
{
    remove();
    delete UndoManager::m;

    if( m_view_login )
        delete m_view_login;

    if( Diary::d )
        delete Diary::d;

    if( s_enchant_broker )
    {
        enchant_broker_free( s_enchant_broker );
        s_enchant_broker = NULL;
	}
}

bool
Lifeograph::on_event( GdkEvent* )
{
    if( ! settings.autologout || loginstatus != LOGGED_IN || ! Diary::d->is_encrypted() ||
        m_flag_auto_logout_frozen )
		return false;
	if( m_secondsremaining < LOGOUT_COUNTDOWN )
		update_title();
	// restart
	m_secondsremaining = LOGOUT_COUNTDOWN;
	m_connection_timeout.disconnect();
	m_connection_timeout = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( this, &Lifeograph::handle_idle ),
			settings.idletime - LOGOUT_COUNTDOWN );

	return false;
}

bool
Lifeograph::on_delete_event( GdkEventAny* )
{
	// BEWARE: handle_button_quit_clicked() calls this function

	PRINT_DEBUG( "on_delete_event()" );

	if( loginstatus == LOGGED_IN )
		if( ! finish_editing() )
			return true;

	// SAVE SETTINGS
	get_size( settings.width, settings.height );
	get_position( settings.position_x, settings.position_y );
    settings.state_maximized = ( get_window()->get_state() & Gdk::WINDOW_STATE_MAXIMIZED );

	settings.write();

	return false;
}

bool
Lifeograph::handle_idle()
{
    if( ! settings.autologout )
        return false;

	if( m_secondsremaining > 0 )
	{
		set_title( Glib::ustring::compose(
				_( "<<<%1 SECONDS TO LOG OUT>>>" ), m_secondsremaining ) );
		m_connection_timeout = Glib::signal_timeout().connect_seconds(
				sigc::mem_fun( *this, &Lifeograph::handle_idle ), 1 );
		m_secondsremaining--;
		return false;
	}

	loginstatus = LOGGED_TIME_OUT;
	logout( true );
	return false;
}

void
Lifeograph::freeze_auto_logout( void )
{
    m_flag_auto_logout_frozen = true;
    if( m_connection_timeout.connected() )
        m_connection_timeout.block();
    update_title();
}

void
Lifeograph::unfreeze_auto_logout( void )
{
    m_flag_auto_logout_frozen = false;
    if( m_connection_timeout.connected() )
        m_connection_timeout.unblock();
}

bool
Lifeograph::write_backup( void )
{
	view_entry->sync();

	std::string path_backup( Diary::d->m_path + LOCK_SUFFIX );
	Result result( Diary::d->m_passphrase.empty() ?
			Diary::d->write_plain( path_backup ) : Diary::d->write_encrypted( path_backup ) );

	return( result == SUCCESS );
}

void
Lifeograph::show_about( void )
{
    if( m_about_dialog == NULL )
    {
        Lifeobase::builder->get_widget( "aboutdialog", m_about_dialog );
        m_about_dialog->set_name( PROGRAM_NAME );
        m_about_dialog->set_version( PROGRAM_VERSION_STRING );
        // XXX if needed: m_about_dialog->signal_activate_link().connect( open_url );
        m_about_dialog->set_transient_for( *this );
    }

    m_about_dialog->run();
    m_about_dialog->hide();
}

void
Lifeograph::handle_undo( void )
{
	UndoManager::m->undo();
}

void
Lifeograph::handle_redo( void )
{
	UndoManager::m->redo();
}

bool
Lifeograph::finish_editing( bool opt_save )
{
	// SAVING
	settings.position_pane = m_hpaned_main->get_position();
	settings.position_pane_tags = m_hpaned_entry->get_position();

    // files added to recent list here if not already there
    if( ! Diary::d->get_path().empty() )
        if( stock_diaries.find( Diary::d->get_path() ) == stock_diaries.end() )
            settings.recentfiles.insert( Diary::d->get_path() );

	if( ! Diary::d->is_read_only() )
	{
	    view_entry->sync();
        Diary::d->m_last_elem = panel_main->get_cur_elem()->get_id();

        if( opt_save )
        {
            if( Diary::d->write() != SUCCESS )
            {
                Gtk::MessageDialog messagedialog(	*this,
                                                    "",
                                                    false,
                                                    Gtk::MESSAGE_WARNING,
                                                    Gtk::BUTTONS_OK,
                                                    true );
                messagedialog.set_message( _( STRING::CANNOT_WRITE ) );
                messagedialog.set_secondary_text( _( STRING::CANNOT_WRITE_SUB ) );
                messagedialog.run();

                return false;
            }
        }
        else
        {
            Gtk::MessageDialog messagedialog(	*this,
                                                "",
                                                false,
                                                Gtk::MESSAGE_WARNING,
                                                Gtk::BUTTONS_CANCEL,
                                                true );
            messagedialog.set_message(
                _( "Are you sure you want to log out without saving?" ) );
            messagedialog.set_secondary_text( Glib::ustring::compose(
                _( "Your changes will be backed up in %1. "
                   "If you exit normally, your diary is saved automatically." ),
                "<b>" + Diary::d->get_path() + ".~unsaved~</b>" ), true );
            messagedialog.add_button( _( "_Log out without Saving" ), Gtk::RESPONSE_ACCEPT );

            if( messagedialog.run() !=  Gtk::RESPONSE_ACCEPT )
                return false;
            // back up changes
            Diary::d->write( Diary::d->get_path() + ".~unsaved~" );
        }
	}

	// CLEARING
	// TODO: m_loginstatus = LOGGING_OUT_IN_PROGRESS;

    m_actions_read->set_sensitive( false );

	m_signal_logout.emit();	// only for DialogEvent

	m_internaloperation++;
	panel_main->handle_logout();
	panel_diary->handle_logout();
	view_entry->get_buffer()->handle_logout();
	panel_extra->handle_logout();

	if( loginstatus == LOGGED_IN )
		loginstatus = LOGGED_OUT;
	Diary::d->clear();
	m_connection_timeout.disconnect();
	m_connection_backup.disconnect();
	m_internaloperation--;

	return true;
}

void
Lifeograph::logout( bool opt_save )
{
    m_flag_open_directly = false;   // should be reset to prevent logging in again
    if( finish_editing( opt_save ) )
        m_view_login->handle_logout();
}

bool
Lifeograph::confirm_dismiss_element( void )
{
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
							    Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return false;
	else
		return true;
}

Gtk::TreeRow
Lifeograph::get_element_row( const DiaryElement *element )
{
	switch( element->get_type() )
	{
		case DiaryElement::ET_ENTRY:
		case DiaryElement::ET_CHAPTER:
			return( panel_diary->get_row( element->m_list_data->treepath ) );

		case DiaryElement::ET_TAG:
		case DiaryElement::ET_TAG_CTG:
		case DiaryElement::ET_THEME:
			return( panel_extra->get_row( element->m_list_data->treepath ) );
		default:
			return( Gtk::TreeRow() );
	}
}

void
Lifeograph::update_title( void )
{
	Glib::ustring title( PROGRAM_NAME );

	if( loginstatus == LOGGED_IN )
	{
		title += " - ";
		title += Glib::filename_display_basename( Diary::d->get_path() );

		if( Diary::d->is_read_only() )
		{
		    title += " <";
		    title += "Read Only";
		    title += ">";
		}
	}

	set_title( title );
}

// MODE DRAWING FUNCTIONS
void
Lifeograph::draw_welcomescreen( void )
{
	Gtk::Alignment	*alignment =
			Gtk::manage( new Gtk::Alignment( 0.5, 0.5, 0.1, 0 ) );
	Gtk::Label		*label_salutation =
			Gtk::manage( new Gtk::Label( _( STRING::SALUTATION ), 0.5, 0.5 ) );
	Gtk::Button		*button_start =
			Gtk::manage( new Gtk::Button( _("_Continue"), true ) );
	Gtk::Image		*icon_forward =
			Gtk::manage( new Gtk::Image( Gtk::Stock::GO_FORWARD, Gtk::ICON_SIZE_MENU ) );
	Gtk::Box		*vbox =
			Gtk::manage( new Gtk::Box( Gtk::ORIENTATION_VERTICAL, 50 ) );

	alignment->set_padding( 120, 120, 150, 150 );
	add( *alignment );

	label_salutation->set_use_markup( true );

	button_start->set_image( *icon_forward );
	button_start->set_image_position( Gtk::POS_RIGHT );

	vbox->pack_start( *label_salutation );
	vbox->pack_start( *button_start );

	alignment->add( *vbox );

	button_start->signal_clicked().connect(
		sigc::mem_fun( *this, &Lifeograph::draw_loginscreen ) );

	show_all();
}

void
Lifeograph::draw_loginscreen( void )
{
    // only to be called on start up
    assert( m_view_login == NULL );

    try
    {
        m_view_login = new ViewLogin;
        m_view_login->handle_start();
    }
    catch( ... )
    {
        throw LIFEO::Error( "creation of login view failed" );
    }
}

void
Lifeograph::login( void )
{
    remove();

    if( m_hpaned_main == NULL )
    {
        Gtk::MenuItem               *menuitem_preferences;
        Gtk::MenuItem               *menuitem_about;

        try
        {
            // GENERAL TOOLS
            Lifeobase::builder->get_widget( "menuitem_preferences", menuitem_preferences );
            Lifeobase::builder->get_widget( "menuitem_about", menuitem_about );

            // PANES
            Lifeobase::builder->get_widget( "hpaned_main", m_hpaned_main );
            Lifeobase::builder->get_widget( "hpaned_second", m_hpaned_entry );

            // TAG PANEL
            Lifeobase::builder->get_widget_derived( "treeview_all_tags", panel_extra );

            // ICONS
            Glib::RefPtr< Gtk::IconTheme > theme = Gtk::IconTheme::get_default();
            icons->diary_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/diary-16.png" );
            icons->diary_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/diary-32.png" );
            icons->entry_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/entry-16.png" );
            icons->entry_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/entry-32.png" );
            icons->entry_favorite_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/favorite-16.png" );
            icons->entry_favorite_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/favorite-32.png" );
            icons->tag_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/tag-16.png" );
            icons->tag_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/tag-32.png" );
            icons->tag_category_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/category_tags-16.png" );
            icons->tag_category_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/category_tags-32.png" );
            icons->chapter_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/chapter-16.png" );
            icons->chapter_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/chapter-32.png" );
            // TODO: update icons on theme change
            icons->theme_16 = theme->load_icon( "gtk-select-color", 16 );
            icons->theme_default_16 = theme->load_icon( "gtk-apply", 16 );
            icons->theme_32 = theme->load_icon( "gtk-select-color", 32 );
            icons->month_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/month-16.png" );
            icons->month_32 = Gdk::Pixbuf::create_from_file( ICONDIR "/month-32.png" );
            icons->filter_16 = Gdk::Pixbuf::create_from_file( ICONDIR "/filter-16.png" );

            // PANELS & VIEWS
            panel_main = new PanelMain;
            panel_diary = new PanelDiary;
            //panel_extra = new PanelExtra;
            m_diary_view = new DiaryView;
            m_theme_view = new ThemeView;
            m_category_tags_view = new CategoryTagsView;
            m_chapter_view = new ChapterView;
            view_entry = new EntryView;
            view_tag = new TagView;
        }
        catch( std::exception &ex )
        {
            throw ex;
        }

        // GEOMETRY
        m_hpaned_main->set_position( settings.position_pane );
        m_hpaned_entry->set_position( settings.position_pane_tags );

		// ACTIONS
        create_action( true, m_action_undo,"Undo", Gtk::Stock::UNDO,
                       _( "Undo" ), "",     // this tooltip will be dynamic
                       Gtk::AccelKey( GDK_KEY_z, Gdk::CONTROL_MASK ),
                       sigc::mem_fun( *this, &Lifeograph::handle_undo ) );

        create_action( true, m_action_redo, "Redo", Gtk::Stock::REDO,
                       _( "Redo" ), "",     // this tooltip will be dynamic
                       Gtk::AccelKey( GDK_KEY_z, Gdk::CONTROL_MASK | Gdk::SHIFT_MASK ),
                       sigc::mem_fun( *this, &Lifeograph::handle_redo ) );

		// SIGNALS
		menuitem_about->signal_activate().connect(
				sigc::mem_fun( this, &Lifeograph::show_about ) );
		menuitem_preferences->signal_activate().connect(
				sigc::ptr_fun( &DialogPreferences::create ) );
	}

	m_connection_backup = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( this, &Lifeograph::write_backup ),
			BACKUP_INTERVAL );

    m_actions_read->set_sensitive( true );
    m_actions_edit->set_visible( Diary::d->is_read_only() == false );
    m_actions_edit->set_sensitive( Diary::d->is_read_only() == false );

	add( *m_hpaned_main );

	// LOGIN
	m_internaloperation++;

	view_tag->handle_login();
	m_diary_view->handle_login();
	view_entry->handle_login();
	panel_diary->handle_login();
	panel_extra->populate();

	m_internaloperation--;

	loginstatus = LOGGED_IN;
    m_flag_auto_logout_frozen = false;
	update_title();
}
