/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_HELPERS_HEADER
#define LIFEOGRAPH_HELPERS_HEADER


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <fstream>
#include <set>

#include <gcrypt.h>

#include <gtkmm.h>

#include <libintl.h>

// DEFINITIONS FOR LIBGETTEXT
#define _(String)               gettext(String)
#define gettext_noop(String)    String
#define N_(String)              gettext_noop(String)
// END OF LIBGETTEXT DEFINITIONS


namespace HELPERS
{

class Error
{
	public:
						Error( const Glib::ustring& );
};

// RESPONSE IDS
static const int		RESPONSE_CANCEL		= 0;
static const int		RESPONSE_GO			= 1;

// DATE ============================================================================================
// order: 10 bits
// day:	   5 bits
// month:  4 bits
// year:  12 bits
// ordinal flag:  1 bit (32nd bit)

class Date
{
	public:
		const static unsigned int	YEAR_MAX         = 2199;
		const static unsigned int	YEAR_MIN         = 1900;
		const static unsigned long	ORDINAL_FLAG     = 0x80000000;
		const static unsigned long	NOTSET           = 0xFFFFFFFF;
		const static unsigned long	DATE_MAX         = 0xFFFFFFFF;
		const static unsigned long	ORDINAL_STEP     = 0x400;
		const static unsigned long	YEARMONTH_FILTER = 0x7FFF8000;

		typedef unsigned long date_t;

		void operator=( const Date &date )
		{ m_date = date.m_date; }
		void operator=( const Date::date_t &date )
		{ m_date = date; }

		bool operator>( const Date& date_r )
		{ return( m_date > date_r.m_date ); }
		bool operator>=( const Date& date_r )
		{ return( m_date >= date_r.m_date ); }
		bool operator<( const Date& date_r )
		{ return( m_date < date_r.m_date ); }

		bool operator==( const Date &date )
		{ return( m_date == date.m_date ); }
		bool operator==( const Date::date_t &date )
		{ return( m_date == date ); }

		bool operator!=( const Date &date )
		{ return( m_date != date.m_date ); }

		explicit					Date( date_t date )
			:	m_date( date ) {}
									Date( unsigned int y, unsigned int m, unsigned int d,
										  unsigned int o = 0 )
			:	m_date( ( y << 19 ) | ( m << 15 ) | ( d << 10 ) | o ) {}
		explicit					Date( const Glib::ustring& );
		// TODO: later: explicit                    Date( time_t );
		// ORDINAL C'TOR
									Date( unsigned int o1, unsigned int o2 )
			:	m_date( ORDINAL_FLAG | ( o1 << 10 ) | o2 ) {}

        static date_t               get_today( int order = 0 ) {
            time_t t = time( NULL );
            struct tm *ti = localtime( &t );
            return make_date( ti->tm_year + 1900, ti->tm_mon + 1, ti->tm_mday, order );
		}

		date_t						get_pure( void ) const
		{ return( m_date & 0xFFFFFC00 ); }
        static date_t               get_pure( const date_t d )
        { return( d & 0xFFFFFC00 ); }

		date_t						get_yearmonth( void ) const
		{ return( m_date & YEARMONTH_FILTER ); }
        static date_t               get_yearmonth( const date_t d )
        { return( d & YEARMONTH_FILTER ); }

		unsigned int				get_year( void ) const
		{ return ( ( m_date & 0x7FF80000 ) >> 19 ); }
		static unsigned int         get_year( const date_t d )
        { return ( ( d & 0x7FF80000 ) >> 19 ); }

		Glib::ustring				get_year_str( void ) const;
		Glib::ustring				get_month_str( void ) const;
		Glib::ustring				get_day_str( void ) const;
		Glib::ustring		        get_weekday_str( void ) const;

		unsigned int                get_days_in_month( void ) const;

		bool                        is_leap_year( void ) const
		{
		    int year( get_year() );
		    if( ( year % 400 ) == 0 )
		        return true;
		    else if( ( year % 100 ) == 0 )
		        return false;
		    else if( ( year % 4 ) == 0 )
		        return true;

		    return false;
		}

		unsigned int				get_month( void ) const
		{ return( ( m_date & 0x78000 ) >> 15 ); }
        static unsigned int         get_month( const date_t d )
        { return( ( d & 0x78000 ) >> 15 ); }

		Glib::Date::Month			get_month_glib( void ) const;

		unsigned int				get_day( void ) const
		{ return( ( m_date & 0x7C00 ) >> 10 ); }
        static unsigned int         get_day( const date_t d )
        { return( ( d & 0x7C00 ) >> 10 ); }

		unsigned int				get_order( void ) const
		{ return( m_date & 0x3FF ); }
        static unsigned int         get_order( const date_t d )
        { return( d & 0x3FF ); }

		unsigned int				get_ordinal_order( void ) const
		{ return( ( m_date & 0x7FFFFC00) >> 10 ); }
        static unsigned int         get_ordinal_order( const date_t d )
        { return( ( d & 0x7FFFFC00) >> 10 ); }

        static Date::date_t         get_order_begin( void ) {
            return ORDINAL_FLAG;
        }

		Glib::Date					get_glib( void ) const
		{ return Glib::Date( get_day(), get_month_glib(), get_year() ); }
		time_t						get_ctime( void ) const;
        static time_t               get_ctime( const date_t );

		void						set_year( unsigned int y )
		{
			m_date &= 0x8007FFFF;
			m_date |= ( y << 19 );
		}
		void						set_month( unsigned int m )
		{
			m_date &= 0xFFF87FFF;
			if( m < 13 )
			    m_date |= ( m << 15 );
		}
		void						set_day( unsigned int d )
		{
			m_date &= 0xFFFF83FF;
			if( d < 32 )
			    m_date |= ( d << 10 );
		}
		void						reset_order_0( void )
		{ m_date &= 0xFFFFFC00; }
		void						reset_order_1( void )
		{ m_date |= 0x1; m_date &= 0xFFFFFC01; }
		static void					reset_order_1( date_t &date )
		{ date |= 0x1; date &= 0xFFFFFC01; }

		Glib::ustring				format_string( void ) const;
		static Glib::ustring        format_string( const date_t );
		static Glib::ustring		format_string( const time_t );
		static Glib::ustring		format_string_do( const time_t );

		static date_t               parse_string( const Glib::ustring& );

		bool						is_valid( void ) const
		{
			return( get_day() > 0 && get_day() <= get_days_in_month() );
		}

		bool						is_ordinal( void ) const
		{ return( m_date & ORDINAL_FLAG ); }
        static bool                 is_ordinal( const date_t d )
        { return( d & ORDINAL_FLAG ); }

		void						forward_months( int months );
		void						forward_month( void );
		void						forward_day( void );

		void						backward_ordinal_order( void )
		{ if( get_ordinal_order() > 0 ) m_date -= ORDINAL_STEP; }
        static void                 backward_ordinal_order( Date::date_t &d )
        { if( get_ordinal_order( d ) > 0 ) d -= ORDINAL_STEP; }
		void						forward_ordinal_order( void )
		{ m_date += ORDINAL_STEP; }
        static void                 forward_ordinal_order( Date::date_t &d )
        { d += ORDINAL_STEP; }

		static date_t				make_year( unsigned int y )
		{ return( y << 19 ); }
		static date_t				make_month( unsigned int m )
		{ return( m << 15 ); }
		static date_t				make_day( unsigned int m )
		{ return( m << 10 ); }
		static date_t               make_date( unsigned int y, unsigned int m, unsigned int d,
                                               unsigned int o = 0 )
		{ return( ( y << 19 ) | ( m << 15 ) | ( d << 10 ) | o ); }

        unsigned int                calculate_days_between( const Date& ) const;

	//protected:
		date_t						m_date;
};

/*
inline  int
subtract_months( Date d1, Date d2 )
{
	return ( ( isolate_month( d1 ) - isolate_month( d2 ) ) +
			 ( ( isolate_year( d1 ) - isolate_year( d2 ) ) * 12 ) );
}
*/

inline bool
compare_dates( const Date::date_t &date_l, const Date::date_t &date_r )
{
	return( date_l > date_r );
}

typedef bool( *FuncCompareDates )( const Date::date_t&, const Date::date_t& ) ;

// GLOBAL FUNCTIONS
void				print_error( const Glib::ustring& );
void				print_info( const Glib::ustring& );

#ifdef LIFEOGRAPH_DEBUG_BUILD
#define PRINT_DEBUG( message )	std::cout << "DEBUG: " << message << std::endl
#else
#define PRINT_DEBUG( message )	;
#endif

long				convert_string( const std::string& );
bool                str_ends_with ( const std::string&, std::string const& );

Gdk::RGBA			contrast( const Gdk::RGBA&, const Gdk::RGBA& );
Gdk::RGBA			midtone( const Gdk::RGBA&, const Gdk::RGBA& );
Gdk::RGBA			midtone( const Gdk::RGBA&, const Gdk::RGBA&, float );

inline Glib::ustring
convert_gdkcolor_to_html( const Gdk::RGBA &gdkcolor )
{
	// this function's source of inspiration is Geany
	char buffer[ 8 ];

	g_snprintf(	buffer, 8, "#%02X%02X%02X",
			gdkcolor.get_red_u() >> 8,
			gdkcolor.get_green_u() >> 8,
			gdkcolor.get_blue_u() >> 8 );
	return buffer;
}

inline Glib::ustring
convert_gdkrgba_to_string( const Gdk::RGBA &gdkcolor )
{
    char buffer[ 14 ];
    g_snprintf( buffer, 14, "#%04X%04X%04X",
            int( gdkcolor.get_red() * 0xFFFF ),
            int( gdkcolor.get_green() * 0xFFFF ),
            int( gdkcolor.get_blue() * 0xFFFF ) );
    return buffer;
}
std::ios::pos_type	get_file_size( std::ifstream& );

void                open_url( Gtk::AboutDialog&, const Glib::ustring& );
void                mail_to( Gtk::AboutDialog&, const Glib::ustring& );

std::string         get_env_lang( void );

Gtk::MenuItem*		create_menuitem_markup( const Glib::ustring&,
											const Glib::SignalProxy0< void >::SlotType& );

// COMMON SIGNALS
typedef sigc::signal< void >		SignalVoid;
typedef std::set< std::string >	    ListPaths;

// MEMORY POOL
template< class T >
class Pool : public std::list< T* >
{
	public:
		virtual					~Pool()
		{ clear(); }

		void					clear( void )
		{
			for(	typename std::list< T* >::const_iterator iter = this->begin();
					iter != this->end();
					++iter )
			{
				delete *iter;
			}

			std::list< T* >::clear();
		}
};


class Cipher
{
	public:
		static const int	cCIPHER_ALGORITHM	= GCRY_CIPHER_AES256;
		static const int	cCIPHER_MODE		= GCRY_CIPHER_MODE_CFB;
		static const int	cIV_SIZE			= 16; // = 128 bits
		static const int	cSALT_SIZE			= 16; // = 128 bits
		static const int	cKEY_SIZE			= 32; // = 256 bits
		static const int	cHASH_ALGORITHM		= GCRY_MD_SHA256;

		static bool			init( void );

		static void			create_iv(	unsigned char** );
		static void			expand_key(	char const*,
										const unsigned char*,
										unsigned char** );
		static void			create_new_key(	char const*,
											unsigned char**,
											unsigned char** );
		static void			encrypt_buffer(	unsigned char*,
											size_t&,
											const unsigned char*,
											const unsigned char* );
		static void			decrypt_buffer(	unsigned char*,
											size_t&,
											const unsigned char*,
											const unsigned char * );

	protected:

	private:

};

// ENTRY WIDGET WITH SELF-HANDLED CLEAR ICON =======================================================
class EntryClear : public Gtk::Entry
{
	public:
                            EntryClear( BaseObjectType*, const Glib::RefPtr<Gtk::Builder>& );

	protected:
		void				handle_clear_icon( Gtk::EntryIconPosition, const GdkEventButton* );
		virtual void		on_changed( void );
		virtual bool		on_key_release_event( GdkEventKey* );
};

// MENU 2 ==========================================================================================
class Menu2 : public Gtk::Menu
{
	public:
		Menu2() { }
		Menu2( BaseObjectType* cobject, const Glib::RefPtr<Gtk::Builder>& )
		:	Gtk::Menu( cobject ) { }

	protected:

	private:

	friend class Menubutton;
};

class Menubutton : public Gtk::ToggleButton
{
	public:
							Menubutton(	const Gtk::StockID&,
										const Glib::ustring&,
										Gtk::ReliefStyle = Gtk::RELIEF_NONE,
										Gtk::IconSize = Gtk::ICON_SIZE_BUTTON,
										Menu2* = NULL );
							Menubutton(	const Glib::RefPtr< Gdk::Pixbuf >*,
										const Glib::ustring&,
										Gtk::ReliefStyle = Gtk::RELIEF_NONE,
										Menu2* = NULL );
							Menubutton( BaseObjectType*,
                                        const Glib::RefPtr< Gtk::Builder >& );
		virtual				~Menubutton( void );

        Menu2*              get_menu( void ) const;
        void                set_menu( Menu2* );
		bool				clear_menu( void );
		void				release( void );

		void				set_label( const Glib::ustring& );
		void				get_menu_position( int&, int&, bool& );

        static Glib::ustring
                            s_builder_name;

	protected:
		bool				on_button_press_event( GdkEventButton* );

		Menu2				*m_menu;
		Gtk::Label			*m_label;
        sigc::connection    m_connection;

	private:
};

/*
 * TO BE REWRITTEN IF NEEDED IN THE FUTURE...
class Menutoolbutton : public Gtk::ToolItem
{
	public:
		Menutoolbutton( const Gtk::StockID&, const std::string&,
						Menu2* = NULL, bool = false );

		Menu2*			get_menu( void );

	protected:
		void			on_toolbar_reconfigured( void );

		bool			m_flag_important;
		Menubutton		m_button;
};*/

// FILE FILTERS
//class FilefilterAny : public Gtk::FileFilter
//{
//	public:
//		FilefilterAny()
//		{
//			set_name( _("All Files") );
//			add_pattern( "*" );
//		}
//};
//
//class FilefilterDiary : public Gtk::FileFilter
//{
//	public:
//		FilefilterDiary()
//		{
//			set_name( _("Diary Files (*.diary)") );
//			add_mime_type( "application/x-lifeographdiary" );
//		}
//};

class MenuitemRecent : public Gtk::MenuItem
{
	public:
		typedef sigc::signal< void, const std::string& >
									Signal_void_string;

									MenuitemRecent( const std::string& );

		Signal_void_string			signal_removerecent( void )
		{ return m_signal_removerecent; }

	protected:
		virtual bool				on_motion_notify_event( GdkEventMotion* );
		virtual bool				on_leave_notify_event( GdkEventCrossing* );
		virtual bool				on_button_release_event( GdkEventButton* );
		virtual bool				on_button_press_event( GdkEventButton* );

		Gtk::Image					m_icon_remove;
		std::string					m_path;
		bool						m_flag_deletepressed;

		Signal_void_string			m_signal_removerecent;
};

#if( GTKMM_MAJOR_VERSION < 3 )

class FilebuttonRecent : public Menubutton
{
	public:
        const unsigned int  MAX_RECENT_FILE_COUNT( 8 );

        typedef std::list< std::string > ListPaths;
//                                    FilebuttonRecent( const Glib::RefPtr< Gdk::Pixbuf >&,
//                                                      ListPaths*, const Glib::ustring& );
                                    FilebuttonRecent( BaseObjectType*,
                                                      const Glib::RefPtr< Gtk::Builder >& );

        void                        set( ListPaths* );

		ListPaths*	                get_recentlist( void )
		{
			return m_list_recent;
		}
		std::string					get_filename( void ) const;
		std::string					get_filename_recent( int ) const;
		void						set_filename( const std::string& );
		void						update_filenames( void );

		bool						add_recent( const std::string& );
		void						remove_recent( const std::string& );

		void						show_filechooser( void );

		SignalVoid					signal_selection_changed( void );
		SignalVoid					signal_create_file( void );

		bool						handle_motion_notify( GdkEventMotion* );

		static Glib::ustring        fallback_label;

	protected:
		virtual void				on_create_file( void );
		virtual void				on_size_allocate( Gtk::Allocation& );
		virtual void				on_drag_data_received(
											const Glib::RefPtr<Gdk::DragContext>&,
											int, int,
											const Gtk::SelectionData&, guint, guint );

	private:
		Gtk::FileChooserDialog		*m_filechooserdialog;
		Gtk::Image					m_icon_new;
		Gtk::Image					m_icon_browse;
		ListPaths					*m_list_recent;
		SignalVoid					m_signal_selectionchanged;
		SignalVoid					m_signal_createfile;
};
#endif

// DIALOGEVENT =====================================================================================
class DialogEvent : public Gtk::Dialog
{
    public:
                            DialogEvent( const Glib::ustring& );
                            DialogEvent( BaseObjectType*,
                                         const Glib::RefPtr< Gtk::Builder >& );

    protected:
        bool                on_event( GdkEvent* );
        void                handle_logout( void );

};

// COMBOBOXTEXT WITH GTK BUILDER (FOR GTKMM 2) =====================================================
#if( GTKMM_MAJOR_VERSION < 3 )
class ComboboxEasy : public Gtk::ComboBoxText
{
    public:
                            ComboboxEasy( BaseObjectType* cobject,
                                          const Glib::RefPtr< Gtk::Builder >& )
    :   Gtk::ComboBoxText( cobject ) {}
};
#endif

// FRAME FOR PRINTING ==============================================================================
Gtk::Frame* create_frame( const Glib::ustring&, Gtk::Widget& );

} // end of namespace HELPERS

#endif

