/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <gtkmm/table.h>
#include <gtkmm/label.h>
#include <gtkmm/stock.h>

#include "lifeobase.hpp"
#include "helpers.hpp"
#include "diary.hpp"	// for LIFEO::PASSPHRASE_MIN_SIZE
#include "dialog_password.hpp"


using namespace LIFEO;


// PASSWORD ADD/CHANGE DIALOG ======================================================================
DialogPassword::DialogPassword(	const std::string &passphrase_old,
								const Glib::ustring &title )
:   DialogEvent( title ), m_passphrase_old( passphrase_old ), m_passphrase_is_set( false )
{
	Gtk::Table *table = Gtk::manage( new Gtk::Table( 3, 2 ) );
	table->set_spacings( 5 );
	table->set_border_width( 5 );

	Gtk::Label *label_current = Gtk::manage(
			new Gtk::Label( _( "Current Password:" ), Gtk::ALIGN_START ) );
	m_entry_current = Gtk::manage( new Gtk::Entry );
	m_entry_current->set_visibility( false );
	m_button_authenticate = Gtk::manage( new Gtk::Button );
	Gtk::Image *icon_apply = Gtk::manage(
			new Gtk::Image( Gtk::Stock::APPLY, Gtk::ICON_SIZE_MENU ) );
	m_button_authenticate->set_image( * icon_apply );
	m_button_authenticate->set_tooltip_text( _( "Authenticate" ) );

	add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );

	// if already encrypted:
	if ( passphrase_old.size() > 0 )
	{
		table->attach( *label_current, 0, 1, 0, 1, Gtk::SHRINK | Gtk::FILL );
		table->attach( *m_entry_current, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL );
		table->attach( *m_button_authenticate, 2, 3, 0, 1, Gtk::SHRINK );
        m_button_change = add_button( _( "C_hange Password" ), Gtk::RESPONSE_OK );
	}
	else
		m_button_change = add_button( _( "_Add Password" ), Gtk::RESPONSE_OK );

	Gtk::Label *label_new = Gtk::manage(
			new Gtk::Label( _( "New Password:" ), Gtk::ALIGN_START ) );
	m_entry_new = Gtk::manage( new Gtk::Entry );
	m_entry_new->set_visibility( false );
	m_entry_new->set_sensitive( passphrase_old.size() <= 0 );
	table->attach( *label_new, 0, 1, 1, 2, Gtk::SHRINK | Gtk::FILL );
	table->attach( *m_entry_new, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL );

	Gtk::Label *label_confirm = Gtk::manage(
			new Gtk::Label( _( "Confirm New Password:" ), Gtk::ALIGN_START ) );
	m_entry_confirm = Gtk::manage( new Gtk::Entry );
	m_entry_confirm->set_visibility( false );
	m_entry_confirm->set_sensitive( passphrase_old.size() <= 0 );
	table->attach( *label_confirm, 0, 1, 2, 3, Gtk::SHRINK | Gtk::FILL );
	table->attach( *m_entry_confirm, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL );

	m_image_match = Gtk::manage(
			new Gtk::Image( Gtk::Stock::NO, Gtk::ICON_SIZE_MENU ) );
	table->attach( *m_image_match, 2, 3, 1, 3, Gtk::SHRINK );

	get_vbox()->pack_start( *table );
	get_vbox()->set_spacing( 5 );

	Gtk::Image *icon_ok = Gtk::manage(
			new Gtk::Image( Gtk::Stock::OK, Gtk::ICON_SIZE_MENU ) );
	m_button_change->set_image( *icon_ok );
	m_button_change->set_sensitive( false );

	m_button_authenticate->signal_clicked().connect(
			sigc::mem_fun( this, &DialogPassword::authenticate ) );
	m_entry_new->signal_changed().connect(
			sigc::mem_fun( this, &DialogPassword::check_match ) );
	m_entry_confirm->signal_changed().connect(
			sigc::mem_fun( this, &DialogPassword::check_match ) );

	show_all();
}

void
DialogPassword::authenticate()
{
	if ( m_entry_current->get_text() == m_passphrase_old )
	{
		m_entry_new->set_sensitive( true );
		m_entry_confirm->set_sensitive( true );
		m_entry_new->grab_focus();
		m_entry_current->set_sensitive( false );
		m_button_authenticate->set_sensitive( false );
	}
}

void
DialogPassword::check_match()
{
	if (	m_entry_confirm->get_text().size() >= LIFEO::PASSPHRASE_MIN_SIZE &&
			m_entry_new->get_text() == m_entry_confirm->get_text() )
	{
		m_image_match->set( Gtk::Stock::YES, Gtk::ICON_SIZE_MENU );
		m_button_change->set_sensitive( true );
	}
	else
	{
		m_image_match->set( Gtk::Stock::NO, Gtk::ICON_SIZE_MENU );
		m_button_change->set_sensitive( false );
	}
}

void
DialogPassword::on_response( int response )
{
	switch( response )
	{
		case Gtk::RESPONSE_OK:
			m_passphrase_new = m_entry_confirm->get_text();
			m_passphrase_is_set = true;
			break;
		default:
			break;
	}
}

// DIALOG PASSWORD PROMPT ==========================================================================
DialogPasswordPrompt::DialogPasswordPrompt( BaseObjectType* cobject,
                                            const Glib::RefPtr< Gtk::Builder > &refbuilder )
:   DialogEvent( cobject, refbuilder )
{
    Lifeobase::builder->get_widget_derived( "entry_dlg_password", m_entry );
    Lifeobase::builder->get_widget( "label_dlg_password_msg", m_label_msg );
    Lifeobase::builder->get_widget( "label_dlg_password_path", m_label_path );

    set_response_sensitive( RESPONSE_GO, false );

    // SIGNALS
    m_entry->signal_changed().connect(
            sigc::mem_fun( this,
                    &DialogPasswordPrompt::handle_entry_changed ) );
}

int
DialogPasswordPrompt::run( void )
{
    m_label_msg->set_text( _( "Please enter password for" ) );
    return DialogEvent::run();
}
int
DialogPasswordPrompt::run_again( void )
{
    m_label_msg->set_text( _( "Wrong password. Please retry..." ) );
    return DialogEvent::run();
}

void
DialogPasswordPrompt::set_message( const Glib::ustring &msg )
{
    m_label_msg->set_text( msg );
}

void
DialogPasswordPrompt::set_path( const Glib::ustring &path )
{
    m_label_path->set_text( path );
}

void
DialogPasswordPrompt::on_hide( void )
{
    m_entry->set_text( "" );
    m_entry->grab_focus();    // closing with cancel steals focus
    Gtk::Dialog::on_hide();
}

void
DialogPasswordPrompt::handle_entry_changed( void )
{
    set_response_sensitive( RESPONSE_GO, m_entry->get_text().size() >= PASSPHRASE_MIN_SIZE );
}
