/***************************************************************************
 *   Copyright (C) 2007 by Miguel Chavez Gamboa                            *
 *   miguel.chavez.gamboa@gmail.com                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/
#include <KLocale>
#include <KMessageBox>
#include <KFileDialog>

#include <QByteArray>
#include <QRegExpValidator>

#include "purchaseeditor.h"


PurchaseEditorUI::PurchaseEditorUI( QWidget *parent )
: QFrame( parent )
{
    setupUi( this );
}

PurchaseEditor::PurchaseEditor( QWidget *parent )
: KDialog( parent )
{
    ui = new PurchaseEditorUI( this );
    setMainWidget( ui );
    setCaption( i18n("Purchase") );
    setButtons( KDialog::Ok|KDialog::Cancel );

    //Set Validators for input boxes
    QRegExp regexpC("[0-9]{1,13}"); //(EAN-13 y EAN-8) .. y productos sin codigo de barras?
    QRegExpValidator * validatorEAN13 = new QRegExpValidator(regexpC, this);
    ui->editCode->setValidator(validatorEAN13);
    ui->editTax->setValidator(new QDoubleValidator(0.00, 999999999999.99, 3,ui->editTax));
    ui->editExtraTaxes->setValidator(new QDoubleValidator(0.00, 999999999999.99, 3,ui->editExtraTaxes));
    ui->editCost->setValidator(new QDoubleValidator(0.00, 999999999999.99, 3, ui->editCost));
    ui->editPoints->setValidator(new QIntValidator(0,999999999, ui->editPoints));
    ui->editFinalPrice->setValidator(new QDoubleValidator(0.00,999999999999.99, 3, ui->editFinalPrice));
    ui->editItemsPerBox->setValidator(new QDoubleValidator(0.00,999999999999.99, 2, ui->editItemsPerBox));
    ui->editPricePerBox->setValidator(new QDoubleValidator(0.00,999999999999.99, 2, ui->editPricePerBox));
    ui->editQty->setValidator(new QDoubleValidator(0.00,999999999999.99, 2, ui->editQty));

    connect( ui->btnPhoto          , SIGNAL( clicked() ), this, SLOT( changePhoto() ) );
    connect( ui->btnCalculatePrice , SIGNAL( clicked() ), this, SLOT( calculatePrice() ) );
    connect( ui->editItemsPerBox , SIGNAL( textEdited(const QString &) ), this, SLOT( calculatePrice() ) );
    connect( ui->editPricePerBox , SIGNAL( textEdited(const QString &) ), this, SLOT( calculatePrice() ) );
//     connect( ui->editCode, SIGNAL(editingFinished()), SLOT(checkIfCodeExists()));
    connect( ui->editCode, SIGNAL(textEdited(const QString &)), SLOT(checkIfCodeExists()));
    connect( ui->btnAddItem, SIGNAL( clicked() ), this, SLOT( addItemToList() ) );

    status = estatusNormal;
    productExists = false;
    productsHash.clear();
    resetEdits();
    totalBuy = 0.0;
    itemCount = 0.0;
    QTimer::singleShot(500, this, SLOT(setupTable()));
}

PurchaseEditor::~PurchaseEditor()
{
    delete ui;
}

void PurchaseEditor::setDb(QSqlDatabase database)
{
  db = database;
  if (!db.isOpen()) db.open();
  populateCategoriesCombo();
  populateMeasuresCombo();
}

void PurchaseEditor::populateCategoriesCombo()
{
  QSqlQuery query(db);
  if (query.exec("select catid,text from categories;")) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      QString text = query.value(fieldText).toString();
      ui->categoriesCombo->addItem(text);
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
}

void PurchaseEditor::populateMeasuresCombo()
{
  QSqlQuery query(db);
  if (query.exec("select id,text from measures;")) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      QString text = query.value(fieldText).toString();
      ui->measuresCombo->addItem(text);
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
}

int PurchaseEditor::getCategoryId()
{
  QSqlQuery query(db);
  int code=-1;
  QString currentText = ui->categoriesCombo->currentText();
  if (query.exec("select catid,text from categories;")) {
    while (query.next() && (code == -1)) {
      int fieldId   = query.record().indexOf("catid");
      int fieldText = query.record().indexOf("text");
      int id = query.value(fieldId).toInt();
      QString text = query.value(fieldText).toString();
      if (text == currentText) {
        code = id;
      }
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
  return code;
}


int PurchaseEditor::getMeasureId()
{
  QSqlQuery query(db);
  int code=-1;
  QString currentText = ui->measuresCombo->currentText();
  if (query.exec("select id,text from measures;")) {
    while (query.next() && (code == -1)) {
      int fieldId   = query.record().indexOf("id");
      int fieldText = query.record().indexOf("text");
      int id = query.value(fieldId).toInt();
      QString text = query.value(fieldText).toString();
      if (text == currentText) {
        code = id;
      }
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
  return code;
}


void PurchaseEditor::setCategory(QString str)
{
 int idx = ui->categoriesCombo->findText(str,Qt::MatchCaseSensitive);
 if (idx > -1) ui->categoriesCombo->setCurrentIndex(idx);
 else { 
  qDebug()<<"Str not found:"<<str;
  }
}

void PurchaseEditor::setCategory(int i)
{
 QString text = getCategoryStr(i);
 setCategory(text);
}

void PurchaseEditor::setMeasure(int i)
{
 QString text = getMeasureStr(i);
 setMeasure(text);
}

void PurchaseEditor::setMeasure(QString str)
{
int idx = ui->measuresCombo->findText(str,Qt::MatchCaseSensitive);
 if (idx > -1) ui->measuresCombo->setCurrentIndex(idx);
 else { 
  qDebug()<<"Str not found:"<<str;
  }
}

QString PurchaseEditor::getCategoryStr(int c)
{
 QString str;
 QSqlQuery query(db);
 QString qstr = QString("select text from categories where catid=%1;").arg(c);
 if (query.exec(qstr)) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      str = query.value(fieldText).toString();
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
 return str;
}

QString PurchaseEditor::getMeasureStr(int c)
{
 QString str;
 QSqlQuery query(db);
 QString qstr = QString("select text from measures where id=%1;").arg(c);
 if (query.exec(qstr)) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      str = query.value(fieldText).toString();
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
 return str;
}

void PurchaseEditor::changePhoto()
{
 QString fname = KFileDialog::getOpenFileName();
  if (!fname.isEmpty()) {
    QPixmap p = QPixmap(fname);
    setPhoto(p);
  }
}

void PurchaseEditor::calculatePrice()
{
 double finalPrice=0.0;
 if (ui->editCost->text().isEmpty() && !ui->groupBoxedItem->isChecked() ) {
   ui->editCost->setFocus();
 }
 else if (ui->editUtility->text().isEmpty()) {
   ui->editUtility->setFocus();
 }
 else if (ui->editTax->text().isEmpty()) {
   ui->editTax->setText("0.0");
   ui->editTax->setFocus();
   ui->editTax->selectAll();
 }
 else {
  if (ui->editExtraTaxes->text().isEmpty()) {
   ui->editExtraTaxes->setText("0.0");
   ui->editExtraTaxes->setFocus();
   ui->editExtraTaxes->selectAll();
  }
  //TODO: if TAXes are included in cost...
  double cost    = ui->editCost->text().toDouble();
  double utility = ui->editUtility->text().toDouble();
  double tax     = ui->editTax->text().toDouble();
  double tax2    = ui->editExtraTaxes->text().toDouble(); 
  //Utility is calculated before taxes... Taxes include utility... is it ok?
  utility = ((utility/100)*cost);
  double cu=cost+utility;
  tax     = ((tax/100)*(cu));
  tax2    = ((tax2/100)*(cu));

  if (ui->groupBoxedItem->isChecked()){
    double itemsPerBox = 0;
    double pricePerBox = 0;
    if (!ui->editItemsPerBox->text().isEmpty()) itemsPerBox = ui->editItemsPerBox->text().toDouble();
    if (!ui->editPricePerBox->text().isEmpty()) pricePerBox = ui->editPricePerBox->text().toDouble();
    if (itemsPerBox>0 || pricePerBox>0) cost = pricePerBox/itemsPerBox;
    ui->editCost->setText(QString::number(cost));
    utility = ((ui->editUtility->text().toDouble()/100)*cost);
    cu = cost + utility;
    tax     = ((ui->editTax->text().toDouble()/100)*(cu));
    tax2    = ((ui->editExtraTaxes->text().toDouble()/100)*(cu));
    finalPrice = cu + tax + tax2;
  }
  else finalPrice = cost + utility + tax + tax2;
 
  ui->editFinalPrice->setText(QString::number(finalPrice));
  ui->editFinalPrice->selectAll();
  }
}


void PurchaseEditor::checkIfCodeExists()
{
  productExists = false;
  QString codeStr = ui->editCode->text();
  QSqlQuery query(db);
  if (query.exec(QString("select * from products where code=%1;").arg(codeStr) )) {
    if (query.size() == 1 ) {
      status = estatusMod;
      productExists = true;
      while (query.next()) {
        int fieldName = query.record().indexOf("name");
        int fieldPrice= query.record().indexOf("price");
        int fieldStockQty = query.record().indexOf("stockqty");
        int fieldCost     = query.record().indexOf("cost");
        int fieldUnits    = query.record().indexOf("units");
        int fieldSoldUnits= query.record().indexOf("soldunits");
        int fieldDateLastS= query.record().indexOf("datelastsold");
        int fieldTaxp     = query.record().indexOf("taxpercentage");
        int fieldEtraTax  = query.record().indexOf("extrataxes");
        int fieldCategory = query.record().indexOf("category");
        int fieldPhoto    = query.record().indexOf("photo");
        int fieldPoints   = query.record().indexOf("points");

        double   price    = query.value(fieldPrice).toDouble();
                 qtyOnDb  = query.value(fieldStockQty).toDouble();
        QString  name     = query.value(fieldName).toString();
        double   cost     = query.value(fieldCost).toDouble();
        int      units    = query.value(fieldUnits).toInt();
        double   tax1     = query.value(fieldTaxp).toDouble();
        double   tax2     = query.value(fieldEtraTax).toDouble();
        int      category = query.value(fieldCategory).toInt();
        QByteArray photoBA= query.value(fieldPhoto).toByteArray();
        qulonglong points = query.value(fieldPoints).toULongLong();
        QString  lastSold = query.value(fieldDateLastS).toString();
        //double   soldUnits= query.value(fieldSoldUnits).toDouble();

        //Prepopulate dialog...
        ui->editDesc->setText(name);
        setCategory(category);
        setMeasure(units);
        ui->editCost->setText(QString::number(cost));
        ui->editTax->setText(QString::number(tax1));
        ui->editExtraTaxes->setText(QString::number(tax2));
        ui->editFinalPrice->setText(QString::number(price));
        ui->editPoints->setText(QString::number(points));
        if (!photoBA.isEmpty()) {
          QPixmap photo;
          photo.loadFromData(photoBA);
          setPhoto(photo);
        }
      }
    } else {
      qDebug()<< "no product found with code "<<codeStr<<" .query.size()=="<<query.size();
      qulonglong codeSaved = getCode();
      resetEdits();
      setCode(codeSaved);
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
}


void PurchaseEditor::slotButtonClicked(int button)
{
  if (button == KDialog::Ok) {
    if (status == estatusNormal) QDialog::accept();
    else {
      qDebug()<< "Button = OK, status == statusMOD";
      done(estatusMod); 
    }
  }
  else QDialog::reject();
}


void PurchaseEditor::setupTable() {
  QSize tableSize = ui->tableView->size();
  ui->tableView->horizontalHeader()->resizeSection(0, (tableSize.width()/7));
  ui->tableView->horizontalHeader()->resizeSection(1, (tableSize.width()/7)*3);
  ui->tableView->horizontalHeader()->resizeSection(2,(tableSize.width()/7)*1.4);
  ui->tableView->horizontalHeader()->resizeSection(3,(tableSize.width()/7)*1.4);
}


void PurchaseEditor::addItemToList()
{
  bool ok=false;
  if (ui->editCode->text().isEmpty()) ui->editCode->setFocus();
  else if (ui->editDesc->text().isEmpty()) ui->editDesc->setFocus();
  else if (ui->editPoints->text().isEmpty()) ui->editPoints->setFocus();
  else if (ui->editCost->text().isEmpty()) ui->editCost->setFocus();
  else if (ui->editTax->text().isEmpty()) ui->editTax->setFocus();
  else if (ui->editFinalPrice->text().isEmpty()) ui->editFinalPrice->setFocus();
  else if (ui->editQty->text().isEmpty() || ui->editQty->text()=="0") ui->editQty->setFocus();
  else if (ui->editUtility->text().isEmpty()) ui->editUtility->setFocus();
  else if (ui->groupBoxedItem->isChecked() && (ui->editItemsPerBox->text().isEmpty() || ui->editItemsPerBox->text()=="0"))  ui->editItemsPerBox->setFocus();
  else if (ui->groupBoxedItem->isChecked() && (ui->editPricePerBox->text().isEmpty() || ui->editPricePerBox->text()=="0")) ui->editPricePerBox->setFocus();
  else ok = true;

  if (ok) {
    ProductInfo info;
    info.code    = getCode();
    info.desc    = getDescription();
    info.price   = getPrice();
    info.cost    = getCost();
    info.tax     = getTax1();
    info.extratax= getTax2();
    info.photo   = getPhotoBA();
    info.units   = getMeasureId();
    info.category= getCategoryId();
    info.utility = getProfit();
    info.points  = getPoints();
    info.stockqty= getQtyOnDb();
    info.purchaseQty = getPurchaseQty();
    double finalCount = info.purchaseQty + info.stockqty;
    info.validDiscount = productExists; //used to check if product is already on db.

    if (!productsHash.contains(info.code) && info.code>0) {
      //insert item to productsHash
      productsHash.insert(info.code, info);
      //insert item to ListView
      int rowCount = ui->tableView->rowCount();
      ui->tableView->insertRow(rowCount);
      ui->tableView->setItem(rowCount, 0, new QTableWidgetItem(QString::number(info.code)));
      ui->tableView->setItem(rowCount, 1, new QTableWidgetItem(info.desc));
      ui->tableView->setItem(rowCount, 2, new QTableWidgetItem(QString::number(info.purchaseQty)));
      ui->tableView->setItem(rowCount, 3, new QTableWidgetItem(QString::number(finalCount)));

      ui->tableView->horizontalHeader()->setResizeMode(QHeaderView::Interactive);
      ui->tableView->resizeRowsToContents();
      resetEdits();
      totalBuy = totalBuy + info.cost*info.purchaseQty;
      itemCount = itemCount + info.purchaseQty;
      qDebug()<<"totalBuy until now:"<<totalBuy<<" itemCount:"<<itemCount<<"info.cost:"<<info.cost<<"info.purchaseQty:"<<info.purchaseQty;
    }  else  qDebug()<<"Item "<<info.code<<" already on hash";
    ui->editCode->setFocus();
  }
}

void PurchaseEditor::resetEdits()
{
  ui->editCode->setText("");
  ui->editDesc->setText("");
  ui->editCost->setText("");
  ui->editTax->setText("");
  ui->editExtraTaxes->setText("0.0");
  ui->editFinalPrice->setText("");
  ui->editUtility->setText("");
  qtyOnDb = 0;
  pix = QPixmap(0,0); //null pixmap.
  ui->labelPhoto->setText(i18n("No Photo"));
  ui->editPoints->setText("0");
  ui->editItemsPerBox->setText("");
  ui->editPricePerBox->setText("");
  ui->editQty->setText("");
}

double PurchaseEditor::getPurchaseQty()
{
  if (ui->groupBoxedItem->isChecked()) {
    if ( ui->editItemsPerBox->text().isEmpty() ) return ui->editQty->text().toDouble();
    else return (ui->editQty->text().toDouble())*(ui->editItemsPerBox->text().toDouble());
  }
  else return ui->editQty->text().toDouble();
}

#include "purchaseeditor.moc"
