package latexDraw.figures;


import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Shape;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.LinkedList;
import java.util.Vector;

import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Draw extends Figure
{
	private static final long serialVersionUID = 1L;
	
	/** All the figures which constitute the drawing. */
	protected Vector<Figure> figures;
	
	/** Defines the borders of the draw without taking account of the thickness of the figures. */
	protected LaTeXDrawRectangle minBorders;

	/** True: the figures of the drawing must be drawn (and the method get code get there code. */
	protected boolean drawFigures;
	
	/** The point is useful to rotate the selection. */
	protected transient LaTeXDrawPoint2D formerGravityC = null;
	
	/** The value by default of the attribute drawFigure */
	public static final boolean DEFAULT_DRAWFIGURE = false;
	
	
	
	
	/**
	 * The constructor by default.
	 */
	public Draw(boolean drawFigures, boolean increaseMeter)
	{
		this(null, drawFigures, increaseMeter);
	}
	
	
	
	public Draw(Vector<Figure> figs, boolean drawFigures, boolean increaseMeter, boolean cloneFigures)
	{
		super(increaseMeter);
		
		canHaveShadow = true;
		isResizable = true;
		isBordersMovable = true;
		isDashableOrDotable  = true;
		isDoubleBoundaryable = true;
		this.drawFigures = drawFigures;
		canBeFilled     = true;
		isCustomizable  = true;
		isThicknessable = true;
		
		initializeBorders();
		figures = new Vector<Figure>();
		
		if(figs!=null)
			if(cloneFigures)
			{
				for(Figure f : figs)
					if(f!=null)
						try { addFigure((Figure)f.clone()); }
						catch(CloneNotSupportedException e) { e.printStackTrace(); }
			}
			else
				for(Figure f : figs)
					if(f!=null)
						addFigure(f);
		
		shape = createShape2D();
	}
	
	
	
	public Draw(Vector<Figure> figs, boolean drawFigures, boolean increaseMeter)
	{
		this(figs, drawFigures, increaseMeter, false);
	}
	
	
	

	/**
	 * Initialise the minBorders and the borders (colour, style, ...).
	 */
	protected void initializeBorders()
	{	
		if(minBorders==null)
			minBorders = new LaTeXDrawRectangle(false);
		
		if(borders==null)
			borders = new LaTeXDrawRectangle(false);
		
		borders.setLinesColor(Color.gray);
		borders.setLineStyle(PSTricksConstants.LINE_DASHED_STYLE);
	}
	
	
	
	
	
	/**
	 * Allows to add a figure to the drawing.
	 * @param f The figure to add.
	 * @throws IllegalArgumentException If f is null.
	 */
	public void addFigure(Figure f)
	{
		if(f==null)
			throw new IllegalArgumentException("f is null.");//$NON-NLS-1$
		
		f.onRelease();
		figures.add(f);
		updateBorders();
		updateGravityCenter();
	}
	
	
	
	
	/**
	 * Allows to add a figure to the drawing at a given position.
	 * @param f The figure to add.
	 * @param index THe position where insert the figure.
	 * @throws IllegalArgumentException If f is null or if the index is bad.
	 */
	public void addFigure(Figure f, int index)
	{
		if(index<0 || (index>=figures.size() && index>0))
			throw new IllegalArgumentException("Bad index.");//$NON-NLS-1$
		
		if(f==null)
			throw new IllegalArgumentException("f is null.");//$NON-NLS-1$
		
		f.onRelease();
		figures.add(index, f);
		updateBorders();
		updateGravityCenter();
	}
	
	



	@Override
	public synchronized void setOnRotation(boolean on)
	{
		super.setOnRotation(on);
		int i, size=figures.size();
		
		for(i=0; i<size; i++)
			figures.elementAt(i).setOnRotation(on);
		
		if(!on)
		{
			borders.rotationAngle = 0;
			updateBorders();
			updateGravityCenter();
		}
	}
	
	
	
	
	/**
	 * Allows to remove a figure of the drawing.
	 * @param f The figure to remove.
	 */
	public void removeFigure(Figure f)
	{
		if(f!=null && figures.contains(f))
		{
			figures.remove(f);
			updateBorders();
		}
	}


	/**
	 * Allows to remove a figure of the drawing.
	 * @param id the position of the figure in the vector.
	 * @exception IllegalArgumentException if the <code>id</code> is invalid.
	 * @return The deleted figure.
	 * @since 1.9.1
	 */
	public Figure removeFigureAt(int id)
	{
		if(id<0 || id>=figures.size())
			throw new IllegalArgumentException();
		
		Figure f = figures.remove(id);
		updateBorders();
		
		return f;
	}
	
	
	
	
	/**
	 * Allows to update the dimension and the position of the borders of the drawing.
	 */
	public void updateBorders()
	{
		if(minBorders==null) initializeBorders();
		if(borders==null) initializeBorders();
		
		if(figures.isEmpty())
		{
			borders.setFirstPoint(0, 0);
			borders.setLastPoint(0, 0);
			minBorders.setFirstPoint(0, 0);
			minBorders.setLastPoint(0, 0);
		}
		else
		{
			int size = figures.size(), i;
			LaTeXDrawPoint2D NW, SE, NW2, SE2;
			LaTeXDrawPoint2D NWb, SEb, NW2b, SE2b;
			Figure f = figures.firstElement();

			NW = (LaTeXDrawPoint2D) f.getTheNWBoundPoint().clone();
			SE = (LaTeXDrawPoint2D) f.getTheSEBoundPoint().clone();
			
			if(f instanceof Draw)
			{
				NWb = (LaTeXDrawPoint2D)((Draw)f).minBorders.getTheNWPoint().clone();
				SEb = (LaTeXDrawPoint2D)((Draw)f).minBorders.getTheSEPoint().clone();
			}
			else
			{
				NWb = (LaTeXDrawPoint2D) f.getTheNWPoint().clone();
				SEb = (LaTeXDrawPoint2D) f.getTheSEPoint().clone();
			}

			for(i=1; i<size; i++)
			{
				f = figures.elementAt(i);

				NW2 = f.getTheNWBoundPoint();
				SE2 = f.getTheSEBoundPoint();
				
				if(f instanceof Draw)
				{
					NW2b = (LaTeXDrawPoint2D)((Draw)f).minBorders.getTheNWPoint().clone();
					SE2b = (LaTeXDrawPoint2D)((Draw)f).minBorders.getTheSEPoint().clone();
				}
				else
				{
					NW2b = (LaTeXDrawPoint2D) f.getTheNWPoint().clone();
					SE2b = (LaTeXDrawPoint2D) f.getTheSEPoint().clone();
				}
				
				if(NW2.x<NW.x) NW.x = NW2.x;
				if(NW2.y<NW.y) NW.y = NW2.y;
				if(SE2.x>SE.x) SE.x = SE2.x;
				if(SE2.y>SE.y) SE.y = SE2.y;
				
				if(NW2b.x<NWb.x) NWb.x = NW2b.x;
				if(NW2b.y<NWb.y) NWb.y = NW2b.y;
				if(SE2b.x>SEb.x) SEb.x = SE2b.x;
				if(SE2b.y>SEb.y) SEb.y = SE2b.y;
			}
			
			borders.setFirstPoint(NW.x, NW.y);
			borders.setLastPoint(SE);
			minBorders.setFirstPoint(NWb);
			minBorders.setLastPoint(SEb);
			shape = createShape2D();
		}
	}
	
	
	
	
	@Override
	public void onDragged(Point formerPt, Point newPt) throws Exception 
	{
		if(formerPt.equals(newPt)) return;
		
		int i, size = figures.size();
		
		if(figures.size()==1 && !drawFigures)
		{
			figures.firstElement().onDragged(formerPt, newPt);
			updateBorders();
			updateGravityCenter();
			return;
		}
		
		if(borders.dSelected!=null)
		{
			if(isOnRotation)
			{
				double angle = computeRotationAngle(formerPt, newPt);
				
				rotationAngle+=angle;
				rotationAngle%=(2*Math.PI);
				
				if(formerGravityC==null)
					throw new IllegalArgumentException("beginRotate must be called before to initialise the rotation"); //$NON-NLS-1$
				
				for(i=0; i<size; i++)
					figures.elementAt(i).rotate(formerGravityC, angle);
				
				return ;
			}
			
			if((borders.dSelected==borders.dNE || borders.dSelected==borders.dNW ||
				borders.dSelected==borders.dN) && newPt.y>=borders.dS.getY()) 
					return ;
			if((borders.dSelected==borders.dSE || borders.dSelected==borders.dSW ||
				borders.dSelected==borders.dS) && newPt.y<=borders.dN.getY()) 
					return ;
			if((borders.dSelected==borders.dSE || borders.dSelected==borders.dNE ||
				borders.dSelected==borders.dE) && newPt.x<=borders.dW.getX()) 
					return ;
			if((borders.dSelected==borders.dSW || borders.dSelected==borders.dNW ||
				borders.dSelected==borders.dW) && newPt.x>=borders.dE.getX()) 
					return ;
				
			boolean rescalable;
			Figure f;

			if(borders.dSelected==borders.dE || borders.dSelected==borders.dNE || borders.dSelected==borders.dSE)
			{
				
				double dEx = borders.dE.getX(), dWx = borders.dW.getX();
				double percent = Math.abs((newPt.x-dWx)/(dEx-dWx));
				rescalable = true;
				i=0;

				while(i<size && rescalable)
				{
					f = getFigureAt(i);
					if(f.isResizable())
					{
						if(f.isTooSmallToBeRescaled() && dEx>=newPt.x)
							rescalable = false;
					}
					else rescalable = false;
					i++;
				}

				// The user must not be able to reduce too much to figure
				if(rescalable)
					rescaleX(minBorders.getTheSEPoint().x, 
						minBorders.getTheSEPoint().x+dEx-newPt.x, percent, minBorders);
			}
			if(borders.dSelected==borders.dW || borders.dSelected==borders.dNW || borders.dSelected==borders.dSW)
			{
				double dEx = borders.dE.getX(), dWx = borders.dW.getX();
				double percent = Math.abs((newPt.x-dEx)/(dWx-dEx));
				rescalable = true;							
				i=0;
				
				while(i<size && rescalable)
				{
					f = getFigureAt(i);
					if(f.isResizable())
					{
						if(f.isTooSmallToBeRescaled() && dWx<=newPt.x)
							rescalable = false;
					}
					else rescalable = false;
					i++;
				}
				
				if(rescalable)
					rescaleX(minBorders.getTheNWPoint().x, 
							minBorders.getTheNWPoint().x+dWx-newPt.x, percent, minBorders);
			}
			if(borders.dSelected==borders.dN || borders.dSelected==borders.dNW || borders.dSelected==borders.dNE)
			{
				double dNy = borders.dN.getY(), dSy = borders.dS.getY();
				double percent = Math.abs((newPt.y-dSy)/(dNy-dSy));
				rescalable = true;	
				i=0;
				
				while(i<size && rescalable)
				{
					f = getFigureAt(i);
					if(f.isResizable())
					{
						if(f.isTooSmallToBeRescaled() && dNy<=newPt.y)
							rescalable = false;
					}
					else rescalable = false;
					i++;
				}
				
				if(rescalable)
				//	 The user must not be able to reduce too much to figure
					rescaleY(minBorders.getTheNWPoint().y, 
							minBorders.getTheNWPoint().y, percent, minBorders);
			}
			if(borders.dSelected==borders.dS || borders.dSelected==borders.dSW || borders.dSelected==borders.dSE)
			{
				double dNy = borders.dN.getY(), dSy = borders.dS.getY();
				double percent = Math.abs((newPt.y-dNy)/(dSy-dNy));
				rescalable = true;	
				i=0;
				
				while(i<size && rescalable)
				{
					f = getFigureAt(i);
					if(f.isResizable())
					{
						if(f.isTooSmallToBeRescaled() && dSy>=newPt.y)
							rescalable = false;
					}
					else rescalable = false;
					i++;
				}
				
				if(rescalable)
				//	 The user must not be able to reduce too much to figure
					rescaleY(minBorders.getTheSEPoint().y, 
								minBorders.getTheSEPoint().y-dSy+newPt.y, percent, minBorders);
			}
		}//if(borders.dSelected==null)
		else
			for(i=0; i<size; i++)
				figures.elementAt(i).onDragged(formerPt, newPt);

		updateBorders();
		updateGravityCenter();
	}

	
	
	
	
	/**
	 * Allows to know if a figure is in the drawing.
	 * @param f The figure that we want to know if it's in the drawing .
	 * @return True if the figure is in the drawing.
	 */
	public boolean contains(Figure f)
	{
		if(f!=null)
			return figures.contains(f);
		return false;
	}
	



	
	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		if(drawFigures)
		{
			int i, size = size();
			
			for(i=0; i<size; i++)
				figures.elementAt(i).draw(g, antiAlias, rendering, alphaInter, colorRendering);
		}
		
		if(figures.size()>1 && isSelected && borders!=null)
			borders.draw(g, antiAlias, rendering, alphaInter, colorRendering);
	}





	/**
	 * Allows to know if there is a figure in the drawing.
	 * @return True if there is at least one figure in the drawing.
	 */
	public boolean isEmpty()
	{
		return figures.isEmpty();
	}





	@Override
	public boolean isIn(LaTeXDrawPoint2D pt) 
	{
		int i, size = figures.size();
		boolean in = false;
		
		for(i=0; i<size && !in; i++)
			in = figures.elementAt(i).isIn(pt);

		if(in) return true;
		
		return borders.isIn(pt);
	}

	
	
	
	
	/**
	 * Allows to get the figure which contains the point <code>point</code>.
	 * @param pt The point of reference.
	 * @return The figure which contains the point.
	 */
	public Figure whereIsIt(Point pt) 
	{
		int i, size = figures.size();
		boolean here = false;
		
		for(i=0; i<size && !here; i++)
			here = figures.elementAt(i).isIn(pt);
		
		if(here) return figures.elementAt(i-1);
		return null;
			
	}
	
	
	
	/**
	 * Allows to get the figure placed at the position <code>id</code> in the vector <code>Figures</code>.
	 * @param id The position of the figure in the vector Figures.
	 * @return The figure corresponding to the identifier.
	 */
	public Figure getFigureAt(int id)
	{
		if(id<0 || id>=figures.size())
			throw new ArrayIndexOutOfBoundsException(id);
		
		return figures.elementAt(id);
	}
	
	
	
	/**
	 * Allows to get the number of figures which contains the drawing.
	 * @return The number of figures in the drawing.
	 */
	public int size()
	{
		return figures.size();
	}
	
	
	
	
	
	@Override
	public String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		int i, size = figures.size();
		if(drawFigures && size>0)
		{
			String code=""; //$NON-NLS-1$
			
			for(i=0; i<size-1; i++)
				code+=figures.elementAt(i).getCodePSTricks(drawBorders, ppc)+"\n"; //$NON-NLS-1$
			code+= figures.lastElement().getCodePSTricks(drawBorders, ppc);
			
			return code;
		}

		return null;
	}

	
	
	@Override
	public void onClick(Point pt) 
	{
		super.onClick(pt);

		if(!drawFigures && size()==1)
				figures.elementAt(0).onClick(pt);
	}

	
	
	@Override
	public void onRelease() 
	{
		super.onRelease();		
		
		int i,size=figures.size();
		
		for(i=0; i<size; i++)
			figures.elementAt(i).onRelease();
	}

	
	
	@Override
	public void onDelimitorRelease()
	{
		super.onDelimitorRelease();	
		
		int i,size=figures.size();
		
		for(i=0; i<size; i++)
			figures.elementAt(i).onDelimitorRelease();
	}
	
	


	@Override
	public void shift(double shiftX, double shiftY)
	{
		if(shiftX==0 && shiftY==0) return ;
		
		for(Figure f: figures)
			f.shift(shiftX, shiftY);
		
		updateBorders();
		updateGravityCenter();
	}


	
	@Override
	public synchronized void setSelected(boolean state)
	{
		super.setSelected(state);

		if(drawFigures || !state)
			for(Figure f : figures)
				f.setSelected(false);
		else
			if(size()>=1)
				figures.firstElement().setSelected(size()==1);
	}
	
	
	
	
	@Override
	public synchronized void setRotationAngle(double theta)
	{
		double angle = theta - rotationAngle;
		super.setRotationAngle(theta);
		
		if(drawFigures)
		{
			int i, size = figures.size();
			
			for(i=0; i<size; i++)
				figures.elementAt(i).rotate(gravityCenter, angle);
		}
		
		borders.setRotationAngle(0);
		updateBorders();
		updateGravityCenter();
	}
	
	
	
	
	@Override
	public synchronized void setLastPoint(double x, double y) 
	{
		updateGravityCenter();
	}

	
	
	
	@Override
	public synchronized void setFirstPoint(double x, double y) 
	{
		updateGravityCenter();
	}
	
	
	
	/**
	 * Allows to clear the drawing: remove and unselect all the figures of the vector <code>figures</code>.
	 */
	public void clear()
	{
		setSelected(false);
		figures.removeAllElements();
		shape = createShape2D();
	}



	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Draw d = (Draw) super.clone();
		d.drawFigures = drawFigures;
		
		int i, size = size();
		d.figures = new Vector<Figure>();

		for(i=0; i<size; i++)
			d.addFigure((Figure) getFigureAt(i).clone());

		d.borders = (LaTeXDrawRectangle) borders.clone();
		d.updateBorders();
		d.updateGravityCenter();

		return d;
	}




	@Override
	public void updateGravityCenter() 
	{
		LaTeXDrawPoint2D nw = new LaTeXDrawPoint2D(Double.MAX_VALUE, Double.MAX_VALUE);
		LaTeXDrawPoint2D se = new LaTeXDrawPoint2D(Double.MIN_VALUE, Double.MIN_VALUE);
		LaTeXDrawPoint2D nw2, se2;
		
		for(Figure f : figures)
		{
			nw2 = f.getTheNWRotatedPoint();
			se2 = f.getTheSERotatedPoint();
			
			if(nw2.x<nw.x) nw.x = nw2.x;
			if(nw2.y<nw.y) nw.y = nw2.y;
			if(se2.x>se.x) se.x = se2.x;
			if(se2.y>se.y) se.y = se2.y;
		}
		
		gravityCenter.x = (nw.x+se.x)/2.;
		gravityCenter.y = (nw.y+se.y)/2.;
	}




	@Override
	public void updateStyleOfDelimitors() 
	{
		super.updateStyleOfDelimitors();
		
		for(int i=0, size=figures.size(); i<size; i++)
			figures.elementAt(i).updateStyleOfDelimitors();
	}




	@Override
	public boolean intersected(Rectangle2D.Double r) 
	{
		boolean inter = false;
		int i=0, size = figures.size();
		
		while(!inter && i<size)
			if(figures.elementAt(i).intersected(r))
					inter = true;
			else i++;

		return inter;
	}





	@Override
	public Shape createShape2D() 
	{
		return borders.shape;
	}





	@Override
	public void rescaleX(double formerX, double newX, double percent, LaTeXDrawRectangle bound) 
	{
		if(percent==1.) return ;
		if(bound==null) throw new IllegalArgumentException();
		
		int i, size = figures.size();
		
		for(i=0; i<size; i++)
			figures.elementAt(i).rescaleX(formerX, newX, percent, bound);
		
		updateBorders();
	}





	@Override
	public void rescaleY(double formerY, double newY, double percent, LaTeXDrawRectangle bound) 
	{
		if(percent==1.) return ;
		if(bound==null) throw new IllegalArgumentException();
		
		int i, size = figures.size();
		
		for(i=0; i<size; i++)
			figures.elementAt(i).rescaleY(formerY, newY, percent, bound);
		
		updateBorders();
	}

	
	
	
	@SuppressWarnings("unchecked")
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		rotationAngle = ois.readDouble();
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		borders =(LaTeXDrawRectangle) ois.readObject();

		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")<0) //$NON-NLS-1$
			ois.readDouble();
		figures = (Vector<Figure>) ois.readObject();
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.5")>=0)//$NON-NLS-1$
			drawFigures = ois.readBoolean();
		else
		{
			ois.readObject();
			drawFigures = DEFAULT_DRAWFIGURE;
		}
			
		updateStyleOfDelimitors();
		canBeFilled = false;	
		initializeBorders();
		
		if(linesColor==null)
		{
			linesColor 		= DEFAULT_BORDERS_COL;
			doubleColor 	= DEFAULT_DOUBLE_COLOR;
			interiorColor 	= DEFAULT_INTERIOR_COL;
			hatchingColor	= DEFAULT_HATCH_COL;
			gradientEndColor	= PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
			gradientStartColor	= PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
			shadowColor		= DEFAULT_SHADOW_COLOR;
		}
	}


	
	/**
	 * @return Returns the drawFigures.
	 */
	public boolean isDrawFigures()
	{
		return drawFigures;
	}




	/**
	 * @param drawFigures The drawFigures to set.
	 */
	public synchronized void setDrawFigures(boolean drawFigures)
	{
		this.drawFigures = drawFigures;
	}



	@Override
	public Shape createNonRotatedShape2D()
	{
		return borders.createNonRotatedShape2D();
	}




	@Override
	@Deprecated
	public void updateShape()
	{
		/* The shape is not used by a drawing. */
	}



	/**
	 * @return the minBorders.
	 */
	public LaTeXDrawRectangle getMinBorders()
	{
		return minBorders;
	}




	/**
	 * @param minBorders the minBorders to set.
	 */
	public void setMinBorders(LaTeXDrawRectangle minBorders)
	{
		this.minBorders = minBorders;
	}




	@Override
	public boolean isTooSmallToBeRescaled()
	{
		// TODO Auto-generated method stub
		return false;
	}



	@Override
	public void rotate(LaTeXDrawPoint2D gravityC, double angle)
	{
		for(Figure f : figures)
			f.rotate(gravityC, angle);
		
		updateBorders();
	}




	@Override
	public void mirrorHorizontal(LaTeXDrawPoint2D origin)
	{
		for(Figure f : figures)
			f.mirrorHorizontal(origin);
		
		updateBorders();
		updateGravityCenter();
	}




	@Override
	public void mirrorVertical(LaTeXDrawPoint2D origin)
	{
		for(Figure f : figures)
			f.mirrorVertical(origin);
		
		updateBorders();
		updateGravityCenter();
	}




	@Override
	public synchronized LaTeXDrawPoint2D getLastPoint()
	{
		return null;
	}
	

	
	
	/**
	 * @since 1.9
	 * @return The figures of the drawing.
	 */
	public synchronized Vector<Figure> getFigures()
	{
		return figures;
	}




	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		for(Figure figure : figures)
			figure.updateToGrid(grid);
		
		updateBorders();
	}




	@Override
	public int getSelectedDelimitorOrientation()
	{
		if(size()==1)
			return getFigureAt(0).getSelectedDelimitorOrientation();
		
		return borders.getSelectedDelimitorOrientation();
	}

	
	
	@Override
	public int hashCode()
	{
		return super.hashCode()^5;
	}
	
	
	
	
	/**
	 * This method must be called before each rotation of the selection. Initialize the rotation.
	 * @since 1.9
	 */
	public void beginRotation()
	{
		updateGravityCenter();
		formerGravityC = (LaTeXDrawPoint2D)gravityCenter.clone();
		
		if(!drawFigures)
			for(Figure f : figures)
				if(f instanceof Draw)
					((Draw)f).beginRotation();
	}
	
	

	/**
	 * This method must be called after each rotation of the selection. Deinitialize the rotation.
	 * @since 1.9
	 */
	public void endRotation()
	{
		updateGravityCenter();
		updateBorders();
		formerGravityC = null;
		
		if(!drawFigures)
			for(Figure f : figures)
				if(f instanceof Draw)
					((Draw)f).endRotation();
	}


	@Override
	public synchronized float getThickness()
	{
		if(figures==null || figures.isEmpty())
			return super.getThickness();
		
		return figures.firstElement().getThickness();
	}




	@Override
	public synchronized void setThickness(float value)
	{
		if(figures!=null)
			for(Figure f : figures)
				f.setThickness(value);
	}




	@Override
	public synchronized Color getDoubleColor()
	{
		if(figures==null || figures.isEmpty())
			return super.getDoubleColor();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).isDoubleBoundaryable())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).getDoubleColor();
			
		return figures.firstElement().getDoubleColor();
	}




	@Override
	public synchronized Color getGradientEndColor()
	{
		if(figures==null || figures.isEmpty())
			return super.getGradientEndColor();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).canBeHatched())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).getGradientEndColor();
			
		return figures.firstElement().getGradientEndColor();
	}




	@Override
	public synchronized Color getGradientStartColor()
	{
		if(figures==null || figures.isEmpty())
			return super.getGradientStartColor();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).canBeHatched())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).getGradientStartColor();
		
		return figures.firstElement().getGradientStartColor();
	}




	@Override
	public synchronized Color getHatchingColor()
	{
		if(figures==null || figures.isEmpty())
			return super.getHatchingColor();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).canBeHatched())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).getHatchingColor();
		
		return figures.firstElement().getHatchingColor();
	}




	@Override
	public synchronized String getHatchingStyle()
	{
		if(figures==null || figures.isEmpty())
			return super.getHatchingStyle();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).canBeHatched())
				ok = false;
			else
				i++;

		if(!ok && size>1)
			return figures.elementAt(i).getHatchingStyle();

		return figures.firstElement().getHatchingStyle();
	}




	@Override
	public synchronized Color getInteriorColor()
	{
		if(figures==null || figures.isEmpty())
			return super.getInteriorColor();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).canBeFilled())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).getInteriorColor();
			
		return figures.firstElement().getInteriorColor();
	}




	@Override
	public synchronized Color getLinesColor()
	{
		if(figures==null || figures.isEmpty())
			return super.getLinesColor();
		
		return figures.firstElement().getLinesColor();
	}




	@Override
	public synchronized String getLineStyle()
	{
		if(figures==null || figures.isEmpty())
			return super.getLineStyle();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).isDashableOrDotable())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).getLineStyle();
			
		return figures.firstElement().getLineStyle();
	}




	@Override
	public synchronized boolean hasDoubleBoundary()
	{
		if(figures==null || figures.isEmpty())
			return super.hasDoubleBoundary();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).isDoubleBoundaryable())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).hasDoubleBoundary();
					
		return figures.firstElement().hasDoubleBoundary();
	}




	@Override
	public synchronized boolean hasShadow()
	{
		if(figures==null || figures.isEmpty())
			return super.hasShadow();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).canHaveShadow())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).hasShadow();
		
		return figures.firstElement().hasShadow();
	}




	@Override
	public synchronized boolean isFilled()
	{
		if(figures==null || figures.isEmpty())
			return super.isFilled();
		
		int i=0, size = figures.size();
		boolean ok=true;
		
		while(ok && i<size)
			if(figures.elementAt(i).canBeFilled())
				ok = false;
			else
				i++;
		
		if(!ok && size>1)
			return figures.elementAt(i).isFilled();

		return figures.firstElement().isFilled();
	}


	
	@Override
	public String toString()
	{
		String str = "["; //$NON-NLS-1$
		
		for(int i=0, size=figures.size()-1; i<size; i++)
			str += figures.elementAt(i).getClass().getName() + '(' + figures.elementAt(i).getNumber() + "), " ;  //$NON-NLS-1$
		
		if(!figures.isEmpty())
			str += figures.lastElement().getClass().getCanonicalName() + '(' + figures.lastElement().getNumber() + ")" ; //$NON-NLS-1$
		
		return str + ']';
	}
	
	


	@Override
	public synchronized void setDoubleColor(Color doublecolor)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.isDoubleBoundaryable())
					f.setDoubleColor(doublecolor);
	}




	@Override
	public synchronized void setGradientEndColor(Color gradientEndColor)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canBeFilled())
					f.setGradientEndColor(gradientEndColor);
	}




	@Override
	public synchronized void setGradientStartColor(Color gradientStartColor)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canBeFilled())
					f.setGradientStartColor(gradientStartColor);
	}




	@Override
	public synchronized void setHasDoubleBoundary(boolean hasDoubleBoundary)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.isDoubleBoundaryable())
					f.setHasDoubleBoundary(hasDoubleBoundary);
	}




	@Override
	public synchronized void setHasShadow(boolean hasShadow)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canHaveShadow())
					f.setHasShadow(hasShadow);
	}




	@Override
	public synchronized void setHatchingColor(Color color)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canBeHatched())
					f.setHatchingColor(color);
	}




	@Override
	public synchronized void setHatchingStyle(String style)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canBeHatched())
					f.setHatchingStyle(style);
	}




	@Override
	public synchronized void setInteriorColor(Color c)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canBeFilled())
					f.setInteriorColor(c);
	}




	@Override
	public synchronized void setLinesColor(Color c)
	{
		if(figures!=null)
			for(Figure f : figures)
				f.setLinesColor(c);
	}




	@Override
	public synchronized void setLineStyle(String style)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.isDashableOrDotable())
					f.setLineStyle(style);
	}




	@Override
	public synchronized void setShadowColor(Color shadowColor)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canHaveShadow())
					f.setShadowColor(shadowColor);
	}
	
	
	
	@Override
	public synchronized void setIsFilled(boolean filled)
	{
		if(figures!=null)
			for(Figure f : figures)
				if(f.canBeFilled())
					f.setIsFilled(filled);
	}
	
	


	/**
	 * Distributes vertically at equal distance between the bottom sides of the selected figures.
	 * @since 2.0.0
	 */
	public void distributeVertBottom()
	{
		if(size()<2)
			return ;
		
		LinkedList<Figure> sortedF = new LinkedList<Figure>();
		LinkedList<Double> ses = new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		boolean ok;
		double gap;
		int i, size;
		
		for(Figure f : figures)
		{
			pt = f.getTheSERotatedPoint();
			ok = true;
			
			for(i=0, size=sortedF.size(); i<size && ok; i++)
				if(pt.y<ses.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				ses.addLast(pt.y);
			}
			else
			{
				sortedF.add(i-1, f);
				ses.add(i-1, pt.y);
			}
		}
		
		gap = (ses.getLast()-ses.getFirst())/(size()-1);
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			sortedF.get(i).shift(0, (ses.getFirst()+i*gap)-ses.get(i));
		
		updateBorders();
	}
	
	
	
	/**
	 * Distributes vertically at equal distance between the middle of the selected figures. 
	 * @since 2.0.0
	 */
	public void distributeVertMiddle()
	{
		LinkedList<Figure> sortedF = new LinkedList<Figure>();
		LinkedList<Double> gcs = new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		int i, size;
		boolean ok;
		double gap;
		
		for(Figure f : figures)
		{
			pt = f.getGravityCenter();
			ok = true;
			
			for(i=0, size=sortedF.size(); i<size && ok; i++)
				if(pt.y<gcs.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				gcs.addLast(pt.y);
			}
			else
			{
				sortedF.add(i-1, f);
				gcs.add(i-1, pt.y);
			}
		}
		
		gap = (gcs.getLast()-gcs.getFirst())/(size()-1);
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			sortedF.get(i).shift(0, (gcs.getFirst()+i*gap)-gcs.get(i));
		
		updateBorders();
	}
	
	
	
	/**
	 * Distributes vertically at equal distance between the top sides of the selected figures. 
	 * @since 2.0.0
	 */
	public void distributeVertTop()
	{
		if(size()<2)
			return ;
		
		LinkedList<Figure> sortedF = new LinkedList<Figure>();
		LinkedList<Double> nws = new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		int i, size;
		boolean ok;
		double gap;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
			ok = true;
			
			for(i=0, size=sortedF.size(); i<size && ok; i++)
				if(pt.y<nws.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				nws.addLast(pt.y);
			}
			else
			{
				sortedF.add(i-1, f);
				nws.add(i-1, pt.y);
			}
		}
		
		gap = (nws.getLast()-nws.getFirst())/(size()-1);
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			sortedF.get(i).shift(0, (nws.getFirst()+i*gap)-nws.get(i));
		
		updateBorders();
	}
	
	
	
	/**
	 * Distributes horizontally at equal distance between the selected figures. 
	 * @since 2.0.0
	 */
	public void distributeVertEqual()
	{
		if(size()<2)
			return ;
		
		LinkedList<Figure> sortedF = new LinkedList<Figure>();
		LinkedList<Double> tops = new LinkedList<Double>();
		LinkedList<Double> bots = new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		int i, size;
		boolean ok;
		double gap;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
				
			ok = true;
			for(i=0, size=tops.size(); i<size && ok; i++)
				if(pt.y<tops.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				tops.addLast(pt.y);
				bots.addLast(f.getTheSERotatedPoint().y);
			}
			else
			{
				sortedF.add(i-1, f);
				tops.add(i-1, pt.y);
				bots.add(i-1, f.getTheSERotatedPoint().y);
			}
		}
		
		gap = tops.getLast() - bots.getFirst();
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			gap -= bots.get(i) - tops.get(i);
		
		gap/=size;
		
		for(i=1; i<size; i++)
			sortedF.get(i).shift(0, (sortedF.get(i-1).getTheSERotatedPoint().y + gap) - tops.get(i));
		
		updateBorders();
	}
	
	
	
	/**
	 * Distributes vertically at equal distance between the left sides of the selected figures.
	 * @since 2.0.0
	 */
	public void distributeHorizLeft()
	{
		if(size()<2)
			return ;
		
		LinkedList<Figure> sortedF = new LinkedList<Figure>();
		LinkedList<Double> nws = new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		int i, size;
		boolean ok;
		double gap;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
			ok = true;
			
			for(i=0, size=sortedF.size(); i<size && ok; i++)
				if(pt.x<nws.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				nws.addLast(pt.x);
			}
			else
			{
				sortedF.add(i-1, f);
				nws.add(i-1, pt.x);
			}
		}
		
		gap = (nws.getLast()-nws.getFirst())/(size()-1);
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			sortedF.get(i).shift((nws.getFirst()+i*gap)-nws.get(i), 0);
		
		updateBorders();
	}
	
	
	
	/**
	 * Distributes vertically at equal distance between the middle of the selected figures. 
	 * @since 2.0.0
	 */
	public void distributeHorizMiddle()
	{
		if(size()<2)
			return ;
		
		LinkedList<Figure> sortedF = new LinkedList<Figure>();
		LinkedList<Double> gcs = new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		int i, size;
		boolean ok;
		double gap;
		
		for(Figure f : figures)
		{
			pt = f.getGravityCenter();
			ok = true;
			
			for(i=0, size=sortedF.size(); i<size && ok; i++)
				if(pt.x<gcs.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				gcs.addLast(pt.x);
			}
			else
			{
				sortedF.add(i-1, f);
				gcs.add(i-1, pt.x);
			}
		}
		
		gap = (gcs.getLast()-gcs.getFirst())/(size()-1);
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			sortedF.get(i).shift((gcs.getFirst()+i*gap)-gcs.get(i), 0);
		
		updateBorders();
	}
	
	
	
	/**
	 * Distributes vertically at equal distance between the right sides of the selected figures. 
	 * @since 2.0.0
	 */
	public void distributeHorizRight()
	{
		if(size()<2)
			return ;
		
		LinkedList<Figure> sortedF = new LinkedList<Figure>();
		LinkedList<Double> ses = new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		int i, size;
		boolean ok;
		double gap;
		
		for(Figure f : figures)
		{
			pt = f.getTheSERotatedPoint();
			
			ok = true;
			for(i=0, size=sortedF.size(); i<size && ok; i++)
				if(pt.x<ses.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				ses.addLast(pt.x);
			}
			else
			{
				sortedF.add(i-1, f);
				ses.add(i-1, pt.x);
			}
		}
		
		gap = (ses.getLast()-ses.getFirst())/(size()-1);
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			sortedF.get(i).shift((ses.getFirst()+i*gap)-ses.get(i), 0);
		
		updateBorders();
	}
	
	
	
	/**
	 * Distributes vertically at equal distance between the selected figures. 
	 * @since 2.0.0
	 */
	public void distributeHorizEqual()
	{
		if(size()<2)
			return ;
		
		LinkedList<Figure> sortedF 	= new LinkedList<Figure>();
		LinkedList<Double> rights 	= new LinkedList<Double>();
		LinkedList<Double> lefts 	= new LinkedList<Double>();
		LaTeXDrawPoint2D pt;
		int i, size;
		boolean ok;
		double gap;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
			ok = true;
			
			for(i=0, size=lefts.size(); i<size && ok; i++)
				if(pt.x<lefts.get(i))
					ok = false;
			
			if(ok)
			{
				sortedF.addLast(f);
				lefts.addLast(pt.x);
				rights.addLast(f.getTheSERotatedPoint().x);
			}
			else
			{
				sortedF.add(i-1, f);
				lefts.add(i-1, pt.x);
				rights.add(i-1, f.getTheSERotatedPoint().x);
			}
		}
		
		gap = lefts.getLast() - rights.getFirst();
		size = sortedF.size()-1;
		
		for(i=1; i<size; i++)
			gap -= rights.get(i) - lefts.get(i);
		
		gap/=size;

		for(i=1; i<size; i++)
			sortedF.get(i).shift((sortedF.get(i-1).getTheSERotatedPoint().x + gap) - lefts.get(i), 0);
	}


	

	/**
	 * Align the selected figures to the figure on the left. 
	 * @since 2.0.0
	 */
	public void alignLeft()
	{
		if(size()<2)
			return ;
		
		LinkedList<LaTeXDrawPoint2D> nws = new LinkedList<LaTeXDrawPoint2D>();
		double minX = Double.MAX_VALUE;
		LaTeXDrawPoint2D pt;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
			nws.add(pt);
			
			if(pt.x<minX)
				minX = pt.x;
		}
		
		for(Figure f : figures)
		{
			pt = nws.poll();
			
			if(((float)minX)!=((float)pt.x))
				f.shift(minX-pt.x, 0);
		}
		
		updateBorders();
	} 

	
	
	
	/**
	 * Align the selected figures to the figure on the right. 
	 * @since 2.0.0
	 */
	public void alignRight()
	{
		if(size()<2)
			return ;
		
		LinkedList<LaTeXDrawPoint2D> ses = new LinkedList<LaTeXDrawPoint2D>();
		double maxX = Double.MIN_VALUE;
		LaTeXDrawPoint2D pt;
		
		for(Figure f : figures)
		{
			pt = f.getTheSERotatedPoint();
			ses.add(pt);
			
			if(pt.x>maxX)
				maxX = pt.x;
		}
		
		for(Figure f : figures)
		{
			pt = ses.poll();
			
			if(((float)maxX)!=((float)pt.x))
				f.shift(maxX-pt.x, 0);
		}
		
		updateBorders();
	} 
	
	
	
	/**
	 * Align the selected figures to the figure on the top. 
	 * @since 2.0.0
	 */
	public void alignTop()
	{
		if(size()<2)
			return ;
		
		LinkedList<LaTeXDrawPoint2D> nws = new LinkedList<LaTeXDrawPoint2D>();
		double minY = Double.MAX_VALUE;
		LaTeXDrawPoint2D pt;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
			nws.add(pt);
			
			if(pt.y<minY)
				minY = pt.y;
		}
		
		for(Figure f : figures)
		{
			pt = nws.poll();
			
			if(((float)minY)!=((float)pt.x))
				f.shift(0, minY-pt.y);
		}
		
		updateBorders();
	} 
	
	
	
	/**
	 * Align the selected figures to the figure on the bottom. 
	 * @since 2.0.0
	 */
	public void alignBottom()
	{
		if(size()<2)
			return ;
		
		LinkedList<LaTeXDrawPoint2D> ses = new LinkedList<LaTeXDrawPoint2D>();
		double maxY = Double.MIN_VALUE;
		LaTeXDrawPoint2D pt;
		
		for(Figure f : figures)
		{
			pt = f.getTheSERotatedPoint();
			ses.add(pt);
			
			if(pt.y>maxY)
				maxY = pt.y;
		}
		
		for(Figure f : figures)
		{
			pt = ses.poll();
			
			if(((float)maxY)!=((float)pt.y))
				f.shift(0, maxY-pt.y);
		}
		
		updateBorders();
	} 
	
	
	
	/**
	 * Align vertically the selected figures to the figure in the middle. 
	 * @since 2.0.0
	 */
	public void alignMiddleVertically()
	{
		if(size()<2)
			return ;
		
		LinkedList<Double> middles = new LinkedList<Double>();
		double maxX = Double.MIN_VALUE;
		double minX = Double.MAX_VALUE, middle, middle2;
		LaTeXDrawPoint2D pt, pt2;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
			pt2 = f.getTheSERotatedPoint();
			
			if(pt.x<minX)
				minX = pt.x;
			
			if(pt2.x>maxX)
				maxX = pt2.x;
			
			middles.add((pt.x+pt2.x)/2.);
		}
		
		middle = (minX+maxX)/2.;
		
		for(Figure f : figures)
		{
			middle2 = middles.poll();
			
			if(((float)middle2)!=((float)middle))
				f.shift(middle-middle2, 0);
		}
		
		updateBorders();
	} 
	
	
	
	/**
	 * Align horizontally the selected figures to the figure in the middle. 
	 * @since 2.0.0
	 */
	public void alignMiddleHorizontally()
	{
		if(size()<2)
			return ;
		
		LinkedList<Double> middles = new LinkedList<Double>();
		double maxY = Double.MIN_VALUE;
		double minY = Double.MAX_VALUE, middle, middle2;
		LaTeXDrawPoint2D pt, pt2;
		
		for(Figure f : figures)
		{
			pt = f.getTheNWRotatedPoint();
			pt2 = f.getTheSERotatedPoint();
			
			if(pt.y<minY)
				minY = pt.y;
			
			if(pt2.y>maxY)
				maxY = pt2.y;
			
			middles.add((pt.y+pt2.y)/2.);
		}
		
		middle = (minY+maxY)/2.;
		
		for(Figure f : figures)
		{
			middle2 = middles.poll();
			
			if(((float)middle2)!=((float)middle))
				f.shift(0, middle-middle2);
		}
		
		updateBorders();
	}

}
