/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef PATTERNCANVAS_H
#define PATTERNCANVAS_H

#include <qobject.h>
#include <qptrvector.h>
#include <qptrlist.h>
#include <qpoint.h>
#include <qrect.h>
#include "stitch.h"

typedef struct
{
  double  stitchLength;
  double  backstitchLength;
  int     full;
  int     half;
  int     quarter;
  int     threeQuarter;
  int     knots;
  int     backstitches;
} FlossUsage;

typedef QMap<int,FlossUsage> UsageMap;

/** PatternCanvas class stores the data related to the patterns stitches.
    A QVector array of the StitchQueue type is defined to store the list of
    stitches within each cell.  A QList of the BackStitch class is defined
    to store the backstitching, and a QList of Knots is defined to store
    the french knots.
    @author Stephen Allewell
  */
class PatternCanvas : public QObject
{
  Q_OBJECT
public:
/** PatternCanvas Constructor
    Creates an empty QVector of type StitchQueue, an empty QList of type BackStitch
    and an empty QList of type Knot.  All storage classes are set as AutoDelete
  */
  PatternCanvas();
/** PatternCanvas destructor
    Deletes the QVector of type StitchQueue, the QList of type BackStitch and
    the QList of type Knot if allocated.
  */
  ~PatternCanvas();

/** Clear the contents of the pattern
  */
  void clear();
/** Set the size of the canvas
    @param width Width of canvas in stitches
    @param height Height of canvas in stitches
  */
  void resize(int width,int height);
/** Extend the pattern to the left.
    @param cols The number of columns of stitches
  */
  void extendLeft(int cols);
/** Extend the pattern at the top.
    @param rows The number of rows of stitches
  */
  void extendTop(int rows);
/** Extend the pattern to the right.
    @param cols The number of columns of stitches
  */
  void extendRight(int cols);
/** Extend the pattern at the bottom.
    @param rows The number of rows of stitches
  */
  void extendBottom(int rows);
/** Center the pattern within the current size of the canvas
  */
  void centerPattern();
/** Return a QRect that bounds the cells containing the pattern.
    @return A QRect containing the top left, width and height of the pattern.
    The QRect will be invalid if the pattern is empty.
  */
  QRect patternBoundingRect();
/** Crop the canvas to a rectangle.
    If stitches exist outside the rectangle, they will be deleted. If a backstitch
    has one of its endpoints outside the rectangle, it will be deleted.
    @param rect A QRect defining the area to crop to.
  */
  void cropCanvasToRect(QRect rect);
/** check if the canvas has any stitches on it
    @return TRUE if stitches, backstitches or knots are not present, FALSE otherwise.
  */
  bool isEmpty();
/** Returns the width of the pattern
    @return width
  */
  int patternWidth();
/** Return the height of the pattern
    @return height
  */
  int patternHeight();
/** Returns a pointer to a StitchQueue for a specified location.
    @return Pointer to a StitchQueue, 0 is returned if no StitchQueue defined
    @param cell QPoint defining the cell entry to check
  */
  Stitch::Queue *stitchAt(QPoint c);
/** Add a stitch to a StitchQueue
    @return FALSE otherwise
    @param c QPoint referencing the cell
    @param t enumerated type of stitch
    @param i index of the color in the floss palette
    @return TRUE if a stitch added
  */
  bool addStitch(QPoint c,Stitch::Type t,int i);
/** Delete the stitchQueue
    @return TRUE if deleted, FALSE otherwise.
    @param c QPoint referencing the cell.
    @param t enumerated type of stitch as a mask.  Default value of Stitch::Delete
    matches all stitch type.
    @param i index of the color in the floss palette as a mask.  Default value or -1
    matches all colors.
  */
  bool deleteStitch(QPoint c,Stitch::Type t=Stitch::Delete, int i=-1);
/** Add a backstitch from start to end with the index color
    @param start QPoint defining the start position
    @param end QPoint defining the end position
    @param i index of the color in the palette
  */
  void addBackstitch(QPoint start, QPoint end, int i);
/** Delete a backstitch
    @param bs Pointer to the backstitch to be deleted
  */
  void deleteBackstitch(BackStitch *bs);
/** Add a french knot
    @param snap Snap point for position
    @param i index of the color in the palette
  */
  void addFrenchKnot(QPoint snap, int i);
/** Delete a french knot
    @param p Snap point of knot
    @param i Floss index if colorMask is true else -1
  */
  void deleteFrenchKnot(QPoint p, int i=-1);
/** Checks if the cell reference is valid for the current canvas.
    @return TRUE if cell is on the canvas, FALSE otherwise.
    @param c QPoint referencing the cell.
  */
  bool validateCell(QPoint c);
/** Checks if the snap point is valid for the current canvas.
    @return TRUE if snap point is on the canvas, FALSE otherwise.
    @param s QPoint referencing the snap point.
  */
  bool validateSnap(QPoint s);
/** Check the specified cell and return the color used, will return the
    color of the currently selected stitch type or the next available color
    @return index of color in the palette, or -1 if the cell is out of range or empty
    @param cell QPoint specifying cell to check
    @param stitchType The current StitchType selected, used to narrow search where
    there are multiple stitches in a cell.
  */
  int findColor(QPoint cell, Stitch::Type stitchType);
/** Replace one color with another.
    @param original The color to be replaced.
    @param replacement The color to replace it.
  */
  void replaceColor(int original,int replacement);
/** Find which colors have been used in the current pattern.
    @return A QValueList of type int specifying the color indexes used.
  */
  QValueList<int> usedColors();
/** Get a pointer to an iterator for the backstitches.
    @return An QListIterator of type BackStitch for the list of backstitches.
  */
  QPtrListIterator<BackStitch> *backstitches();
/** Get a pointer to an iterator for the french knots.
    @return An QListIterator of type Knot for the list of frech knots.
  */
  QPtrListIterator<Knot> *knots();
/** Write the contents of the canvas to the stream.
    @param s QDatastream to write to
  */
  void writeCanvas(QDataStream &s);
/** Read the contents of the  canvas from the stream.
    @return TRUE if the canvas was read correctly, FALSE if an error occured
    @param s QDataStream to read from.
    @param v File version number.
  */
  bool readKXStitchCanvas(QDataStream& s,int v);
  bool readPCStitch5Canvas(QDataStream& s);
/** Calculate the usage of floss for the current stitches
  */
  UsageMap calculateUsage();
  
private:
/** Return the index of a cell in the QVector.
    @return The index of the cell.
    @param c QPoint referencing the cell
  */
  uint index(QPoint c);

  int                         m_width;
  int                         m_height;
  QPtrVector<Stitch::Queue>*  m_pCanvas;
  QPtrList<Knot>*             m_pKnot;
  QPtrList<BackStitch>*       m_pBackstitch;
};

#endif
