/***************************************************************************
 *   Copyright (C) 2006 by Dusan Zatkovsky   *
 *   msk.conf@seznam.cz   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "krecordmydesktop.h"

#include <qlabel.h>

#include <kmainwindow.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <qfile.h>
#include <qtooltip.h>
#include <iostream>
#include <fstream>
#include <sstream>
#include <unistd.h>

#include <assert.h>

#include "delayStart.h"

/*!
    Basic preparation, creates popup menu, system tray, set icons etc...
*/
krecordmydesktop::krecordmydesktop()
    : KMainWindow( 0, "krecordmydesktop" ), recordProcess ( this ), delay ( -1 )
{
    // set the shell's ui resource file
    setXMLFile("krecordmydesktopui.rc");

    // create and show system tray
    DBG ( "Creating system tray icon" );
    p_tray = new KRecordMyDesktopSystemTray;// ( this, "system tray" );
    p_tray->setPixmap ( UserIcon("stopped") );

    DBG ( "Creating popup menu" );
    KPopupMenu* p_stmenu = p_tray->contextMenu();
    p_stmenu->insertItem ( UserIcon ( "start" ), "Start recording", this, SLOT ( slotStartRecording() ) );
    p_stmenu->insertItem ( UserIcon ( "encoding" ), "Start delayed recording", this, SLOT ( slotStartDelayedRecording() ) );
    p_stmenu->insertItem ( UserIcon ( "stop" ), "Stop recording", this, SLOT ( slotStopRecording() ) );
    p_stmenu->insertItem ( UserIcon ( "preferences" ), "Preferences", this, SLOT ( slotSetPreferences() ) );
    
    //  connect quit button
    connect ( p_tray, SIGNAL ( quitSelected () ), this, SLOT ( slotQuitApp ( void ) ) );
    
    //  connect delay start timer
    connect ( &delayTimer, SIGNAL ( timeout() ), this, SLOT ( slotDelayTimer ( void ) ) );
    
    ////  connect userInputWaitTimer
    //connect ( &userInputWaitTimer, SIGNAL ( timeout() ), this, SLOT ( slotEverythingDone ( void ) ) );
    
    //connect end of recording process signal to ending slot
    connect ( &recordProcess, SIGNAL (  processExited ( KProcess* ) ), this, SLOT ( slotEncodingDone ( KProcess* ) ) );
        
    // connect recordmydesktop process stdout to slot
    connect ( &recordProcess, SIGNAL ( receivedStdout ( KProcess *, char *, int ) ), this, SLOT ( slotRecordmydesktopStdout ( KProcess*, char*, int ) ) );
    connect ( &recordProcess, SIGNAL ( receivedStderr ( KProcess *, char *, int ) ), this, SLOT ( slotRecordmydesktopStdout ( KProcess*, char*, int ) ) );
    
    // modify permissions of created files to user=rw
    umask ( 0066 );
    
    // show systray
    p_tray->show();



}

krecordmydesktop::~krecordmydesktop()
{

    if ( p_tray ) delete p_tray;

}


/*!
    This slot is called when user click on "Start recording" item in system tray menu. It starts recordmydesktop as separate process.
    File will be recorded to temp file located in /tmp. After successfuly record, user will be asked for destination file name and temp
    file will be moved/renamed.
*/
void krecordmydesktop::slotStartRecording ( void ) {

    DBG ( "Start recording" );

    
    // must not be recording now
    if ( !recordProcess.isRunning() ) {

        
        
        // construct temp file name
        int cnt = 0;
        DBG ( "Constructing temp file name" );
        while ( true ) {
            std::ostringstream fn;
            fn << cfg.params.workingDir << "/.krecordmydesktop." << getuid() << "." << getpid() << "." << rand() << ".tmp";
            tempFileName = fn.str();
	    DBG ( "Trying " << tempFileName );
            if ( ! QFile ( tempFileName ).exists() ) break;
            if ( cnt++ > 1000 ) {
                DBG("Fatal error while creating temp file name");
                KMessageBox::error ( this, "Fatal error! Recording can't start.", "Error" );
                return;
            }
        }

        // construct command and parameters to execute
        recordProcess.clearArguments();
        {
            // recordmydesktop
            recordProcess << cfg.params.rmdPath;

            // sound parameters
            if ( !cfg.params.audioEnabled ) {
                 recordProcess << "--no-sound";
            }
            else {
                recordProcess << "-s_quality";
                recordProcess << QString ( "%1" ).arg ( ( unsigned ) ( cfg.params.audioQuality / 10 ) );
            }
            


            // video parameters
            recordProcess << "-v_quality";
            recordProcess << QString( "%1" ).arg ( ( unsigned ) ( cfg.params.videoQuality * 0.63 ) );

            recordProcess << "-fps";
            recordProcess << QString( "%1" ).arg ( cfg.params.fps );
            
            // on the fly encoding
            if ( cfg.params.encodeOnTheFly ) recordProcess << "--on-the-fly-encoding";
    
            // working dir ( there also will be temp output file saved )
            recordProcess << "-workdir";
            recordProcess << QString( "%1" ).arg ( cfg.params.workingDir );
            
            // other parameters
            if ( cfg.params.otherParams.length() ) {
                recordProcess << cfg.params.otherParams;
            }

            // output file name ( temp )
            recordProcess << "-o";
            recordProcess << QString( "%1" ).arg ( tempFileName );

        }   // end of command construction block


        // start recording
        DBG ( "Calling start()" );
        bool ret = recordProcess.start ( KProcess::NotifyOnExit, ( KProcess::Communication ) ( KProcess::Stdout | KProcess::Stderr ) );
        
        // set state
        state = recording;
        
        // debug
        /*
        DBG ( "p_recordProcess->isRunning()=" << (int)p_recordProcess->isRunning() << ", p_recordProcess->exitStatus()=" << p_recordProcess->exitStatus() );
        QValueList<QCString> l = p_recordProcess->args();
        QValueList<QCString>::iterator i = l.begin();
        while ( i != l.end() ) {
            DBG ( "Arg: " << (const char*)*i );
            i++;
        }
        */
        
        // error while starting command
        if ( !ret ) {

            DBG("Error while recording");

            // error message and return
            KMessageBox::error ( this, "Recording failed!", "Error" );
            return;
        }

        // change tray icon to "recording"
        p_tray->setPixmap ( UserIcon ( "start" ) );

        // set tray tooltip
        QToolTip::remove ( p_tray );
        QToolTip::add ( p_tray, "Now recording your work ..." );

    }

    // already recording, show information message and return
    else {
        DBG ( "Already recording!" );
        KMessageBox::error ( this, "Already recording", "Error" );
        return;
    }


}

/*!
    This slot starts delayed recording. At first, user is asked for time to wait.
    After that, app will wait selected time and start recording. The internal timer is used instead of
    recordmydesktop -delay parameter.
    Until delay timeouted, user could cancel timer by pressing stop button.
*/
void krecordmydesktop::slotStartDelayedRecording ( void ) {

    DBG ( "Delayed recording" );
    
    // if currently recording, stop processing and inform user
    if ( recordProcess.isRunning() ) {
        DBG ( "Already recording!" );
        KMessageBox::error ( this, "Already recording", "Error" );
        return;
    }

    // show delay setup dialog
    delayStartDialog dsd ( this );
    dsd.show();
    int ret = dsd.exec();

    // cancel pressed
    if ( !ret ) {
        DBG ( "Cancelled" );
        return;
    }

    // get user specified delay
    delay = dsd.getDelay();
    DBG ( "User specified " << delay << " seconds delay" );
    
    //  start delay timer to emit 1 sec ticks ( see connection for delayTimer )
    delayTimer.start ( 1000, false );
    
    //  set pixmap to inform that we are waiting specified delay
    p_tray->setPixmap ( UserIcon("encoding") );
    
    // set state
    state = countdown;
    

    DBG ( "Done, now will wait until timer timeouts" );

}

/*!
    This slot is called, when user selects "Stop recording" from content menu in systray. It asks user for output file name and
    calls SIGTERM for recordmydesktop. If user cancelled the save file dialog, recordmydesktop is killed by SIGKILL and temp file is removed.
*/

void krecordmydesktop::slotStopRecording ( void ) {

    DBG ( "Stop recording" );
    
    // cancel delay timer if is active
    if ( delayTimer.isActive() || delay != -1 ) {
        delayTimer.stop();
        delay = -1;
        QToolTip::remove ( p_tray );
        DBG ( "Timer cancelled" );
    }
    
    // if not running, do nothing
    if ( !recordProcess.isRunning() ) {
        DBG ( "Nothing to stop" );
        return;
    }

    // reset pixmap to default in systray
    p_tray->setPixmap ( UserIcon("stopped") );
    
    // set state
    state = stopped;
    
    // remove tooltip
    QToolTip::remove ( p_tray );

    // ask user for filename
    while ( true ) {

        // get file name
        outputFileName = KFileDialog::getSaveFileName ( defaults::defaultDirWhenOpenFileDialog );
        
        // if user cancelled saving, break
        if ( outputFileName.isNull() ) break;

        // add extension to output file name
        outputFileName += ".ogg";
        
        // if file doesn't exist, continue
        if ( !QFile::exists ( outputFileName ) ) break;
        
        // else ask user if he want to overwrite
        else {
            if ( KMessageBox::Yes == KMessageBox::warningYesNo ( this, QString ( "Do you like to overwrite existing file %1?" ).arg ( outputFileName ), "Overwrite file?" ) ) break;
        }
    
    }
    
    
    // terminate recording and start encoding ( if encoding on the fly wasn't specified )

    
    DBG ( "Output will be saved as " << ( outputFileName.isNull() ? "**cancel**" : outputFileName.ascii() ) );
    
    // if cancel
    if ( outputFileName.isNull() ) {

        DBG ( "Saving was cancelled" );
        
        // kill and cleanup
        if ( recordProcess.isRunning() ) {
            recordProcess.kill ( SIGKILL );
            recordProcess.wait();
        }
        
        cleanup();

    }
    
    // else not cancel, stop recording and start encoding
    else {
    
        // stop recording ( encoding will follow )
        DBG ( "Sending SIGTERM" );
        recordProcess.kill ( SIGTERM );
        DBG ( "Recording stopped, will follow encoding" );
            
        // set state
        state = encoding;
    
        // set systray icon to encoding
        p_tray->setPixmap ( UserIcon( "encoding" ) );
    
        // disable systray context menu while encoding
        DBG ( "Disabling systray until encode will ends" );
        p_tray->contextMenu()->setEnabled ( false );
    
    }
    

}

/*!
    This method removes temp file, restores systray and status and it puts application to default state.
*/
void krecordmydesktop::cleanup ( void ) {
    
        DBG ( "Cleaning up and restoring default state" );

        // remove temp file
        remove ( tempFileName.ascii() );

        // set state
        state = stopped;
        
        // re-enable context menu
        DBG ( "Re-enabling systray" );
        p_tray->contextMenu()->setEnabled ( true );
    
    
        // reset systray pixmap to default
        p_tray->setPixmap ( UserIcon ( "stopped" ) );
    
        // remove tooltip
        QToolTip::remove ( p_tray );
        
        DBG ( "Restoring default state" );
        
        return;

}

void krecordmydesktop::slotEncodingDone ( KProcess* p ) {

    // check if it is signal for me
    if ( p != &recordProcess ) {
        DBG ( "Signal not for me, ignoring ..." );
        return;
    }

    DBG ( "Debug: process exitstatus: " << p->exitStatus() << ", normalExit=" << ( int )p->normalExit() );

    // check exit status and inform user on error
    if ( p->exitStatus() != 0 || !p->normalExit() ) {

        DBG ( "Error while recording" );

        KMessageBox::error ( this, "Recording failed! Check debug output.", "Error" );

        // remove temp file
        if ( 0 != remove ( tempFileName.ascii() ) ) {
            KMessageBox::error ( this, "Can't remove temp file.", "Error" );
        }

    }
    
    // exit status ok, rename file and cleanup
    else {

        // move/rename temp file to destination file
        if ( 0 != rename ( tempFileName.ascii(), outputFileName.ascii() ) ) {
    
            // rename failed, make a copy and remove original ( when source and destinations are on different filesystems )
            unlink ( outputFileName.ascii() );
            std::ifstream i ( tempFileName.ascii() );
            if ( !i.good() ) {
                KMessageBox::error ( this, "Temp file not found.", "Error" );
            }
            else {
                std::ofstream o ( outputFileName.ascii() );
                o << i.rdbuf();
                if ( !i.good() || !o.good() ) {
                    KMessageBox::error ( this, "Can't save target file.", "Error" );
                }
            }
        }
    
        else {
    
            DBG ( "Done. Recorded video saved as " << outputFileName.ascii() );
    
        }


    }


    // cleanup
    cleanup();

}

void krecordmydesktop::slotQuitApp ( void ) {

    DBG ( "Quitting application" );
    
    
    // if recording process is running, ask user
    if ( recordProcess.isRunning() ) {

            if ( KMessageBox::Yes == KMessageBox::questionYesNo ( this, "Still recording your desktop. Would you like to cancel recording?", "Warning" ) ) {
            
                // disconnect recording process ( because after SIGKILL slotEncodingDone() will be called )
                DBG ( "Disconnection slotEncodingDone(), slotRecordmydesktopStdout()" );
                disconnect ( &recordProcess, SIGNAL (  processExited ( KProcess* ) ), this, SLOT ( slotEncodingDone ( KProcess* ) ) );
                disconnect ( &recordProcess, SIGNAL ( receivedStdout ( KProcess *, char *, int ) ), this, SLOT ( slotRecordmydesktopStdout ( KProcess*, char*, int ) ) );
                disconnect ( &recordProcess, SIGNAL ( receivedStderr ( KProcess *, char *, int ) ), this, SLOT ( slotRecordmydesktopStdout ( KProcess*, char*, int ) ) );

                // kill recording process
                recordProcess.kill ( SIGKILL );
                recordProcess.wait();
                
                // remove temp file
                remove ( tempFileName.ascii() );

            }
        }


    exit ( 0 );

}



void krecordmydesktop::slotSetPreferences ( void ) {

    DBG ( "Setting preferences" );
    
    configParameters tmp ( cfg );

    // show configuration dialog
    configDialog* p_pref = new configDialog ( this, &tmp );
    p_pref->show();
    int ret = p_pref->exec();
    // if accepted
    if ( ret == configDialog::Accepted ) {
        DBG ( "Config dialog accepted, storing value changes" );
        cfg.copy ( tmp );
    }
    // if cancelled, load last saved values
    else {
        DBG ( "Config dialog cancelled, ignoring value changes and loading last saved values" );
        cfg.load();
    }

    // delete configuration dialog
    delete p_pref;

}



void krecordmydesktop::slotRecordmydesktopStdout ( KProcess* p, char* p_data, int data_size ) {

    if ( p_data && data_size ) DBG ( "Received from stderr/cout: " << replaceInvalidChars ( QString ( p_data ) ) );

    // we care about recordmydesktop output only when encoding recorded video ( to get progress )
    if ( state != encoding ) return;

    // handle only recordmydesktop stdout/stderr
    if ( p != ( KProcess* ) &recordProcess ) return;
    
    // nothing received
    if ( !p_data || data_size <= 0 ) return;
    
    // convert data to string
    std::string data;
    data.assign ( p_data, data_size );
    //DBG ( "Received: \"" << data << "\"" << std::endl );
    
    // find [ and ]
    size_t pos1 = data.find ( "[" );
    size_t pos2 = data.find ( "%]" );
    if ( ( pos1 != std::string::npos && pos2 != std::string::npos ) && ( pos2 - pos1 - 1 > 0 ) ) {
        bool x;
        unsigned int progress = QString::fromAscii ( data.c_str() + pos1 + 1, pos2 - pos1 - 1 ) . toUInt ( &x, 10 );
        if ( x ) {
            // do not allow more than 100% ( recordmydesktop sometimes report more than 100%, don't know why )
            if ( progress > 100 ) progress = 100;
        
            // set tooltip that we are encoding
            //DBG ( "Setting enc tooltip" );
            QToolTip::add ( p_tray, QString ( "Now encoding recorded video, please wait ( %1 % ) ..." ) . arg ( progress ) );
        }
        
    }
        


}


/*!
    This slot is used to count ticks from delay timer. When user select delay record after 20 sec, this
    slot is called 20 times. It changes tooltip on system tray icon to tell, how much time remain to start
    recording. After last tick, timer is stopped and recording started.
*/
void krecordmydesktop::slotDelayTimer ( void ) {

    DBG ( "slotDelayTimer reached, remaining " << delay );

    // if delay timeouted, start recording
    if ( !delay-- ) {
        delayTimer.stop();
        delay = -1;
        slotStartRecording();
        return;
    }
    
    // else change systray tooltip and wait for next tick
    std::ostringstream oss;
    oss << "Waiting " << delay << " seconds to start recording ...";
    QToolTip::add ( p_tray, oss.str().c_str() );

}




#include "krecordmydesktop.moc"
