/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "stationlistmodel.h"
#include "stationlistwidget.h"
#include "settings_general.h"
#include <QAbstractItemModel>
#include <QApplication>
#include <QClipboard>
#include <QFile>
#include <QItemSelection>
#include <QMimeData>
#include <KProgressDialog>
#include <KLocale>
#define AND  &&
#define OR  ||
#define NOT  !
#define EQUAL  ==

stationlistModel::stationlistModel(stationlistWidget *parent,
                                   QWidget *mainWidget) : QAbstractTableModel(parent)
{
  // variables
  QStringList m_stringList;
  int i;  // using int because this is the return type of QStringList::size()

  // general setup
  m_mainWidget = mainWidget;
  view = parent;

  // setup data model
  m_stringList = settings_general::streamConfigFiles();
  for (i = 0; i < m_stringList.size(); ++i) {
    m_stationlist.append(new radioStation(this, m_mainWidget, m_stringList.at(i), i));
    helper_connectSignalsAndSlots(m_stationlist.last());
  };
  helper_writeStationListToGeneralSettings();  /* This is because when radioStation can't
  open the file, it creates a new one. Probaby this won't happen, but we have to make sure
  that the list in the general config file is actual. */

  // initialize attributes
  recalculate_numberOfActiveStreams_and_bandwidth(); // now possible (after making stationlist)
  connect(this, SIGNAL(rowsInserted(const QModelIndex &, int, int)),
          this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
  connect(this, SIGNAL(rowsRemoved(const QModelIndex &, int, int)),
          this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
  m_listOfStreamsOfWhichTheUserWantsThatTheyRip = 0;
}

stationlistModel::~stationlistModel()
{
  delete m_listOfStreamsOfWhichTheUserWantsThatTheyRip;
}

void stationlistModel::helper_connectSignalsAndSlots(radioStation *m_stream)
{
    connect(m_stream, SIGNAL(streamNameChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadStreamName(qlonglong)));
    connect(m_stream, SIGNAL(uriChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadUri(qlonglong)));
    connect(m_stream, SIGNAL(statusChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadStatus(qlonglong)));
    connect(m_stream, SIGNAL(errorChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadStatus(qlonglong)));
    connect(m_stream, SIGNAL(songChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadSong(qlonglong)));
    connect(m_stream, SIGNAL(dataSizeChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadDataSize(qlonglong)));
    connect(m_stream, SIGNAL(bitrateChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadBitrate(qlonglong)));
    connect(m_stream, SIGNAL(bitrateChanged(qlonglong, PropertyValue)),
            this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
    connect(m_stream, SIGNAL(metaIntervalChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadMetaInterval(qlonglong)));
    connect(m_stream, SIGNAL(metaInterval_milliSecondsChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadMetaInterval_milliSeconds(qlonglong)));
    connect(m_stream, SIGNAL(serverNameChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadServerName(qlonglong)));
    connect(m_stream, SIGNAL(relayPortChanged(qlonglong, PropertyValue)),
            this, SLOT(reloadRelayPort(qlonglong)));
    connect(m_stream, SIGNAL(running()),
            this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
    connect(m_stream, SIGNAL(not_running()),
            this, SLOT(recalculate_numberOfActiveStreams_and_bandwidth()));
}

int stationlistModel::rowCount (const QModelIndex & parent) const
{
  if (parent.isValid()) {  // no tree structure in our table-based model! Return a neutral value...
    return 0;
  } else {
    return m_stationlist.size();
  };
}

int stationlistModel::columnCount (const QModelIndex & parent) const
{
  if (parent.isValid()) {  // no tree structure in our table-based model! Return a neutral value...
    return 0;
  } else {
    return 10;
  };
}

QVariant stationlistModel::columnInfo(const columnInfoType type,
                                      const int column,
                                      const qint64 row,
                                      const quint64 value,
                                      QList<radioStation *> listOfStations) const
{
  QVariant temp;

  if (listOfStations.isEmpty()) {
    listOfStations = m_stationlist;
  };

  switch (type) {  // make sure that row is valid to prevent a memory access failure
    case columnHeaderTitle:
    case columnHeaderToolTip:
    case columnHeaderWhatsThis:
    case columnWidth:
    case setColumnWidth:
    case columnVisibility:
    case setColumnVisibility:
      break;  // These functions are known to not make use of the row.
    default:  // Else we break as a precaution we break up here.
      if ((row >= listOfStations.size()) OR (row < 0)) {
        return QVariant();
      };
  }
  /* IF data is requested that is NOT depending on the "row"
  *  THEN no check is needed.
  *  ELSE (=when data is depending on the "row") we check if it is a valid row.
  *  If not, we return a QVariant(). This is important, because the rest of this
  *  function body assumes that "row" is valid and uses it for QList::at(row) - and
  *  when "row" is invalid, this function leeds to a crash.
  *
  *  I use the "switch" systax because this looks clearer than an "if" construct.
  *  And I test for the cases where the row is NOT important - this way (when one
  *  day the enum gets more items that depend on the row - and we forget to change
  *  it here) at least we don't get a crash.
  *
  *  We don't test if "column" is valid, because the rest of the function is a
  *  switch-case, and when there nothing matches, an invalid QVariant is returned. */

  switch (column) {

    case 0:
    {
      switch (type) {
        case columnHeaderTitle:
          return i18nc("@title:column", "stream name");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return QVariant();
        case columnWidth:
          return settings_general::columnWidth_streamName();
        case setColumnWidth:
          settings_general::setColumnWidth_streamName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_streamName();
        case setColumnVisibility:
          settings_general::setColumnVisibility_streamName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->streamName());
          return temp;
      };
      break;
    }

    case 1:
    {
      switch (type) {
        case columnHeaderTitle:
          return i18nc("@title:column", "URL");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return QVariant();
        case columnWidth:
          return settings_general::columnWidth_uri();
        case setColumnWidth:
          settings_general::setColumnWidth_uri(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_uri();
        case setColumnVisibility:
          settings_general::setColumnVisibility_uri(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->uri());
          return temp;
      };
      break;
    }

    case 2:
    {
      switch (type) {
        case columnHeaderTitle:
          return i18nc("@title:column", "status");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return QVariant();
        case columnWidth:
          return settings_general::columnWidth_statusAndError();
        case setColumnWidth:
          settings_general::setColumnWidth_statusAndError(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_statusAndError();
        case setColumnVisibility:
          settings_general::setColumnVisibility_statusAndError(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          PropertyValue my_statusAndError;
          if (listOfStations.at(row)->error().formatedValue.isEmpty()) {
            my_statusAndError.formatedValue = listOfStations.at(row)->status().formatedValue;
            my_statusAndError.toolTip = listOfStations.at(row)->status().toolTip;
            my_statusAndError.whatsThis = listOfStations.at(row)->status().whatsThis;
          } else {
            my_statusAndError.toolTip = listOfStations.at(row)->error().toolTip;
            my_statusAndError.whatsThis = listOfStations.at(row)->error().whatsThis;
            if (listOfStations.at(row)->status().formatedValue.isEmpty()) {
              my_statusAndError.formatedValue = listOfStations.at(row)->error().formatedValue;
            } else {
              my_statusAndError.formatedValue = QString(i18nc(
                "@item:intable This produces the status message from the actual status (%1) and "
                  "the error message (%3). Both of them are guaranteed to be not empty. %2 is "
                  "replaced by an en-dash (Unicode U+2013). Use it or replace it by something "
                  "what looks nicer in your language.",
                "%1 %2 %3",
                listOfStations.at(row)->status().formatedValue,
                QString(QChar(0x2013)),
                listOfStations.at(row)->error().formatedValue));
            };
          };
          my_statusAndError.internalValue = listOfStations.at(row)->status().internalValue;
          if ((!(listOfStations.at(row)->error().type == PropertyValue::unset)) ||
              (listOfStations.at(row)->status().type == PropertyValue::error)) {
            my_statusAndError.type = PropertyValue::error;
          } else if (my_statusAndError.formatedValue.isEmpty()) {
            my_statusAndError.type = PropertyValue::unset;
          } else {
            my_statusAndError.type = PropertyValue::value;
          };
          temp.setValue<PropertyValue>(my_statusAndError);
          return temp;
      };
      break;
    }

    case 3:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column header of the column with the name of the actual track",
                        "track");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_song();
        case setColumnWidth:
          settings_general::setColumnWidth_song(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_song();
        case setColumnVisibility:
          settings_general::setColumnVisibility_song(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->song());
          return temp;
      };
      break;
    }

    case 4:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column", "track size");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_dataSize();
        case setColumnWidth:
          settings_general::setColumnWidth_dataSize(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_dataSize();
        case setColumnVisibility:
          settings_general::setColumnVisibility_dataSize(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->dataSize());
          return temp;
      };
      break;
    }

    case 5:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column", "bit rate");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_bitrate();
        case setColumnWidth:
          settings_general::setColumnWidth_bitrate(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_bitrate();
        case setColumnVisibility:
          settings_general::setColumnVisibility_bitrate(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->bitrate());
          return temp;
      };
      break;
    }

    case 6:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column The unit is KiB instead of kB. See "
                          "http://en.wikipedia.org/wiki/Binary_prefix for details.",
                        "meta data interval (KiB)");
        case columnHeaderToolTip:
          return  i18nc("@info:tooltip", "interval of meta data in bytes");
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_metaInterval();
        case setColumnWidth:
          settings_general::setColumnWidth_metaInterval(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_metaInterval();
        case setColumnVisibility:
          settings_general::setColumnVisibility_metaInterval(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->metaInterval());
          return temp;
      };
      break;
    }

    case 7:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column unit is millisecond", "meta data interval (ms)");
        case columnHeaderToolTip:
          return  i18nc("@info:tooltip", "interval of meta data in milliseconds");
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_metaInterval_milliSeconds();
        case setColumnWidth:
          settings_general::setColumnWidth_metaInterval_milliSeconds(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_metaInterval_milliSeconds();
        case setColumnVisibility:
          settings_general::setColumnVisibility_metaInterval_milliSeconds(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->metaInterval_milliSeconds());
          return temp;
      };
      break;
    }

    case 8:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column", "server name");
        case columnHeaderToolTip:
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_serverName();
        case setColumnWidth:
          settings_general::setColumnWidth_serverName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_serverName();
        case setColumnVisibility:
          settings_general::setColumnVisibility_serverName(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->serverName());
          return temp;
      };
      break;
    }

    case 9:
    {
      switch (type) {
        case columnHeaderTitle:
          return  i18nc("@title:column 'relay port' means 'the port that the relay server uses'",
                        "relay port");
        case columnHeaderToolTip:
          return  i18nc("@info:tooltip", "port of the relay server");
        case columnHeaderWhatsThis:
          return  QVariant();
        case columnWidth:
          return settings_general::columnWidth_relayPort();
        case setColumnWidth:
          settings_general::setColumnWidth_relayPort(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnVisibility:
          return settings_general::columnVisibility_relayPort();
        case setColumnVisibility:
          settings_general::setColumnVisibility_relayPort(value);
          settings_general::self()->writeConfig();
          return QVariant();
        case columnData:
          temp.setValue<PropertyValue>(listOfStations.at(row)->relayPort());
          return temp;
      };
      break;
    }

  /* No "default:"! This way we get a compiler warning when one day
  *  the enum type has more items and we forget to implement this here. */

  };
  return QVariant();  // Return an invalid QVariant if the given "column" was invalid.
}

void stationlistModel::recalculate_numberOfActiveStreams_and_bandwidth()
{
  // variables
  int i;
  int temp_numberOfActiveStreams = 0;
  quint64 temp_bandwidth = 0;

  // code

  // calculate new values
  for (i=0; i < m_stationlist.size(); ++i) {
    if (m_stationlist.at(i)->status().internalValue.value<ripping::statusType>() != ripping::idle) {
      temp_numberOfActiveStreams++;
      if (m_stationlist.at(i)->bitrate().type == PropertyValue::value) {
        temp_bandwidth = temp_bandwidth +
                             m_stationlist.at(i)->bitrate().internalValue.toULongLong();
      };
    };
  };

  // actualize properties if necessary
  if (internal_numberOfActiveStreams != temp_numberOfActiveStreams) {
    internal_numberOfActiveStreams = temp_numberOfActiveStreams;
    emit numberOfActiveStreamsChanged();
    if (internal_numberOfActiveStreams == 0) {
      emit numberOfActiveStreamsIsZero();
    };
  };
  if (internal_bandwidth != temp_bandwidth) {
    internal_bandwidth = temp_bandwidth;
    emit bandwidthChanged();
  };
}

void stationlistModel::reloadStreamName(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 0), index(stationIndex, 0));
}

void stationlistModel::reloadUri(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 1), index(stationIndex, 0));
}

void stationlistModel::reloadStatus(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 2), index(stationIndex, 2));
}

void stationlistModel::reloadSong(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 3), index(stationIndex, 3));
}

void stationlistModel::reloadDataSize(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 4), index(stationIndex, 4));
}

void stationlistModel::reloadBitrate(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 5), index(stationIndex, 5));
}

void stationlistModel::reloadMetaInterval(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 6), index(stationIndex, 6));
}

void stationlistModel::reloadMetaInterval_milliSeconds(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 7), index(stationIndex, 7));
}

void stationlistModel::reloadServerName(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 8), index(stationIndex, 8));
}

void stationlistModel::reloadRelayPort(const qlonglong stationIndex)
{
  emit dataChanged(index(stationIndex, 9), index(stationIndex, 9));
}

QVariant stationlistModel::headerData(int section, Qt::Orientation orientation, int role) const
{
     if (orientation EQUAL Qt::Vertical) {
       if (role == Qt::DisplayRole) {
         return (int(section+1));
       } else {
         return QVariant();
       };
     } else {
       switch (role) {
         case Qt::DisplayRole:
           return columnInfo(columnHeaderTitle, section);
         case Qt::ToolTipRole:
           return columnInfo(columnHeaderToolTip, section);
         case Qt::WhatsThisRole:
           return columnInfo(columnHeaderWhatsThis, section);
         default:
           return QVariant();  // other "display roles" aren't supported.
       };
     };
     return QVariant();  // This point should never be reached. It just exists
                         // to be sure to deliver a return value, also when
                         // I introduce new bugs in the code above.
}

QVariant stationlistModel::data(const QModelIndex &index, int role) const
{
  //variables
  QVariant returnValue;

  //code
  if ((index.model() EQUAL this) AND (!index.parent().isValid()) AND (index.isValid())) {
    switch (role) {
      case Qt::DisplayRole:
        returnValue =
          columnInfo(columnData, index.column(), index.row()).value<PropertyValue>().formatedValue;
        break;
      case Qt::ToolTipRole:
        returnValue =
          columnInfo(columnData, index.column(), index.row()).value<PropertyValue>().toolTip;
        break;
      case Qt::WhatsThisRole:
        returnValue =
          columnInfo(columnData, index.column(), index.row()).value<PropertyValue>().whatsThis;
        break;
      case Qt::TextAlignmentRole:
        returnValue = static_cast<int>(columnInfo(columnData, index.column(), index.row())
                                         .value<PropertyValue>().formatedValueAlignment);
        break;
      default: // other "display roles" aren't supported.
        returnValue = QVariant();
        break;
    };
  }

  //return
  return returnValue;
}

void stationlistModel::sort(int column, Qt::SortOrder order)
{
  if (m_stationlist.isEmpty() || column < 0 || column >= columnCount()) {  // nothing to do
    return;
  };

  // variables
  int i;
  int j;
  bool abort;
  PropertyValue temp;
  QList<radioStation *> sorted_list;
  QModelIndexList temp_indexList;
  QStringList selection;

  // code

  // save the selection
  temp_indexList = view->selectionModel()->selectedRows();
  foreach (const QModelIndex & tempModelIndex, temp_indexList) {
    selection.append(m_stationlist.at(tempModelIndex.row())->configFileName());
  };

  // generate a sorted list
  for (i = 0; i < m_stationlist.size(); ++i) {
    temp = columnInfo(columnData, column, i).value<PropertyValue>();
    j = 0;
    abort = (j >= sorted_list.size());
    while (!abort) {
      if (helper_firstIsAfterSecond(
            temp,
            columnInfo(columnData, column, j, -1, sorted_list).value<PropertyValue>(),
            order)) {
        ++j;
        abort = (j >= sorted_list.size());
      } else {
        abort = true;
      };
    };
    sorted_list.insert(j, m_stationlist.at(i));
  };

  // use the sorted list instead of the old one
  m_stationlist = sorted_list;

  // adapt index values and restore selection
  view->selectionModel()->clear();
  for (i = 0; i < m_stationlist.size(); ++i) {
    m_stationlist.at(i)->setIndex(i);
    if (selection.contains(m_stationlist.at(i)->configFileName())) {
      view->selectionModel()->select(
        index(i, 0),
        QItemSelectionModel::Select | QItemSelectionModel::Rows);
    };
  };

  // save the new sort order
  settings_general::setSortingAscendingly(order == Qt::AscendingOrder);
  settings_general::setSortByColumn(column);
  settings_general::self()->writeConfig();

  // emit the necessary signals
  emit dataChanged(index(0, 0),
                   index(m_stationlist.size() - 1, columnCount() - 1));

}

bool stationlistModel::helper_isNumericValue(const QVariant & value)
{
  // variables
  QMetaType::Type type;

  // code
  type = static_cast<QMetaType::Type>(value.type());
  return (type == QMetaType::Int ||
          type == QMetaType::UInt ||
          type == QMetaType::Double ||
          type == QMetaType::Long ||
          type == QMetaType::LongLong ||
          type == QMetaType::Short ||
          type == QMetaType::Char ||
          type == QMetaType::ULong ||
          type == QMetaType::ULongLong ||
          type == QMetaType::UShort ||
          type == QMetaType::UChar ||
          type == QMetaType::Float ||
          // I don't understand why QString() is necessary, but without it doesn't work.
          value.typeName() == QString("ripping::statusType"));
}

qlonglong stationlistModel::helper_toLongLong(const PropertyValue & value)
{
  // I don't understand why QString() is necessary, but without it doesn't work
  if (value.internalValue.typeName() == QString("ripping::statusType")) {
    return value.internalValue.value<ripping::statusType>();
  } else {
    return value.internalValue.toLongLong();
  };
}

bool stationlistModel::helper_firstIsAfterSecond(const PropertyValue & firstValue,
                                                 const PropertyValue & secondValue,
                                                 Qt::SortOrder order)
{
  // code
  if (order == Qt::AscendingOrder) {
      switch (firstValue.type) {
          case PropertyValue::unset:
              return (secondValue.type == PropertyValue::unset);
          case PropertyValue::value:
              if (secondValue.type == PropertyValue::unset) {
                  return true;
              } else if (secondValue.type == PropertyValue::error) {
                  return false;
              } else {
                  if (helper_isNumericValue(firstValue.internalValue) &&
                      helper_isNumericValue(secondValue.internalValue)) {
                      return helper_toLongLong(firstValue) >= helper_toLongLong(secondValue);
                  } else {
                      return QString::localeAwareCompare(firstValue.formatedValue,
                                                         secondValue.formatedValue) >= 0;
                  };
              };
          case PropertyValue::error:
              if (secondValue.type == PropertyValue::error) {
                  return QString::localeAwareCompare(firstValue.formatedValue,
                                                     secondValue.formatedValue) >= 0;
              } else {
                  return true;
              };
      };
  } else {
      switch (firstValue.type) {
          case PropertyValue::unset:
              return true;
          case PropertyValue::value:
              if (secondValue.type == PropertyValue::unset) {
                  return false;
              } else if (secondValue.type == PropertyValue::error) {
                  return true;
              } else {
                  if (helper_isNumericValue(firstValue.internalValue) &&
                      helper_isNumericValue(secondValue.internalValue)) {
                      return helper_toLongLong(firstValue) <= helper_toLongLong(secondValue);
                  } else {
                      return QString::localeAwareCompare(firstValue.formatedValue,
                                                         secondValue.formatedValue) <= 0;
                  };
              };
          case PropertyValue::error:
              if (secondValue.type == PropertyValue::error) {
                  return QString::localeAwareCompare(firstValue.formatedValue,
                                                     secondValue.formatedValue) <= 0;
              } else {
                  return false;
              };
      };
  };
  kDebug() << "Function hasn't calculated a return value. TRUE is returned to prevent a crash.";
  return true;
}

QModelIndex stationlistModel::addNewStation()
{
  // consts
  const int size_of_list = m_stationlist.size();

  // variables
  QPointer<radioStation> m_newStation;
  int exitCode;
  QString m_fileName;
  QModelIndex returnValue;

  // code
  m_newStation = new radioStation(this, m_mainWidget, QString(), size_of_list);
  exitCode = m_newStation->execSettingsDialog();
  if (exitCode == QDialog::Accepted) {  // insert the new station in our model
    beginInsertRows(QModelIndex(), size_of_list, size_of_list);
    m_stationlist.append(m_newStation);
    helper_connectSignalsAndSlots(m_stationlist.last());
    helper_writeStationListToGeneralSettings();
    endInsertRows();
    returnValue = index(size_of_list, 0);
  } else {  // delete the station object and the config file
    m_fileName = m_newStation->configFileName();
    delete m_newStation;
    // We must wait with constructing QFile until the station
    // object has been deleted (we shouldn't open a file that's yet/still open!).
    QFile theFile(m_fileName);
    theFile.remove();
  };
  return returnValue;
}

bool stationlistModel::removeRows(int row, int count, const QModelIndex & parent)
{
  // variables
  int i;
  QString filename;
  QFile the_file;

  // code
  if (count == 0) {
    return true;
  };
  // test if the index is valid (if not, QList.at() would crash!):
  if ((row >= 0) &&
      (count > 0) &&
      ((row + count - 1) < m_stationlist.size()) &&
      (!parent.isValid())) {
    beginRemoveRows(QModelIndex(), row, row + count - 1);
    for (i = row; i <= row+count-1; ++i) {
      // save the filename of the config file, we'll need it later to delete the file:
      filename = m_stationlist.at(i)->configFileName();
      delete m_stationlist.at(i);  // delete the radioStation object
      // remove the pointer to the (now deleted) radioStation object from our internal list:
      m_stationlist.removeAt(i);
      // delete the config file of the stream
      the_file.setFileName(filename);
      the_file.remove();
    };
    // synchronize the general config file with our internal list:
    helper_writeStationListToGeneralSettings();
    // correct the index for all following rows (which is now changed):
    for (i = row; i < m_stationlist.size(); ++i) {
      m_stationlist.at(i)->setIndex(i);
    };
    endRemoveRows();
    return true;
  } else {
    return false;
  };
}

inline void stationlistModel::helper_writeStationListToGeneralSettings()
{
  // variables
  int i;
  QStringList helper_stringlist;

  // code
  for (i = 0; i < m_stationlist.size(); ++i) {
    helper_stringlist.append(m_stationlist.at(i)->configFileName());
  };
  settings_general::setStreamConfigFiles(helper_stringlist);
  settings_general::self()->writeConfig();
}

void stationlistModel::record(const int index)
{
  // test if the index is valid (when it is not valid, we may not use QList.at()!):
  if ((index >= 0) AND (index < m_stationlist.size())) {
    m_stationlist.at(index)->startStreamripper();
  };
}

void stationlistModel::stopRecording(const int index)
{
  // test if the index is valid (when it is not valid, we may not use QList.at()!):
  if ((index >= 0) AND (index < m_stationlist.size())) {
    m_stationlist.at(index)->shutDown();
  };
}

void stationlistModel::showConfigDialog(const int index)
{
  // test if the index is valid (when it is not valid, we may not use QList.at()!):
  if ((index >= 0) AND (index < m_stationlist.size())) {
    m_stationlist.at(index)->showSettingsDialog();
  };
}

quint64 stationlistModel::bandwidth() const
{
  return internal_bandwidth;
}

int stationlistModel::numberOfActiveStreams() const
{
  return internal_numberOfActiveStreams;
}

void stationlistModel::rememberListOfStreamsWhichTheUserWantsToRip_ifNotYetDone()
{
  // variables
  int i;
  QStringList temp_stringList;

  // code
  if (m_listOfStreamsOfWhichTheUserWantsThatTheyRip == 0) {
    for (i=0; i < rowCount(); i++) {
      if (m_stationlist.at(i)->doesTheUserWantsThatTheStreamIsRipping()) {
        temp_stringList.append(m_stationlist.at(i)->configFileName());
      };
    };
    m_listOfStreamsOfWhichTheUserWantsThatTheyRip = new QStringList(temp_stringList);
  };
}

bool stationlistModel::queryClose()
{
  // variables
  int i;

  // code

  rememberListOfStreamsWhichTheUserWantsToRip_ifNotYetDone();

  // start shutdown for all processes
  if (numberOfActiveStreams() > 0) {
    for (i = 0; i < m_stationlist.size(); ++i) {
      m_stationlist.at(i)->shutDown();
    };
  };

  /* Now we test a second time if there are still running streams. (This could
  have changed because we have shutted down before.) If there remain running
  streams, we display a "busy" dialog until the shutdown is finished. */
  if (numberOfActiveStreams() > 0) {
    QPointer<KProgressDialog> m_dialog = new KProgressDialog(m_mainWidget,
      i18nc("@title:window", "Saving files..."),
      i18nc("@label", "Please wait while last files are saved..."));
    connect(this, SIGNAL(numberOfActiveStreamsIsZero()), m_dialog, SLOT(reject()));
    // min AND max = 0 means: show "busy" indicator instead of progress:
    m_dialog->progressBar()->setMinimum(0);
    m_dialog->progressBar()->setMaximum(0);
    m_dialog->setModal(true);
    m_dialog->setAutoClose(false);
    m_dialog->setMinimumDuration(0);
    // The following line hasn't any effect because of a bug in KProgressDialog.
    // Is fixed in KDE 4.2 (and maybe yet in 4.1.4).
    m_dialog->setButtonText(i18nc("@action:button", "Quit without saving"));
    m_dialog->exec();
    delete m_dialog;
  };

  return true;
}

void stationlistModel::saveProperties(KConfigGroup & m_configGroup)
{
  rememberListOfStreamsWhichTheUserWantsToRip_ifNotYetDone();
  m_configGroup.writePathEntry("streamsToResume",
                                * m_listOfStreamsOfWhichTheUserWantsThatTheyRip);
  // No need to save the changes with sync() - KMainWindow and friends take care of this.
}

void stationlistModel::readProperties(const KConfigGroup & m_configGroup)
{
  // variables
  QStringList temp_stringList;
  int i;

  // code
  temp_stringList = m_configGroup.readPathEntry("streamsToResume", temp_stringList);
  for (i = 0; i < rowCount(); i++) {
    if (temp_stringList.contains(m_stationlist.at(i)->configFileName())) {
      m_stationlist.at(i)->startStreamripper();
    };
  };
}

QList<QUrl> stationlistModel::helper_convertToUrl(const QMimeData *data)
{
  // variables
  QList<QUrl> returnValue;

  // code
  if (data->hasUrls()) {
    returnValue = data->urls();
  } else if (data->hasText()) {
    QStringList temp = data->text().split(QRegExp("\\s"), QString::SkipEmptyParts);
    foreach (const QString & singleString, temp) {
      returnValue.append(singleString);
    };
  };
  int i = 0;
  while (i < returnValue.size()) {
    if (returnValue.at(i).isValid() && (!returnValue.at(i).isEmpty())) {
      ++i;
    } else {
      returnValue.removeAt(i);
    };
  };

  return returnValue;
}

void stationlistModel::paste()
{
  dropMimeData(QApplication::clipboard()->mimeData(),
               Qt::CopyAction,
               0,
               0,
               QModelIndex());
}

void stationlistModel::pasteSelection()
{
  dropMimeData(QApplication::clipboard()->mimeData(QClipboard::Selection),
               Qt::CopyAction,
               0,
               0,
               QModelIndex());
}

bool stationlistModel::dropMimeData(const QMimeData *data,
                                    Qt::DropAction action,
                                    int,
                                    int,
                                    const QModelIndex &)
{
  if (action == Qt::IgnoreAction) {
    return true;
  };

  if (!(action == Qt::CopyAction)) {
    return false;
  };

  QList<QUrl> urlList = helper_convertToUrl(data);

  if (urlList.isEmpty()) {
    return false;
  } else {
    // consts
    const int size_of_list = m_stationlist.size();
    // variables
    QPointer<radioStation> m_newStation;
    // code
    beginInsertRows(QModelIndex(), size_of_list, size_of_list + urlList.size() - 1);
    for (int i = 0; i < urlList.size(); ++i) {
      m_newStation = new radioStation(this, m_mainWidget, QString(), size_of_list + i);
      m_newStation->setServerUri(urlList.at(i));
      m_stationlist.append(m_newStation);
      helper_connectSignalsAndSlots(m_stationlist.last());
    };
    helper_writeStationListToGeneralSettings();
    endInsertRows();
    if (!view.isNull()) {
      view->selectionModel()->clear();
      view->selectionModel()->select(
        QItemSelection(index(size_of_list, 0),
                       index(size_of_list + urlList.size() - 1, columnCount() - 1)),
        QItemSelectionModel::SelectCurrent);
    };
    return true;
  };

  return false;  // safety fallback
}

Qt::ItemFlags stationlistModel::flags(const QModelIndex & index) const
{
  return (Qt::ItemIsDropEnabled | QAbstractItemModel::flags(index));
}

QStringList stationlistModel::mimeTypes () const
{
  QStringList value;
  value << "text/plain" << "text/uri-list";
  return value;
}

void stationlistModel::enableListeningIn()
{
  QModelIndexList actuallySelectedRows = view->selectionModel()->selectedRows();
  if (actuallySelectedRows.size() == 1) {
    listen.setStation(m_stationlist.at(actuallySelectedRows.at(0).row()));
  } else {
    listen.setStation(0);
  };
}

void stationlistModel::disableListeningIn()
{
  listen.setStation(0);
}
