/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "radiostation.h"

#include <QFileInfo>
#include <KStandardDirs>
#include <KConfig>
#include <KTemporaryFile>
#include "settings_general.h"

#define AND  &&
#define OR  ||
#define NOT  !
#define EQUAL  ==

radioStation::radioStation(QObject *parent,
                           QWidget *mainWidget,
                           const QString & configFileName) : ripping(parent)
{
  m_mainWidget = mainWidget;
  helper_setupConfigSystem(configFileName);
  helper_write_properties_from_file_to_class_ripping();
}

radioStation::radioStation(QObject *parent,
                           QWidget *mainWidget,
                           const QString & configFileName,
                           qlonglong index) : ripping(parent)
{
  setIndex(index);
  m_mainWidget = mainWidget;
  helper_setupConfigSystem(configFileName);
  helper_write_properties_from_file_to_class_ripping();
}

void radioStation::helper_write_properties_from_file_to_class_ripping()
{
  ripping::setBitrate(config_skeleton->info_bitrate());
  ripping::setMetaInterval(config_skeleton->info_metaInterval());
  ripping::setServerName(config_skeleton->info_serverName());
  ripping::setStreamName(config_skeleton->info_streamName());
  emit uriChanged(index(), formatedUri(config_skeleton->serverUri()));
}

radioStation::~radioStation()
{
  delete config_skeleton;  /*  We have to delete the object on which config_skeleton
    points to - _befor_ deleting the object shared_config points on because the first
    makes use of the second. As the object shared_config points on is deleted when it's
    pointer (shared_config is a special pointer managing this) is deleted (and that's
    done - as the pointer is a member of this class - after the descturctor call) we
    can make sure by deleting config_skeleton explicitly here that the object config_skeleton
    points on is deleted _befor_ the object shared_config points on. I don't want to
    relay on the QObject-parentship-system because I don't know _when_ exactly the child
    object is deleted, and this way it is sure. */
  delete settingsDialog;  /* Maybe there exists still a settingsDialog. (Because settingsDialog
    deletes itself after closing, this is only the case if the settingsDialog is still open.)
    Because the parent isn't this object, but m_mainWidget, the settingsDialog will not be
    automatically deleted by Qt and we have to do it manually. */
}

inline void radioStation::helper_setupConfigSystem(const QString & configFileName)
{
  // variables
  QFileInfo info;

  // code
  info.setFile(configFileName);
  if (info.exists() && info.isFile() && info.isReadable() && info.isWritable()) {
    internal_configFileName = configFileName;
  } else {  // create a new file
    KTemporaryFile newConfigFile;
    // Maybe directories are missing in our desired path - locateLocal() will
    // create them if nesessary...
    newConfigFile.setPrefix(KStandardDirs::locateLocal("appdata", "stream_"));
    newConfigFile.setSuffix("_rc");
    newConfigFile.open();
    newConfigFile.setAutoRemove(false);
    internal_configFileName = newConfigFile.fileName();
    newConfigFile.close();
  };

  // setup configuration system
  shared_config = KSharedConfig::openConfig(internal_configFileName, KConfig::SimpleConfig);
  // We don't use a full-featured config system (with kiosk system and system-wide settings),
  // but we only read our streamConfig file.
  config_skeleton = new settings_stream(shared_config); // no constructor where I
  // can pass a parent
  /* config_skeleton->setParent(this) would also be a bad idea;
  So here I don't use QObject's parent-child-system but delete
  the object in the destructor. See there for further explications. */

  connect(config_skeleton,
          SIGNAL(configChanged()),
          this,
          SLOT(helper_write_properties_from_file_to_class_ripping()));
}

QString radioStation::configFileName() const
{
  return internal_configFileName;
}

void radioStation::setStreamName(const QString & streamName)
{
  if (config_skeleton->info_streamName() != streamName) {
    config_skeleton->setInfo_streamName(streamName);
    config_skeleton->writeConfig();
  };
}

void radioStation::setServerName(const QString & serverName)
{
  if (config_skeleton->info_serverName() != serverName) {
    config_skeleton->setInfo_serverName(serverName);
    config_skeleton->writeConfig();
  };
}

void radioStation::setBitrate(const qint64 bitrate)
{
  // config_skeleton makes sure that no invalid value is stored
  if (config_skeleton->info_bitrate() != bitrate) {
    config_skeleton->setInfo_bitrate(bitrate);
    config_skeleton->writeConfig();
  };
}

void radioStation::setMetaInterval(const qint64 metaInterval)
{
  // config_skeleton makes sure that no value < -1 is stored...
  if (config_skeleton->info_metaInterval() != metaInterval) {
    config_skeleton->setInfo_metaInterval(metaInterval);
    config_skeleton->writeConfig();
  };
}

QString radioStation::serverUri() const
{
  return config_skeleton->serverUri();
}

QStringList radioStation::parameterList() const
{
  // variables
  QStringList parameters;

  //code
  // calculate all parameters
  parameters += ripping::parameterList();

  parameters.append(QString("-u"));
  parameters.append(config_skeleton->advanced_userAgentString());

  parameters.append(QString("--xs_offset=%1").
                     arg(config_skeleton->offset()));

  /* The following code produces a search window which length corresponds exactly
  to the value in the settings. When the value is odd, the last half of the window
  is 1 ms longer. */
  parameters.append(QString("--xs_search_window=%1:%2").
                     arg(config_skeleton->searchWindow() / 2).
                     arg((config_skeleton->searchWindow() / 2) +
                          (config_skeleton->searchWindow() % 2)));

  parameters.append(QString("--xs_silence_length=%1").
                     arg(config_skeleton->silenceWindow()));

  if (config_skeleton->writeSingleFile()) {
    parameters.append(QString("-a"));    //TODO file name schemes for single files: append here
  };

  if (config_skeleton->writeSplittedFiles()) {
    //TODO file name schemes splitted files
    parameters.append(QString("--xs_padding=%1:%2").
                       arg(config_skeleton->splittedFilesPostpad()).
                       arg(config_skeleton->splittedFilesPrepad()));
  } else {
    parameters.append(QString("-A"));
  };

  parameters.append(QString("-k"));
  parameters.append(QString::number(config_skeleton->skipFirstXTracks()));

  if (config_skeleton->advanced_useTimeout()) {
    parameters.append(QString("-m"));
    parameters.append(QString::number(config_skeleton->advanced_timeoutAfterXSeconds()));
  }

  if (config_skeleton->truncateDuplicatesInIncomplete()) {
    parameters.append(QString("-T"));
  };

  return parameters;
}

int radioStation::helper_displaySettingsDialog(const bool returnImmediately)
{
  // variables
  int exitCode;

  // code
  /* If the dialog object yet exists, we can display it directly and return.
  (Should normally not happen, because we delete it after the use. And the dialog
  is modal, so the user can't open a second dialog. We do this here just to be
  absolutly sure.) We are using name of the config file as name of the dialog because
  this name is unique! */
  if (settingsDialog.isNull()) {
    settingsDialog = new settings_stream_dialog(m_mainWidget,
                                                internal_configFileName,
                                                config_skeleton);
    connect(settingsDialog,
             SIGNAL(finished()),
             settingsDialog,
             SLOT(deleteLater()));
  };
  if (returnImmediately) {
    settingsDialog->show();
    exitCode = KDialog::Accepted;
  } else {
    exitCode = settingsDialog->exec();
  };
  return exitCode;
}

void radioStation::showSettingsDialog()
{
  helper_displaySettingsDialog(true);
}

int radioStation::execSettingsDialog()
{
  return helper_displaySettingsDialog(false);
}

void radioStation::setServerUri(const QUrl & uri)
{
  if (QUrl(config_skeleton->serverUri()) != uri) {
    // delete old metadata
    setBitrate(default_value_of_bitrate());
    setMetaInterval(default_value_of_metaInterval());
    setServerName(default_value_of_serverName());
    setStreamName(default_value_of_streamName());

    // save new URI
    config_skeleton->setServerUri(uri.toString());
    // notify signal is emitted automatically when config file changes...
    config_skeleton->writeConfig();

    // get new metadata
    updateMetaData();
  };
}

PropertyValue radioStation::uri() const
{
  return formatedUri(config_skeleton->serverUri());
}

PropertyValue radioStation::formatedUri(const QString & theUri)
{
    // variables
    PropertyValue temp_uri;

    // code
    temp_uri.internalValue = theUri;
    temp_uri.formatedValue = theUri;
    if (theUri.isEmpty()) {
      temp_uri.type = PropertyValue::unset;
    } else {
      temp_uri.type = PropertyValue::value;
    };
    // there's never a toolTip or whatsThis, so no need to clear them.
    return temp_uri;
}

QString radioStation::workingDirectory() const
{
  return settings_general::saveDirectory();
}

void radioStation::updateMetaData()
{
  delete infoCatcher;
  infoCatcher = new get_stream_info(this);
  infoCatcher->setServerUri(serverUri());
  connect(infoCatcher,
          SIGNAL(bitrateChanged(qlonglong, PropertyValue)),
          this,
          SLOT(setBitrate(qlonglong, PropertyValue)));
  connect(infoCatcher,
          SIGNAL(metaIntervalChanged(qlonglong, PropertyValue)),
          this,
          SLOT(setMetaInterval(qlonglong, PropertyValue)));
  connect(infoCatcher,
          SIGNAL(serverNameChanged(qlonglong, PropertyValue)),
          this,
          SLOT(setServerName(qlonglong, PropertyValue)));
  connect(infoCatcher,
          SIGNAL(streamNameChanged(qlonglong, PropertyValue)),
          this,
          SLOT(setStreamName(qlonglong, PropertyValue)));
  infoCatcher->startStreamripper();
}

void radioStation::setBitrate(const qlonglong, const PropertyValue & newBitrate)
{
  setBitrate(newBitrate.internalValue.toLongLong());
}

void radioStation::setMetaInterval(const qlonglong,
                                   const PropertyValue & newMetaInterval)
{
  setMetaInterval(newMetaInterval.internalValue.toLongLong());
}

void radioStation::setServerName(const qlonglong, const PropertyValue & newServerName)
{
  setServerName(newServerName.internalValue.toString());
}

void radioStation::setStreamName(const qlonglong, const PropertyValue & newStreamName)
{
  setStreamName(newStreamName.internalValue.toString());
}
