/*
    Copyright (C) 2008-2009  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef STREAMDIRECTORYENTRY_H
#define STREAMDIRECTORYENTRY_H

#include <QString>
#include <QList>
#include <QVariant>
#include <KUrl>

/** \brief Abstract data type representing an entry in a tree structure
*
* You can use it like ordinary data types. This class doesn't provide much intellegence.
* You should reimplement #data() when deriving from this class.
*
* Objects of this class have its data orginized in columns. Furthermore, they can have children.
* The children are organized in rows; each of them is accessible through its row number.
*
* This class is designed for internal use and serves to providing a stream directory.
*
* \sa #streamDirectoryModel
* \note Objects of this class <i>provide information</i> about the actually existing tree
* structure, but they <i>can't create or modify</i> the tree structure. This is the unique
* responsibility of the code that creates the objects. */
class streamDirectoryEntry
{

  public:
    /** The constructor. */
    streamDirectoryEntry();
    /** The destructor.
    *
    * Will delete all child objects. */
    virtual ~streamDirectoryEntry();
    /** Makes an object become a child of \e this object.
    * @param child A pointer to the new child.
    * \warning Don't call this function with a child that has yet a parent! */
    void appendChild(streamDirectoryEntry *child);
    /** \returns whether this entry can fetch missing children through #fetchMore()
    *
    * The default implementation returns \e false. Reimplement this to provide custom behaviour.
    * \sa #fetchMore() */
    virtual bool canFetchMore() const;
    /** \returns A pointer to the child entry at the specified row (or \e NULL if the row doesn't
    * exist).
    * @param row The row number for which the pointer is requested. */
    streamDirectoryEntry *child(int row) const;
    /** \returns The number of children of this object. */
    int childCount() const;
    /** \returns The data of this entry at the specified column, being formated.
    * \note This default implementation returns always an invalid QVariant. When deriving from
    * this class you should reimplement this function to provide a useful behaviour. */
    virtual QVariant data(const int column) const;
    /** Fetchs missing children entries.
    *
    * The default implementation does nothing. Reimplement it for custom behavior.
    * \sa #canFetchMore() */
    virtual void fetchMore();
    /** \returns whether the item has children */
    virtual bool hasChildren() const = 0;
    /** \returns #entryName */
    QString name() const;
    /** \returns A pointer to the parent of this object (or \e NULL if it has no parent). */
    streamDirectoryEntry *parent() const;
    /** \returns The data of this entry at the specified column, but without being formated.
    * \note This default implementation returns #data(). Reimplement it in derived classes
    * to make possible proper sorting of data. */
    virtual QVariant rawData(const int column) const;
    /** \returns The row number that this object has within it's parent (or 0 if it
    * has no parent). */
    int row() const;
    /** Sets #entryName.
    * @param name The new value */
    void setName(const QString & name);
    /** Sets #entryValue.
    * @param value The new value */
    void setValue(const KUrl & value);
    /** \returns #entryValue */
    KUrl value() const;

  private:
    /** The list of child items. */
    QList<streamDirectoryEntry *> childItems;
    /** The name of this entry.
    *
    * In this class, this member is useless. However, when deriving from this class, you should
    * return this value for column 0.
    *
    * \sa #name()
    * \sa #setName() */
    QString entryName;
    /** The value of this entry. This is the URL that is used for Drag and Drop.
    *
    * In this class, this member is useless. The implementation is up to the derived classes.
    *
    * \sa #value()
    * \sa #setValue() */
    KUrl entryValue;
    /** A pointer to the parent of this object (or \e NULL if this object doesn't have a
    * parent). */
    streamDirectoryEntry *parentItem;
    /** A pointer to \e this. */
    streamDirectoryEntry *self;
};

#endif
