/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "disc.h"
#include "project.h"

#include <qdom.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>

#include <kmessagebox.h>
#include <klocale.h>
#include <kio/netaccess.h>
#include <kurl.h>

#include "../config.h"



namespace KoverArtist
{

Disc::Disc()
:mTitle()
,mEntries()
,mSnapShots()
{
}


Disc::Disc(const Disc& o)
:mTitle(o.mTitle)
,mEntries(o.mEntries)
,mSnapShots(o.mSnapShots)
{
}


Disc::~Disc()
{
}


Disc& Disc::operator=(const Disc& o)
{
   if (this!=&o)
   {
      mTitle = o.mTitle;
      mEntries = o.mEntries;
      mSnapShots = o.mSnapShots;
   }
   return *this;
}


void Disc::setEntry(int idx, const QString& str)
{
   mEntries[idx] = str;
}


void Disc::clear()
{
   mTitle = "";
   mEntries.clear();
   mSnapShots.clear();
}


bool Disc::isEmpty() const
{
   return mTitle.isEmpty() && mEntries.isEmpty();
}


void Disc::toDom(QDomDocument& aDoc, QDomElement& aElem) const
{
   QStringList::const_iterator it;
   QDomElement ent;

   QDomElement title = aDoc.createElement("Title");
   title.appendChild(aDoc.createTextNode(mTitle));
   aElem.appendChild(title);

   for (it=mEntries.begin(); it!=mEntries.end(); ++it)
   {
      ent = aDoc.createElement("Entry");
      ent.appendChild(aDoc.createTextNode(*it));
      aElem.appendChild(ent);
   }

   for (it=mSnapShots.begin(); it!=mSnapShots.end(); ++it)
   {
      ent = aDoc.createElement("Snapshot");
      ent.appendChild(aDoc.createTextNode(*it));
      aElem.appendChild(ent);
   }
}


bool Disc::fromDom(const QDomElement& aDisc)
{
   QDomElement elem;
   QString nm;

   clear();

   for (elem=aDisc.firstChild().toElement(); !elem.isNull();
        elem=elem.nextSibling().toElement())
   {
      nm = elem.tagName();
      if (nm=="Title") setTitle(elem.text());
      else if (nm=="Title") mSnapShots.append(elem.text());
      else if (nm=="Entry") mEntries.append(elem.text());
      else if (nm=="Snapshot") mSnapShots.append(elem.text());
      else
      {
         int rc = KMessageBox::warningContinueCancel(0,
                  i18n("Unknown XML element: %1").arg("Disc/"+nm));
         if (rc!=KMessageBox::Continue) return false;
      }
   }

   return true;
}


bool Disc::load(const KURL& aUrl, QWidget *aWindow)
{
   QString fileName;
   bool ok = false;
   bool isLocal = aUrl.isLocalFile();

   if (isLocal) fileName = aUrl.path();
   else
   {
      if (!KIO::NetAccess::download(aUrl, fileName, aWindow))
      {
         KMessageBox::error(aWindow, KIO::NetAccess::lastErrorString());
         return false;
      }
   }

   QFileInfo fi(fileName);
   if (fi.exists() && fi.isReadable() && !fi.isDir())
   {
      ok = loadFile(fileName);
   }
   else
   {
      KMessageBox::error(aWindow, i18n("Cannot open disc %1\n"
         "File does not exist or is not readable").arg(aUrl.prettyURL()));
   }

   if (!isLocal)
      KIO::NetAccess::removeTempFile(fileName);

   return ok;
}


bool Disc::loadFile(const QString& aFileName)
{
   QString ext(Project::fileExt(aFileName));

   if (ext=="txt") return loadFileText(aFileName);

   QDomDocument doc;
   QFile in(aFileName);

   if (!in.open(IO_ReadOnly)) return false;
   if (!doc.setContent(&in)) return false;

   QDomDocumentType dt = doc.doctype();
   QString type = dt.name();

   if (type=="kmf_project") return loadFileKmf(aFileName, doc);
   else if (type=="koverartist_disc") return loadFileKoad(aFileName, doc);
   else if (type=="kinoartist_project") return loadFileKiap(aFileName, doc);

   KMessageBox::error(0, i18n("File %1 has unsupported document type %2")
      .arg(aFileName).arg(dt.name()));

   return true;
}


bool Disc::loadFileText(const QString& aFileName)
{
   QFile fin(aFileName);
   QTextStream in(&fin);
   QString line;

   if (!fin.open(IO_ReadOnly)) return false;

   clear();

   while (!in.atEnd())
   {
      line = in.readLine().stripWhiteSpace();
      if (!line.isEmpty()) mEntries.append(line);
   }

   return true;
}


bool Disc::loadFileKoad(const QString& aFileName, QDomDocument& aDoc)
{
   QDomElement root = aDoc.documentElement();
   QString nm, str, type;

   QDomElement elem=root.firstChild().toElement();
   if (elem.tagName()!="Disc")
   {
      KMessageBox::error(0, i18n("%1: expected XML element %2")
         .arg(aFileName).arg("Disc"));
      return false;
   }

   clear();
   return fromDom(elem);
}


bool Disc::loadFileKiap(const QString& aFileName, QDomDocument& aDoc)
{
   QDomElement root = aDoc.documentElement();
   QDomElement elem, elem2, elem3;
   QString nm, nm3, str;

   if (root.tagName()!="kinoartist_project")
   {
      KMessageBox::error(0, i18n("%1: expected XML element %2")
         .arg(aFileName).arg("kinoartist_project"));
      return false;
   }

   clear();

   for (elem=root.firstChild().toElement(); !elem.isNull();
        elem=elem.nextSibling().toElement())
   {
      nm = elem.tagName();
      if (nm=="Title") mTitle = elem.text();
      else if (nm=="Media")
      {
	 for (elem2=elem.firstChild().toElement(); !elem2.isNull();
	       elem2=elem2.nextSibling().toElement())
	 {
	    if (elem2.tagName()=="Medium")
	    {
	       for (elem3=elem2.firstChild().toElement(); !elem3.isNull();
		     elem3=elem3.nextSibling().toElement())
	       {
	          nm3 = elem3.tagName();
		  if (nm3=="Title")
		     mEntries.append(elem3.text());
		  else if (nm3=="Snapshot" && elem3.attribute("type")=="main")
		  {
		     str = elem3.text();
		     if (str.left(6)=="file:/") str = str.mid(6);
                     if (!str.isEmpty()) mSnapShots.push_back(str);
                  }
	       }
	    }
	 }
      }
   }

   return true;
}


bool Disc::loadFileKmf(const QString& aFileName, QDomDocument& aDoc)
{
   QDomElement root = aDoc.documentElement();
   QDomElement elem, elem2;
   QString nm, str, type;

   if (root.tagName()!="project")
   {
      KMessageBox::error(0, i18n("%1: expected XML element %2")
         .arg(aFileName).arg("project"));
      return false;
   }

   clear();
   mTitle = root.attribute("title");

   for (elem=root.firstChild().toElement(); !elem.isNull();
        elem=elem.nextSibling().toElement())
   {
      nm = elem.tagName();
      if (nm=="media")
      {
         type = elem.attribute("object");
         if (type=="video")
         {
            for (elem2=elem.firstChild().toElement(); !elem2.isNull();
                 elem2=elem2.nextSibling().toElement())
	    {
	       nm = elem2.tagName();
               if (nm=="video")
               {
                  str = elem2.attribute("title");
                  if (!str.isEmpty()) mEntries.append(str);
                  str = elem2.attribute("custom_preview");
                  if (str.left(6)=="file:/") str = str.mid(6);
                  if (!str.isEmpty()) mSnapShots.append(str);
	       }
	    }
	 }
      }
   }

   return true;
}


bool Disc::saveFile(const QString& aFileName) const
{
   QString ext(Project::fileExt(aFileName));

   if (ext=="txt") return saveFileText(aFileName);
   if (ext=="koad") return saveFileKoad(aFileName);

   KMessageBox::error(0, i18n("Unsupported file format %1").arg(ext));
   return false;
}


bool Disc::saveFileKoad(const QString& aFileName) const
{
   QDomDocument doc("koverartist_disc");
   QDomElement elem;

   // Root
   QDomElement root = doc.createElement("KoverArtistDisc");
   root.setAttribute("creator", PACKAGE);
   root.setAttribute("version", VERSION);
   doc.appendChild(root);

   // The disc
   elem = doc.createElement("Disc");
   toDom(doc, elem);
   root.appendChild(elem);

   // Save the DOM
   QFile out(aFileName);
   if (!out.open(IO_WriteOnly)) return false;

   QCString hdr("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
   out.writeBlock(hdr, strlen(hdr));

   QCString domStr = doc.toString().utf8();
   int domStrLen = strlen(domStr);
   return (out.writeBlock(domStr, domStrLen)==domStrLen);
}


bool Disc::saveFileText(const QString& aFileName) const
{
   QStringList::const_iterator it;
   QDomElement ent;

   QFile fout(aFileName);
   if (!fout.open(IO_WriteOnly)) return false;
   QTextStream out(&fout);

   for (it=mEntries.begin(); it!=mEntries.end(); ++it)
      out<<*it<<endl;

   return true;
}


bool Disc::canLoad(const QString& aFileName)
{
   QString ext = Project::fileExt(aFileName);

   if (ext=="koad")
      return true;

   return false;
}



} //namespace
