/*
   Adapted for KoverArtist by Stefan Taferner <taferner@kde.org>
   Original version Copyright (C) 2001-2003 by Adrian Reber <adrian@lisas.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#ifndef koverartist_cdinfo_h
#define koverartist_cdinfo_h

#include <qstring.h>
#include <qvaluevector.h>

// CDDB Support is only available for those platforms that
// have either linux/cdrom.h or sys/cdio.h
//
// It can also be disabled with the configure option --without-cddb
//
#include "../config.h"
#if defined(AUDIOCD_SUPPORT) && (defined(HAVE_LINUX_CDROM_H) || defined(HAVE_SYS_CDIO_H))
# define ENABLE_AUDIOCD
#endif


#ifdef ENABLE_AUDIOCD

/** Class for reading information from an audio-cd.
 */
class CdInfo
{
public:
   CdInfo(const char *device="/dev/cdrom");
   virtual ~CdInfo();

   /** Read the audio-cd. Returns true on success. On error, false is
    * returned and @ref errorText contains the error message.
    */
   virtual bool readDisc();

   class Track
   {
   public:
      QString title, performer;
      int min, sec; // starting minute/second of the track
      int length;   // length of the track in seconds or KB
      int start;    // starting position
   };

   /** Returns a track. */
   Track& track(int idx) { return mTracks[idx]; }
   const Track& track(int idx) const { return mTracks[idx]; }

   /** Returns the disc's title. */
   const QString& title() const { return mTitle; }

   /** Returns the disc's performer. */
   const QString& performer() const { return mPerformer; }

   /** Returns the disc id. */
   const QString& discid() const { return mDiscId; }

   /** Returns the upc. */
   const QString& upc() const { return mUpc; }

   /** Returns the cddb id. */
   unsigned long cddbId() const { return mCddbId; }

   /** Returns a string suitable for a cddb query that contains:
    * discid ntrks off1 off2 ... nsecs
    */
   virtual QString cddbQueryString() const;

   /** Returns the number of tracks. */
   int count() const { return mTracks.size()-1; }

   /** Returns the total playing length of the CD in seconds. */
   int length() const { return mLength; }

   /** Set the disc title. */
   void setTitle(const QString& str) { mTitle = str; }

   /** Set the disc performer. */
   void setPerformer(const QString& str) { mPerformer = str; }

   /** Set the disc-id. */
   void setDiscId(const QString& str) { mDiscId = str; }

   /** Set the disc upc. */
   void setUpc(const QString& str) { mUpc = str; }

   /** Open the disc. */
   bool open();

   /** Close the disc. */
   void close();

   /** Test if the disc is opened. */
   bool isOpen() const { return mFd>=0; }

   /** Returns the error code of the last failed operation. The
    * error code may be < 0 if the error is application specific (@see errorText).
    */
   int errorCode() const { return mErrorCode; }

   /** Returns the error message for the last failed operation. */
   virtual const QString& errorText() const;

protected:
   /** Read the toc of the inserted audio-cd. */
   virtual bool readToc();

//    virtual bool parseCdInfo(unsigned char *buffer);
//    virtual void saveCdInfo(char code, char track_no, const char *data);
//    virtual unsigned short from2Byte(const unsigned char *d);
   virtual int cddbSum(int n);
   virtual int readTocHeader();
   virtual bool readTocEntry(int trackNo, Track& track);

private:
   CdInfo(const CdInfo&);
   CdInfo& operator=(const CdInfo&);

protected:
   int mFd, mErrorCode, mLength;
   QCString mDevice;
   QString mTitle, mPerformer, mDiscId, mUpc;
   unsigned long mCddbId;
   QValueVector<Track> mTracks;
   mutable QString mErrorText;
};

#endif /*ENABLE_AUDIOCD*/


#endif /*koverartist_cdinfo_h*/
