/***************************************************************************
 *   Copyright (C) 2004 by Spiros Georgaras                                *
 *   sngeorgaras@otenet.gr                                                 *
 *                                                                         *
 *   This library is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser Public License as published by   *
 *   the Free Software Foundation; either version 2.1 of the License, or   *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU Lesser Public License for more details.                           *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser Public License      *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>

#include <qfile.h>
#include <kurl.h>
#include <kdebug.h>
#include <kinstance.h>
#include <kmimemagic.h>
#include <qregexp.h>
#include <kdebug.h>

#include <errno.h> // to be removed

#include "mhtml.h"

using namespace KIO;

extern "C" { int kdemain(int argc, char **argv); }
/*================================
================================*/
int kdemain( int argc, char **argv ){
	KInstance instance( "kio_mhtml" );
	kdDebug() << "[kdemain]: Starting " << getpid() << endl;
	if (argc != 4){
		fprintf(stderr, "Usage: kio_mhtml protocol domain-socket1 domain-socket2\n");
		exit(-1);
	}
	mhtmlProtocol slave(argv[2], argv[3]);
	slave.dispatchLoop();
	kdDebug() << "[kdemain]: Done" << endl;
	return 0;
}
/*================================
================================*/
mhtmlProtocol::mhtmlProtocol( const QCString &pool, const QCString &app ) : SlaveBase( "mht", pool, app ){
	kdDebug() << "[mhtmlProtocol::mhtmlProtocol]" << endl;
	m_mhtmlFile = 0L;
}
/*================================
================================*/
mhtmlProtocol::~mhtmlProtocol(){
	delete m_mhtmlFile;
}
/*================================
================================*/
bool mhtmlProtocol::checkNewFile( const KURL & url, QString & path ){
	QString fullPath = url.path();
	kdDebug() << "[mhtmlProtocol::checkNewFile]: full path = " << fullPath << endl;
	int c=-1;
	c=fullPath.find(".mht",0,FALSE);
	if(c==-1) c=fullPath.find(".mhtml",0,FALSE);
	if(c==-1){
		redirection( QString::fromLatin1("file:") + fullPath );
		finished();
		return true;
	}

	// Are we already looking at that file ?
	if ( m_mhtmlFile && m_mhtmlName == fullPath.left(m_mhtmlName.length()) ){
		// Has it changed ?
		struct stat statbuf;
		if ( ::stat( QFile::encodeName( m_mhtmlName ), &statbuf ) == 0 ){
			if ( m_mtime == statbuf.st_mtime ){
				path = fullPath.mid( m_mhtmlName.length() );
				kdDebug() << "[mhtmlProtocol::checkNewFile]:  path before fix = " << path << endl;
				//if(path==QString("") || path.isEmpty() || path.isEmpty()) path=QString("/");
				//if(path.isNull() || path.isEmpty()) path=QString("/index.html");
				if(path.isNull() || path.isEmpty()) path="/";
				else if(path.left(1)=="/" && path.length()>1) path=path.right(path.length()-1);
				kdDebug() << "[mhtmlProtocol::checkNewFile]:  returning " << path << endl;
				return true;
			}
		}
	}
	kdDebug() << "[mhtmlProtocol::checkNewFile]: Need to open a new file" << endl;

	// Close previous file
	if ( m_mhtmlFile ){
		//m_mhtmlFile->close();
		delete m_mhtmlFile;
		m_mhtmlFile = 0L;
	}
	kdDebug() << "[mhtmlProtocol::checkNewFile]: Going on" << endl;
	// Find where the mhtml file is in the full path
	int pos = 0;
	QString archiveFile;
	path = QString::null;

	int len = fullPath.length();
	if(fullPath.right(4).lower()==".mht" || fullPath.right(6).lower()==".mhtml") fullPath+="/";
	kdDebug() << "[mhtmlProtocol::checkNewFile]: the full path is " << fullPath << endl;
	while ( (pos=fullPath.find( '/', pos+1 )) != -1 ){
		QString tryPath = fullPath.left( pos );
		archiveFile = tryPath;
		kdDebug() << "[mhtmlProtocol::checkNewFile]: " << fullPath << "  trying " << tryPath << endl;
		struct stat statbuf;
		if ( ::stat( QFile::encodeName(tryPath), &statbuf ) == 0 && !S_ISDIR(statbuf.st_mode) ){
			archiveFile = tryPath;
			kdDebug() << "[mhtmlProtocol::checkNewFile]: archiveFile = " << archiveFile << endl;
			m_mtime = statbuf.st_mtime;
			path = fullPath.mid( pos + 1 );
			kdDebug() << "[mhtmlProtocol::checkNewFile]: "<< "fullPath=" << fullPath << " path=" << path << endl;
			len = path.length();
			if ( len > 0 ){
				if ( path[ len - 1 ] == '/' ) path.truncate( len - 1 );
			}else path = QString::fromLatin1("/");
			kdDebug() << "[mhtmlProtocol::checkNewFile]: "<< "Found archiveFile = " << archiveFile << " path=" << path << endl;
			break;
		}
	}
	// Open new file
	kdDebug() << "[mhtmlProtocol::checkNewFile]: Opening mht file on " << archiveFile << endl;
	m_mhtmlFile = new KmhtmlArchive( archiveFile );
	if ( !m_mhtmlFile->open( IO_ReadOnly ) ){
		kdDebug() << "[mhtmlProtocol::checkNewFile]: Opening " << archiveFile << " failed." << endl;
		KURL disp(url.fileName());
		error( KIO::ERR_DOES_NOT_EXIST, disp.prettyURL() );
		//finished();
		delete m_mhtmlFile;
		m_mhtmlFile = 0L;
		return false;
	}
	m_mhtmlName = archiveFile;
	return TRUE;
}
/*================================
================================*/
void mhtmlProtocol::createUDSEntry( KmhtmlEntry* archFile, UDSEntry & entry ){
	UDSAtom atom;
	entry.clear();
	atom.m_uds = UDS_NAME;
	atom.m_str = archFile->name();
	kdDebug(  ) << " -- [mhtmlProtocol::createUDSEntry]: adding file = " << atom.m_str << endl;
	kdDebug(  ) << " -- [mhtmlProtocol::createUDSEntry]: original file = " << archFile->originalName() << endl;
	entry.append(atom);
	 
	atom.m_uds = UDS_FILE_TYPE;
	atom.m_long = archFile->permissions() & S_IFMT; // keep file type only
	entry.append( atom );

	atom.m_uds = UDS_SIZE;
	atom.m_long = archFile->size();
	entry.append( atom );

	atom.m_uds = UDS_MODIFICATION_TIME;
	atom.m_long = archFile->date();
	entry.append( atom );

	atom.m_uds = UDS_ACCESS;
	atom.m_long = archFile->permissions() & 07777; // keep permissions only
	//atom.m_long = archFile->permissions() & 05555; // show it read only 
	entry.append( atom );
    
	atom.m_uds = UDS_USER;
	atom.m_str = archFile->user();
	entry.append( atom );

	atom.m_uds = UDS_GROUP;
	atom.m_str = archFile->group();
	entry.append( atom );
}
/*================================
================================*/
void mhtmlProtocol::createUDSEntry( UDSEntry & entry ){
	UDSAtom atom;
	entry.clear();
	atom.m_uds = UDS_NAME;
	atom.m_str = QString("/");
	entry.append(atom);
 
	atom.m_uds = UDS_FILE_TYPE;
	atom.m_long = S_IFDIR;
	entry.append( atom );

	atom.m_uds = UDS_SIZE;
	atom.m_long = m_mhtmlFile->size();
	entry.append( atom );

	atom.m_uds = UDS_MODIFICATION_TIME;
	atom.m_long = m_mhtmlFile->date();
	entry.append( atom );

	atom.m_uds = UDS_ACCESS;
	atom.m_long = m_mhtmlFile->permissions() & 07777; // keep permissions only
	//atom.m_long = m_mhtmlFile->permissions() & 05555; // show it read only 
	entry.append( atom );
    
	atom.m_uds = UDS_USER;
	atom.m_str = m_mhtmlFile->user();
	entry.append( atom );

	atom.m_uds = UDS_GROUP;
	atom.m_str = m_mhtmlFile->group();
	entry.append( atom );
}
/*================================
================================*/
void mhtmlProtocol::listDir( const KURL & url ){
	kdDebug(  ) << " -- [mhtmlProtocol::listDir] (starting): " << url.url() << endl;

	QString path;
	if ( !checkNewFile( url, path ) ){
		QCString _path( QFile::encodeName(url.path()));
		kdDebug(  ) << " -- [mhtmlProtocol::listDir]: Checking (stat) on " << _path << endl;
		struct stat buff;
		if ( ::stat( _path.data(), &buff ) == -1 || !S_ISDIR( buff.st_mode ) ) {
			KURL disp(url.fileName());
			error( KIO::ERR_DOES_NOT_EXIST, disp.prettyURL() );
			finished();
			delete m_mhtmlFile;
			m_mhtmlFile = 0L;
			return;
		}
		// It's a real dir -> redirect
		KURL redir;
		redir.setPath( url.path() );
		kdDebug(  ) << " -- [mhtmlProtocol::listDir]: Ok, redirection to " << redir.url() << endl;
		redirection( redir );
		finished();
		// And let go of the mhtml file - for people who want to unmount a cdrom after that
		delete m_mhtmlFile;
		m_mhtmlFile = 0L;
		return;
	}

	if ( path.isEmpty() ){
		//KURL redir( url.protocol() + QString::fromLatin1( ":/") );
		KURL redir( QString::fromLatin1( "file:/") );
		kdDebug(  ) << " -- [mhtmlProtocol::listDir]: url.path()==" << url.path() << endl;
		redir.setPath( url.path() + QString::fromLatin1("/") );
		kdDebug(  ) << " -- [mhtmlProtocol::listDir]: redirection " << redir.url() << endl;
		redirection( redir );
		finished();
		return;
	}

	totalSize( m_mhtmlFile->count() );

	UDSEntry entry;
	kdDebug(  ) << " -- [mhtmlProtocol::listDir]: items = " << m_mhtmlFile->count() << endl;
	for(int i=0;i<m_mhtmlFile->count();i++){
		KmhtmlEntry archEntry(m_mhtmlFile,i);
		createUDSEntry( &archEntry, entry );
		listEntry( entry, false );
	}
	listEntry( entry, true ); // ready
	finished();
	kdDebug(  ) << " -- [mhtmlProtocol::listDir]: done" << endl;
}
/*================================
================================*/
void mhtmlProtocol::stat( const KURL & url ){
	kdDebug() << "[mhtmlProtocol::stat]" << endl;
	QString path;
	UDSEntry entry;
	int c=-1;
	c=url.url().find(".mht",0,FALSE);
	if(c==-1) c=url.url().find(".mhtml",0,FALSE);
	if(c==-1){
		kdDebug(  ) << "[mhtmlProtocol::stat]: redirection( \"file:\" + url.path() )"<< endl;
		redirection( "file:" + url.path() );
		finished();
		return;
	}
	if ( !checkNewFile( url, path ) ){
		kdDebug(  ) << "[mhtmlProtocol::stat]: url = " << url << endl;
		// We may be looking at a real directory - this happens
		// when pressing up after being in the root of an archive
		QCString _path( QFile::encodeName(url.path()));
		kdDebug(  ) << "mhtmlProtocol::stat (stat) on " << _path << endl;
		struct stat buff;
		if( ::stat( _path.data(), &buff ) == -1 || !S_ISDIR( buff.st_mode )){
			kdDebug(  ) << "isdir=" << S_ISDIR( buff.st_mode ) << "  errno=" << strerror(errno) << endl;
			KURL disp(url.fileName());
			error( KIO::ERR_DOES_NOT_EXIST, disp.prettyURL() );
			finished();
			delete m_mhtmlFile;
			m_mhtmlFile = 0L;
			return;
		}
		// Real directory. Return just enough information for KRun to work
		kdDebug() << "[mhtmlProtocol::stat]: Real directory" << endl;
		UDSAtom atom;
		atom.m_uds = KIO::UDS_NAME;
		atom.m_str = url.fileName();
		entry.append( atom );
		kdDebug(  ) << "mhtmlProtocol::stat returning name=" << url.fileName() << endl;
		atom.m_uds = KIO::UDS_FILE_TYPE;
		atom.m_long = buff.st_mode & S_IFMT;
		entry.append( atom );
		statEntry( entry );
		finished();
		// And let go of the mht file - for people who want to unmount a cdrom after that
		delete m_mhtmlFile;
		m_mhtmlFile = 0L;
		return;
	}
	kdDebug() << "[mhtmlProtocol::stat]: Going on..." << endl;
// 	if(path!="/"){
// 		kdDebug() << "[mhtmlProtocol::stat]: checking" << path << endl;
// 		KmhtmlEntry fileExist(m_mhtmlFile,path);
// 		kdDebug() << "[mhtmlProtocol::stat]: created entry" << path << endl;
// 		if(!fileExist.exists()){
// 			error( KIO::ERR_DOES_NOT_EXIST, path );
// 			finished();
// 			return;
// 		}
// 	}
	if(path=="/"){
		kdDebug(  ) << "[mhtmlProtocol::stat]: Returning Contents " << endl;
		createUDSEntry( entry );
		statEntry( entry );
		finished();
		//listDir(url);
	}else{
		KmhtmlEntry b(m_mhtmlFile,path);
		if(!b.exists()){
			kdDebug(  ) << "[mhtmlProtocol::stat]: !exists: " << b.name() << endl;
			KURL disp(url.fileName());
			error( KIO::ERR_DOES_NOT_EXIST, disp.prettyURL() );
			finished();
			delete m_mhtmlFile;
			m_mhtmlFile = 0L;
			return;
		}
		createUDSEntry( &b, entry );
		statEntry( entry );
		finished();
	}
}
/*================================
================================*/
void mhtmlProtocol::get( const KURL & url ){
	kdDebug(  ) << "[mhtmlProtocol::get]: " << url.url() << endl;
	QString path;
	if ( !checkNewFile( url, path ) ){
		KURL disp(url.fileName());
		error( KIO::ERR_DOES_NOT_EXIST, disp.prettyURL() );
		finished();
		delete m_mhtmlFile;
		m_mhtmlFile = 0L;
		return;
	}
	KmhtmlEntry archiveEntry(m_mhtmlFile,path);
	if(!archiveEntry.exists()){
		error( KIO::ERR_DOES_NOT_EXIST, url.prettyURL() );
		finished();
		return;
	}
	mimeType( archiveEntry.mimeType() );
	totalSize( archiveEntry.size() );
	kdDebug(  ) << "[mhtmlProtocol::get]: totalSize=" << archiveEntry.size() << endl;
	QByteArray ret=*archiveEntry.data();
	data(ret);
	kdDebug(  ) << "[mhtmlProtocol::get]: data size=" << ret.size() << endl;
	processedSize( archiveEntry.size() );
	kdDebug(  ) << "[mhtmlProtocol::get]: processedSize=" << archiveEntry.size() << endl;
	data( QByteArray() );
	finished();
}
/*================================
================================*/
QByteArray mhtmlProtocol::fixHtmlFile(KmhtmlArchive* arc, QByteArray* htmlData){
	QCString str;
	str=*htmlData;
	QRegExp findReg;
	findReg.setCaseSensitive(TRUE);
	findReg.setMinimal(TRUE);
	findReg.setWildcard(FALSE);
	for(int j=arc->count()-1;j>=0;j--){
		KmhtmlEntry arcF(arc, j);	
		if(arcF.mimeType()!=arc->ContentType()){
		// fix links
			QString arcForiginalName=escapeString(arcF.originalName());
			QString arcFname=escapeString(arcF.name());
			if(arcF.originalName()!=arcF.name()){
				findReg.setPattern(arcForiginalName);
				str.replace(findReg,arcF.name());
			}
			findReg.setPattern(QString("http[^ ]*") + arcFname );
			str.replace(findReg,arcF.name());
			findReg.setPattern(QString("file://.*") + arcFname );
			str.replace(findReg,arcF.name());
			findReg.setPattern(QString("ftp://.*") + arcFname );
			str.replace(findReg,arcF.name());
		}
	}
	kdDebug(  ) << "[mhtmlProtocol::fixHtmlFile]: done!!!" << endl;
	QByteArray result;
	result=str;
	result.resize(result.size()-1);
	return result;
}
/*================================
================================*/
QString mhtmlProtocol::escapeString(QString str){
	QString ret=QString::null;
	uint strLength=str.length();
	for(int i=0;(uint)i<strLength;i++){
		QChar c=str.at((uint)i);
		if(c==QChar('\\') || c==QChar('/') || c==QChar('(') || c==QChar(')') || 
			c==QChar('[') || c==QChar(']') || c==QChar('{') || c==QChar('}'))
			ret.append(QChar('\\'));
		ret.append(str.mid(i,1));
	}
	return ret;
}
