/***************************************************************************
                          emoticonwidget.cpp -  description
                             -------------------
    begin                : Sun Dev 11 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : "vdboor" --at-- "codingdomain.com"
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "emoticonwidget.h"

#include <kiconview.h>
#include <klocale.h>
#include <klistbox.h>
#include <kmessagebox.h>
#include <kstddirs.h>

#include <qdir.h>
#include <qpushbutton.h>

#include "../account.h"
#include "../kmessdebug.h"
#include "../emoticonmanager.h"
#include "../dialogs/addemoticondialog.h"



#ifdef KMESSDEBUG_EMOTICONS
#define KMESSDEBUG_EMOTICONS_SETTINGS
#endif



/**
 * Constructor
 *
 * @param parent  Parent widget
 * @param name    Qt object name
 */
EmoticonWidget::EmoticonWidget( QWidget* parent, const char* name )
  : EmoticonWidgetInterface(parent, name)
  , emoticonTheme_(0)
{
  connect( addEmoticonButton_,    SIGNAL(               clicked() ),
           this,                    SLOT( showAddEmoticonDialog() ) );
  connect( renameEmoticonButton_, SIGNAL(               clicked() ),
           this,                    SLOT(  renameCustomEmoticon() ) );
  connect( removeEmoticonButton_, SIGNAL(               clicked() ),
           this,                    SLOT(  removeCustomEmoticon() ) );

  connect( customEmoticonsView_,  SIGNAL( itemRenamed(QIconViewItem*)           ),
           this,                    SLOT( customEmoticonRenamed(QIconViewItem*) ) );
  connect( customEmoticonsView_,  SIGNAL( selectionChanged()                                   ),
           this,                    SLOT( customEmoticonSelected()                             ) );

  // the other properties of the widget are initialized
  // in the loadSettings() method
}



/**
 * Destructor
 */
EmoticonWidget::~EmoticonWidget()
{
  if( emoticonTheme_ != 0 )
  {
#ifdef KMESSDEBUG_EMOTICONS_SETTINGS
    kdDebug() << "EmoticonWidget::~EmoticonWidget() - Deleting temporary theme." << endl;
#endif
    delete emoticonTheme_;
  }
}



/**
 * Rename the currently selected custom emoticon
 *
 * @param item  The changed item
 * @param name  New item's name
 */
void EmoticonWidget::customEmoticonRenamed( QIconViewItem *item )
{
  Emoticon *emoticon;
  QString oldName = item->key();
  QString newName = item->text();

  // Do nothing if the name ain't changed
  if( oldName == newName )
  {
    return;
  }

#ifdef KMESSDEBUG_EMOTICONS_SETTINGS
  kdDebug() << "EmoticonWidget::customEmoticonRenamed() - Renaming emoticon '" << oldName << "' to '" << newName << "'" << endl;
#endif

  // First, check if an emoticon with the new name already exists
  for( emoticon = emoticonTheme_->first(); emoticon != 0; emoticon = emoticonTheme_->next() )
  {
    if( emoticon->getShortcut() != newName )
    {
      continue;
    }

    int result = KMessageBox::questionYesNo( 0,
                                             i18n("The emoticon \"%1\" already exists, do you want to replace it?")
                                               .arg( newName ),
                                             i18n("Add New Emoticon") );

      // The user doesn't want to overwrite the emoticon, cancel the operation
    if( result != KMessageBox::Yes )
    {
      return;
    }

    // Remove the icon of the replaced emoticon from both the icon view and the theme
    delete item;
    emoticonTheme_->removeEmoticon( newName );

    // Find the icon which will replace the deleted one
    item = customEmoticonsView_->findItem( oldName, Qt::ExactMatch );
    if( ! item )
    {
      // Something has gone wrong, abort. At this point, the user has lost an emoticon :(
      kdWarning() << "EmoticonWidget::customEmoticonRenamed() - Unable to find temporary item during rename of '" << oldName << "' to '" << newName << "'" << endl;
      return;
    }

    // Change the icon's name to match the new one
    item->setText( newName );
    item->setKey( newName );

    // Set focus to the new item
    customEmoticonsView_->setFocus();
    customEmoticonsView_->ensureItemVisible( item );

    // Stop searching, there can be only one emoticon to replace
    break;
  }

  // If no other emoticons with that shortcut exist, update the name
  for( emoticon = emoticonTheme_->first(); emoticon != 0; emoticon = emoticonTheme_->next() )
  {
    if( emoticon->getShortcut() == oldName )
    {
      // If there isn't a custom text for the tooltip, but it is the shortcut text, update it
      if( emoticon->getTooltip() == emoticon->getShortcut() )
      {
        emoticon->setTooltip( newName );
      }

      // Update the shortcut text
      emoticon->setShortcut( newName );
      break;
    }
  }

  // Update the viewport and the buttons
  customEmoticonSelected();
}



/**
 * Update the Remove Button status when emoticons get selected
 */
void EmoticonWidget::customEmoticonSelected()
{
  renameEmoticonButton_->setEnabled( customEmoticonsView_->currentItem() != 0 );
  removeEmoticonButton_->setEnabled( customEmoticonsView_->currentItem() != 0 );

  // Other than resetting the buttons, also update the layout to show any changes
  customEmoticonsView_->arrangeItemsInGrid();
}



/**
 * Load the settings of the dialog
 *
 * @param account  Account instance which settings will be loaded
 */
void EmoticonWidget::loadSettings( Account *account )
{
  accountHandle_ = account->getHandle();

#ifdef KMESSDEBUG_EMOTICONS_SETTINGS
  kdDebug() << "EmoticonWidget::loadSettings() - Loading emoticon settings for account " << accountHandle_ << " - Current is " << accountHandle_ << " ." << endl;
#endif

  // Read the settings and set current emoticons
  emoticonStyle_ = account->getEmoticonStyle();

  emoticonTheme_ = new EmoticonTheme();
  emoticonTheme_->loadTheme( accountHandle_, true );

  connect( emoticonTheme_,  SIGNAL(                  updated() ),
           this,              SLOT( updateCustomEmoticonView() ) );

  // Update the emoticon themes widget
  updateThemesList();

  // Update the custom emoticons widget
  updateCustomEmoticonView();

#ifdef KMESSDEBUG_EMOTICONS_SETTINGS
  kdDebug() << "EmoticonWidget::loadSettings() - Done!" << endl;
#endif
}



/**
 * Delete the selected custom emoticon from the theme and the view
 */
void EmoticonWidget::removeCustomEmoticon()
{
  if( customEmoticonsView_->currentItem() == 0 )
  {
    return;
  }

  if( emoticonTheme_->removeEmoticon( customEmoticonsView_->currentItem()->key() ) )
  {
    delete customEmoticonsView_->currentItem();

    // Forcibly update the buttons status to disable the rename&remove buttons
    customEmoticonSelected();
  }
}



/**
 * Starts in-place renaming of the selected custom emoticon
 */
void EmoticonWidget::renameCustomEmoticon()
{
  QIconViewItem *item = customEmoticonsView_->currentItem();

  if( item == 0 )
  {
    return;
  }

  customEmoticonsView_->setFocus();
  customEmoticonsView_->ensureItemVisible( item );
  item->rename();
}



/**
 * Save the settings to the account object
 *
 * @param account  Account instance which settings will be saved
 */
void EmoticonWidget::saveSettings( Account *account )
{
  // Save the changes we've made
  emoticonTheme_->saveTheme();
  account->setEmoticonStyle( emoticonThemesList_->currentText() );

  disconnect( emoticonTheme_,  SIGNAL(                  updated() ),
              this,              SLOT( updateCustomEmoticonView() ) );

  // Apply the changes to the current account, too
  if( accountHandle_ == CurrentAccount::instance()->getHandle() )
  {
    EmoticonManager::instance()->replaceCustomTheme( emoticonTheme_ );
  }
}



/**
 * Show the Add New Emoticon dialog to create a new custom emoticon
 */
void EmoticonWidget::showAddEmoticonDialog()
{
  new AddEmoticonDialog( emoticonTheme_, this, "AddEmoticonDialog" );
}



/**
 * Fill the widget which contains all the user's custom emoticons
 */
void EmoticonWidget::updateCustomEmoticonView()
{
  QIconViewItem *item;
  Emoticon      *emoticon;

  customEmoticonsView_->clear();


  // Add all emoticons to the list
  for( emoticon = emoticonTheme_->first(); emoticon != 0; emoticon = emoticonTheme_->next() )
  {
    if( ! emoticon->isValid() )
    {
      continue;
    }

    item = new QIconViewItem( customEmoticonsView_, emoticon->getShortcut(), QPixmap( emoticon->getPicturePath() ) );

    item->setRenameEnabled( true );
    item->setDragEnabled( false );
    item->setDropEnabled( false );
    item->setSelectable( true );

   customEmoticonsView_->insertItem( item );
  }
}



/**
 * Initialize the standard emoticon theme chooser
 */
void EmoticonWidget::updateThemesList()
{
  emoticonThemesList_->clear();

  // Browse all emoticon dirs, to get all emoticons
  QStringList themeDirs = KGlobal::dirs()->findDirs("emoticons", "");

#ifdef KMESSDEBUG_EMOTICONS_SETTINGS
  kdDebug() << "EmoticonWidget::updateEmoticons() - Theme dirs: " << themeDirs << endl;
#endif

  for(unsigned int x = 0; x < themeDirs.count(); x++)
  {
    // Browse for all themes in the emoticon dir
    QDir themeDir(themeDirs[x]);
    QStringList themes = themeDir.entryList( QDir::Dirs, QDir::Name | QDir::IgnoreCase );
    for( QStringList::Iterator it = themes.begin(); it != themes.end(); ++it )
    {
      QString dirName = *it;

      // Ignore '.' and '..'
      if( dirName == "." || dirName == ".." )
      {
        continue;
      }

#ifdef KMESSDEBUG_EMOTICONS_SETTINGS
      kdDebug() << "EmoticonWidget::updateEmoticons() - Folder: " << themeDirs[x] << " - Theme: " << dirName << endl;
#endif

      // Add the theme to the list, using the directory name; its first valid emoticon is used as a preview
      QPixmap previewPixmap( EmoticonTheme::getThemeIcon( themeDirs[x] + dirName ) );
      emoticonThemesList_->insertItem(previewPixmap, dirName);
    }
  }

  // Highlight the selected emoticon
  QListBoxItem *item = emoticonThemesList_->findItem( emoticonStyle_, Qt::ExactMatch | Qt::CaseSensitive );
  if(item == 0)
  {
    kdWarning() << "Current emoticon style was not found, attempting to revert to the default setting." << endl;
    item = emoticonThemesList_->findItem("KMess-new", Qt::ExactMatch);
  }

  // If there is an item, select it
  if(item != 0)
  {
    emoticonThemesList_->setCurrentItem( item );
  }
  else
  {
    // select the first item otherwise
    emoticonThemesList_->setCurrentItem( 0 );
  }
}



#include "emoticonwidget.moc"
