/***************************************************************************
                          emoticonsidebar.cpp -  description
                             -------------------
    begin                : Mon 12 09 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : "vdboor" --at-- "codingdomain.com"
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qgrid.h>
#include <qlayout.h>
#include <qobjectlist.h>

#include "emoticonsidebar.h"
#include "emoticonbutton.h"

#include "../emoticonmanager.h"
#include "../emoticontheme.h"
#include "../kmessdebug.h"



// The constructor
EmoticonSidebar::EmoticonSidebar( QWidget *parent, const char *name, bool useCustomTheme )
 : QVBox(parent, name)
 , cols_(3) // Bogus minimum value, the grid will be redrawn later on anyways
 , isCustom_(useCustomTheme)
{
  // Create the grid to lay the buttons into, and a spacer to keep the grid on top of the sidebar
  grid_ = new QGrid( cols_, Horizontal, this );
  spacer_ = new QSpacerItem( 20, 0, QSizePolicy::Ignored, QSizePolicy::Ignored );

  // Get and connect to the emoticon manager. We'll need to update the sidebar when the emoticon
  // theme changes
  manager_ = EmoticonManager::instance();
  connect( manager_, SIGNAL(          updated() ),
           this,     SLOT  ( slotThemeUpdated() ) );

  // Force adding the theme's emoticons
  slotThemeUpdated();
}



// The destructor
EmoticonSidebar::~EmoticonSidebar()
{
}



// Create an emoticon button on the grid
void EmoticonSidebar::createEmoticonButton(const Emoticon *emoticon)
{
  EmoticonButton *button;

  // Create the button
  // Use a separate EmoticonButton, so we can also include the
  // emoticon code in the button signal.
  button = new EmoticonButton( emoticon, grid_, "EmoticonButton" );

  // Connect the button's insertEmoticon signal
  connect( button, SIGNAL(     insertEmoticon(QString) ) ,
           this,   SLOT  ( slotInsertEmoticon(QString) ) );
}



// Overwritten method to detect resizes
void EmoticonSidebar::resizeEvent(QResizeEvent *event)
{
#ifdef KMESSDEBUG_EMOTICONSIDEBAR
  kdDebug() << "EmoticonSidebar::resizeEvent() - Resizing from " << event->oldSize() << " to " << size() << endl;
#endif

  // Update emoticon layout only if the sidebar size has changed
  if( event->oldSize().width() != width() )
  {
    updateEmoticonLayout();
  }

  // Call the parent
  QWidget::resizeEvent( event );
}



// Layout the emoticon buttons
void EmoticonSidebar::updateEmoticonLayout()
{
  // Insert the button to the layout
  QObjectList *childWidgets = grid_->queryList( "EmoticonButton" );

  // The list is null if there aren't any children.
  if( childWidgets == 0 )
  {
    return;
  }

  // Calculate the number of button columns that will fit in the new sidebar size
  cols_ = width() / EMOTICON_BUTTON_SIZE;
  // Create a new grid with the new size
  QGrid *newGrid = new QGrid( cols_, Horizontal, this );

#ifdef KMESSDEBUG_EMOTICONSIDEBAR
  kdDebug() << "EmoticonSidebar::updateEmoticonLayout() - Changing grid to " << cols_ << " columns, totally " << childWidgets->count() << " buttons." << endl;
#endif

  QWidget *obj;
  QObjectListIt it( *childWidgets );

  // Transfer all buttons from the old and now misshaped grid to the new one
  while( it.current() != 0 )
  {
    obj = static_cast<QWidget*>( it.current() );
    // Using QPoint with zeros because the old pos will be ignored anyways. And it avoids a call to QWidget::pos().
    obj->reparent( newGrid, QPoint(0,0), true );
    ++it;
  }
  // We're finished using the list, we can delete it
  delete childWidgets;

  // Replace the old grid with the updated one
  delete grid_;
  grid_ = newGrid;

  // Set some attributes to it, to make it display correctly
  grid_->setMargin(0);
  grid_->setSpacing(0);
  grid_->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  grid_->setMaximumWidth( parentWidget()->width() );
  grid_->show();

  // Also readd the spacer to the bottom of the layout, to always have on top the grid and on bottom the spacer.
  // This is needed because new items are added to the layout bottom, so we would get the spacer on top after an update.
  layout()->removeItem( spacer_ );
  layout()->addItem( spacer_ );
}



// A button was clicked
void EmoticonSidebar::slotInsertEmoticon(QString html)
{
  emit insertEmoticon(html);
}


// The theme has changed, we need to reload it
void EmoticonSidebar::slotThemeUpdated()
{
#ifdef KMESSDEBUG_EMOTICONSIDEBAR
  kdDebug() << "EmoticonSidebar::slotThemeUpdated() - Removing old theme from the sidebar." << endl;
#endif

  // Find the existing buttons in the sidebar
  QObjectList *childWidgets = grid_->queryList( "EmoticonButton" );

  // The list is null if there aren't any children.
  if( childWidgets != 0 )
  {
    QObjectListIt it( *childWidgets );

    // Delete them
    while( it.current() != 0 )
    {
      delete it.current();
      ++it;
    }
  }
  // We're finished using the list, we can delete it
  delete childWidgets;

#ifdef KMESSDEBUG_EMOTICONSIDEBAR
  kdDebug() << "EmoticonSidebar::slotThemeUpdated() - Updating sidebar with current emoticons." << endl;
#endif

  Emoticon *emoticon;
  EmoticonTheme *theme = manager_->getTheme( isCustom_ );

  // Add all theme emoticons to the list
  for(emoticon = theme->first(); emoticon != 0; emoticon = theme->next() )
  {
    if( emoticon->isValid() )
    {
      createEmoticonButton( emoticon );
    }
  }

  // Update the buttons layout to fit the current number of buttons and sidebar size
  updateEmoticonLayout();
}



#include "emoticonsidebar.moc"
