/*************************************************/
/* methods for class Circuit                  */
/*                                               */
/* widget containing a circuit                   */
/* actions with/at objects in the net-widget     */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qstring.h>
#include <qpoint.h>
#include <qpaintdevice.h>

#include "klogic.h"
#include "klogicIO.h"
#include "mainw.h"
#include "netw.h"
#include "circuit.h"
#include "xwire.h"
#include "devLib.h"
#include "selection.h"
#include "xmlExport.h"
#include "deviceTypes.h"
#include "deviceFactory.h"
#include "deviceMuxDemux.h"
#include "deviceOsc.h"
#include "deviceRAM.h"
#include "deviceJKFF.h"
#include "performance.h"

/***************************************************/
/* methods of Circuit                           */
/***************************************************/
int Circuit::instance = 0;
int Circuit::magic_version = 0;	// to be removed soon!
DeviceFactory * Circuit::m_poDeviceFactory = DeviceFactory::getInstance();
int Circuit::m_iCalcCalls = 0;
int Circuit::m_iPropCalls = 0;

Circuit::Circuit()
	:CustomDevice(DeviceType::fNET, QPoint(0,0))
{

	instance++;
	m_parent = 0;
}

Circuit::Circuit(const QPoint& oPos)
	:CustomDevice(DeviceType::fNET, oPos)
{

	instance++;
	m_parent = 0;
}

Circuit::Circuit(const QPoint& oPos, int iSize)
	:CustomDevice(DeviceType::fNET, oPos, iSize)
{

	instance++;
	m_parent = 0;
}

Circuit::~Circuit()
{
	instance--;
	deleteAll();
}

void Circuit::deleteAll()
{
	// destroy all devices
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter)
		delete iter.getDevice();
	m_oDeviceList.clear();

	// destroy all wires
	KlogicList<XWire> *w = netwire.First();
	while(w) {
		delete w->Get();
		w = w->Next();
	}

	// destroy lists
	netwire.Destroy();

	initImport();
}

bool Circuit::hasImpossibleEquations()
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		XDevice *poDevice = iter.getDevice();
		if (poDevice->type() == DeviceType::fRAM || poDevice->isTristate())
			return true;
	}
	return false;
}

// returns 1 if all names are unique, otherwise 0
int Circuit::checkCircuitNames()
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		// sub circuits ...
		XDevice *poDevice = iter.getDevice();
		Circuit *poCircuit = poDevice->devIsCircuit();
		if (poCircuit && !poCircuit->checkCircuitNames())
			return 0;

		// devices ...
		for (DeviceList::iterator iter2 = m_oDeviceList.begin(); iter2 != m_oDeviceList.end(); ++iter2) {
			if (iter2.getDevice() != poDevice)
				if (!strcmp(poDevice->getName(), iter2.getDevice()->getName()))
					return 0;
		}
	}
	return 1;
}

// change all duplicate names to be unique
int Circuit::unifyCircuitNames()
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		// sub circuits ...
		XDevice *poDevice = iter.getDevice();
		Circuit *poCircuit = poDevice->devIsCircuit();
		if (poCircuit && !poCircuit->unifyCircuitNames())
			return 0;

		// devices ...
		for (DeviceList::iterator iter2 = m_oDeviceList.begin(); iter2 != m_oDeviceList.end(); ++iter2) {
			if (iter2.getDevice() != poDevice) {
				if (!strcmp(poDevice->getName(), iter2.getDevice()->getName())) {
					QString buffer;
					buffer.sprintf("uni%d", iter2.getDevice()->getID());
					setName(iter2.getDevice()->getID(), buffer);
				 }
			}
		}
	}
	return 1;
}

void Circuit::getAllEquations(map<QString,OutputInfo>& oEquations, const QString& sQPrefix, bool bIsolate)
{
	QString sNewPrefix = sQPrefix + getName() + ".";
	sNewPrefix.replace(QRegExp(" "), "_" );

	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		Device *dev = iter.getDevice();
		dev->getAllEquations(oEquations, sNewPrefix, bIsolate);
	}
}

Circuit * Circuit::rootParent()
{
	Circuit *top;

	top = this;
	while(top->parent()) top = top->parent();
	return top;
}

void Circuit::setParent(XDevice *parent)
{
	m_parent = parent->devIsCircuit();
}

Circuit * Circuit::parent()
{
	return m_parent;
}

DeviceList::iterator Circuit::getDeviceIterator()
{
	return m_oDeviceList.begin();
}

DeviceList::iterator Circuit::getDeviceEndIterator()
{
	return m_oDeviceList.end();
}

bool Circuit::empty()
{
	return (!m_oDeviceList.size() && !netwire.First());
}

void Circuit::applyDefaults()
{
	setDelay(Global::Device::getDelay());
	setInverted(Global::Device::isInverted());
	setWidth(Global::Device::getWidth());
	setUndefinedValue(Global::Device::getUndefinedValue());
	displayName(Global::Device::isDisplayNames());
	setImage();

	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		XDevice *poDevice = iter.getDevice();
		Circuit *poCircuit = poDevice->devIsCircuit();
		if (poCircuit) poCircuit->applyDefaults();
		else {
			poDevice->setDelay(Global::Device::getDelay());
			poDevice->setInverted(Global::Device::isInverted());
			poDevice->setWidth(Global::Device::getWidth());
			poDevice->setUndefinedValue(Global::Device::getUndefinedValue());
			poDevice->displayName(Global::Device::isDisplayNames());
			poDevice->setEquation();
			poDevice->parseEquation();
			setName(poDevice->getID(), poDevice->getName());
			poDevice->setImage();
		}
	}
}

void Circuit::mapImportID(int iOldDeviceID, int iNewDeviceID)
{
	XDevice *poInputDev = m_oInputList[iNewDeviceID];
	if (poInputDev)
		mapInputName(iOldDeviceID, iNewDeviceID, poInputDev->getName());

	XDevice *poOutputDev = m_oOutputList[iNewDeviceID];
	if (poOutputDev)
		mapOutputName(iOldDeviceID, iNewDeviceID, poOutputDev->getName());
}

void Circuit::setName(int iDeviceID, const QString& sName)
{
	XDevice *poDevice = m_oDeviceList[iDeviceID];
	if (poDevice) {
		poDevice->setName(sName);
		if ((poDevice->type() == DeviceType::fIN || poDevice->type() == DeviceType::fOUT) && !sName.isEmpty() && sName.length()) {
			// actualize names within circuit image
			if (poDevice->type() == DeviceType::fIN) {
				if (!m_oNamedInput.setName(iDeviceID, sName))
					warning("Circuit::setName - no such input");
			}
			if (poDevice->type() == DeviceType::fOUT) {
				if (!m_oNamedOutput.setName(iDeviceID, sName))
					warning("Circuit::setName - no such output");
			}
			if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
		}
	}
}

bool Circuit::hasBooleanOutput(int output_id) const
{
	map<int, bool>::const_iterator iter = m_mHasBooleanOutput.find(output_id);
	if (iter != m_mHasBooleanOutput.end())
		return iter->second;
	return true;
}

void Circuit::setHasBooleanOutput(int output_id, bool bHasBooleanInput)
{
	m_mHasBooleanOutput[output_id] = bHasBooleanInput;
}

void Circuit::garbageCollection(QPainter *p)
{
	KlogicList<XWire> *lw = netwire.First();

	while(lw) {
		lw->Get()->Wire::garbageCollection();
		if (!wireOK(p, lw->Get())) {
			lw = netwire.First();
		} else {
			lw = lw->Next();
		}
	}
}

//*************************************************************************
//* drawing methods
//*************************************************************************

// erase device, then draw all
void Circuit::drawDev(QPaintDevice *paintd, QPainter *p, XDevice *dev)
{
	// erase device and its connected wires
	dev->erase(p);
	// redraw all
	draw(paintd, p);
}

// erase all devices and wires in the net
void Circuit::erase(QPainter *p)
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter)
		iter.getDevice()->erase(p);

	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		lw->Get()->erase(p);
		lw = lw->Next();
	}
}

// draw all devices and wires in the net
// if paintd is NULL then drawing device is a printer
void Circuit::draw(QPaintDevice *paintd, QPainter *p)
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		if (paintd) iter.getDevice()->drawImage(paintd, p);
		else iter.getDevice()->drawImage(p);
	}

	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		if (paintd) {
			lw->Get()->drawImage(p);
		}
		else {
			// printer output
			lw->Get()->Wire::setColor(Qt::black);
			lw->Get()->drawImage(p);
			lw->Get()->setColor(true);
		}
		lw = lw->Next();
	}
}

// force setting status dependant images
void Circuit::forceOutputChange()
{
	Device::forceOutputChange();
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter)
		iter.getDevice()->forceOutputChange();
}

// draw all wires with its actual potential
// draw all devices with a changed image
// returns wether a runtime shortcut occured (WSHORT)
int Circuit::drawStatus(QPaintDevice *paintd, QPainter *p)
{
	int ret_code = 0;

	// some devices like 7S or LED
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		// if image changes..
		if (iter.getDevice()->setColor()) {
			// ..draw device
			iter.getDevice()->drawImage(paintd, p);
		}
	}

	// draw all wires in color depending of device-output
	int ret_tmp;
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		// if color changes..
		if (0 != (ret_tmp = lw->Get()->setColor())) {
			// ..draw wire
			lw->Get()->drawImage(p);
			// .. runtime shortcut detection
			if (ret_tmp == WSHORT)
				ret_code = WSHORT;
		}
		lw = lw->Next();
	}

	// .. runtime shortcut detection
	return ret_code;
}

void Circuit::setUndefinedValue(int iUndefinedValue)
{
	for (DeviceList::iterator iter = m_oOutputList.begin(); iter != m_oOutputList.end(); ++iter)
		iter.getDevice()->setUndefinedValue(iUndefinedValue);
	Device::setUndefinedValue(iUndefinedValue);
}

//*************************************************************************
//* device methods
//*************************************************************************

// create new device
XDevice * Circuit::newDevice(int iFunction, const QPoint& oPos, int iSize)
{
	return m_poDeviceFactory->createDevice(this, iFunction, oPos, iSize);
}

// add new device
bool Circuit::addChild(XDevice *dev, bool bImport)
{
	int type = dev->type();
	if (type == DeviceType::fIN || type == DeviceType::fOUT) {
		QString text = dev->getName();
		if (type == DeviceType::fIN) {
			if (!bImport) {
				if (addInputName(text, -1, dev->getID()) < 0)
					return false;
			}
			m_oInputList.add(dev, dev->getID());
		}
		if (type == DeviceType::fOUT) {
			if (!bImport) {
				QString text = dev->getName();
				if (addOutputName(text, -1, dev->getID()) < 0)
					return false;
			}
			dev->setUndefinedValue(getUndefinedValue());
			m_oOutputList.add(dev, dev->getID());
		}
	}
	m_oDeviceList.add(dev, dev->getID());

	return true;
}

// virtual
// returns pointer to the net-device if device is a net-device
Circuit * Circuit::devIsCircuit()
{
	return this;
}

// look for a device containing point
XDevice * Circuit::containsDev(QPoint p)
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		if (iter.getDevice()->contains(p))
			return iter.getDevice();
	}
	return 0;
}

// look for a device containing device
XDevice * Circuit::containsDev(XDevice *dev)
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		XDevice *poDevice = iter.getDevice();
		if (poDevice != dev) {
			if (poDevice->contains(dev->getPos())) {
				return poDevice;
			}
		}
	}
	return 0;
}

// mark all devices in rect as selected
void Circuit::selectDev(QRect r, KlogicList<XDevice> *d)
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		XDevice *poDevice = iter.getDevice();
		if (poDevice->contains(r)) {
			if (!d->With(poDevice))
				d->Append(poDevice);
			poDevice->select(true);
		}
		else {
			d->Destroy(poDevice);
			poDevice->select(false);
		}
	}
}

// called from class Selection
// add all selected devices to the selection (paste operation)
void Circuit::grabSelection(Selection *sel)
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		XDevice *poDevice = iter.getDevice();
		if (poDevice->isSelected())
			sel->add(poDevice);
	}

	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		if (lw->Get()->isSelected()) sel->add(lw->Get());
		lw = lw->Next();
	}
}

// delete a device
void Circuit::deleteDevice(QPainter *p, XDevice *dev)
{
	if (dev->type() == DeviceType::fIN) {
		m_oInputList.remove(dev->getID());
		removeInputName(dev->getID());
	}
	if (dev->type() == DeviceType::fOUT) {
		m_oOutputList.remove(dev->getID());
		removeOutputName(dev->getID());
	}

	// erase bitmap from screen
	dev->erase(p);
	m_oDeviceList.remove(dev->getID());
	delete dev;
}

//*************************************************************************
//* wire methods
//*************************************************************************

// look for a wire containing point
XWire * Circuit::containsWire(QPoint p)
{
	KlogicList<XWire> *l = netwire.First();

	while(l) {
		// if wire contains point, the node will be locked automaticly
		// if not, a new node is created
		if (l->Get()->createNode(p)) return l->Get();
		l = l->Next();
	}
	return (XWire *)NULL;
}

// look for a wire containing (parts of) rect and select them
void Circuit::selectWire(QRect r, KlogicList<XWire> *w)
{
	KlogicList<XWire> *l = netwire.First();

	while(l) {
		// catch nodes, mark them as selected
		if (l->Get()->select(r)) {
			if (!w->With(l->Get())) w->Append(l->Get());
		}
		else {
			w->Destroy(l->Get());
		}
		l = l->Next();
	}
}

int Circuit::removeNode(QPainter *p, QPoint pt)
{
	KlogicList<XWire> *l = netwire.First();
	XWire *w1, *w2;
	int ret = 0;
	int nextl = 1;

	// can remove more than one node if required
	while(l) {
		w1 = l->Get();
		if (w1->contains(pt)) {
			w1->erase(p);
			w2 = w1->removeNode();
			if (!wireOK(p, w1)) {
				nextl = 0;
				l = netwire.First();
			}
			if (w2 && !wireOK(p, w2)) {
				nextl = 0;
				l = netwire.First();
			}
			ret++;
		}
		if (nextl) l = l->Next();
		else nextl = 1;
	}
	return ret;
}

// look for a device or wire, connected to given wire
// connections will be made automaticly on active node of wire
int Circuit::checkConnection(XWire *w)
{
	int ret;

	// look for device connection/disconnection
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		if (DOK != (ret = iter.getDevice()->checkConnection(w))) {
			w->setColor(true);
			return ret;
		}
	}
	// look for wire connection/disconnection
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		if (w != lw->Get()) {
			if (WOK != (ret = lw->Get()->checkConnection(w))) {
				w->setColor(true);
				return ret;
			}
		}
		lw = lw->Next();
	}
	return NOK;
}

// look for a device, connected to given wire
// (used as import-method only)
int Circuit::checkConnection(int devid, int inverted, XWire *w)
{
	XDevice *poDevice = m_oDeviceList[devid];
	if (!poDevice)
		return NFAIL;

	int iRet = poDevice->checkConnection(w, inverted);
	if (iRet != DOK)
		return iRet;

	return NFAIL;
}

// create a new wire (import)
XWire *Circuit::newWire()
{
	XWire *w = new XWire();
	netwire.Append(w, w->getID());
	return w;
}

// get a wire (import)
XWire *Circuit::getWire(int id)
{
	return netwire.Get(id);
}

// create a new wire
XWire *Circuit::newWire(QPainter *p, QPoint pt)
{
	XWire *w;
	int ret;

	// create new wire and its first node
	w = new XWire(pt);

	// look for connection in the first point
	ret = checkConnection(w);
	switch (ret) {
		case WCONN:
		case DCONN:
			pt = w->getActive();	// same point, but aligned to the grid
			break;
	}

	// wire possibly has changed
	w->erase(p);
	w->draw(p);

	// lock new wire, create second node
	if (!w->createNode(pt)) {
		delete w;
		warning(NEWWIRE);
		return (XWire *)NULL;
	}
	// append new wire to wirelist
	netwire.Append(w, w->getID());
	return w;
}

// no wires with single nodes allowed
int Circuit::wireOK(QPainter *p, XWire *w)
{
	XWire *w2;

	if (w->countNodes() < 2) {
		w->lockLastNode();
		// erase last locked node
		w->erase(p);
		w2 = w->removeNode();
		if (w2) wireOK(p, w2);	// recursive call
		// remove nodeless wire
		netwire.Destroy(w);
		delete w;
		return 0;
	}
	return 1;
}

// wire released after clicked with the mouse
int Circuit::releaseWire(QPainter *p, XWire *w)
{
	w->Wire::garbageCollection();

	// check, that wire has at leat two nodes
	if (!wireOK(p, w)) return 0;
	w->releaseNode();
	return 1;
}

// delete wire
void Circuit::deleteWire(XWire *w)
{
	netwire.Destroy(w);
	delete w;
}

//*************************************************************************
//* calculation methods
//*************************************************************************

// set net to init-state (flush all calculation-fifos)
void Circuit::flush()
{
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter)
		iter.getDevice()->flush(0);
}

// calculate burst
// usually, burst will be 1 that means single calculation and propagation
// delays are reduced until no delay is left when burst is greater than 1
void Circuit::Burst(int burst)
{
/*
	Performance p;
	m_iCalcCalls = 0;
	m_iPropCalls = 0;
	p.start();
*/
	// calculate circuit <burst> times
	for (int burst_step = 0; burst_step < burst; ++burst_step) {
		Calculate(burst_step);
		Propagate(burst_step);
	}
/*
	p.end();
	cout << "measurement complete:" << endl;
	cout << "\t" << m_iCalcCalls << " calls to ::Calculate" << endl;
	cout << "\t" << m_iPropCalls << " calls to ::Propagate" << endl;
	cout << "\t" << "time (ms) " << p.getMilliSeconds() << endl;
*/
}

// switch input- or output devices of this
void Circuit::switchInterface(int if_type)
{
	// set output-values of input-devices by requesting named inputs of this sub circuit
	// input devices are part of this, and connected to named inputs of this
	if (if_type == IF_INPUT) {
		for (DeviceList::iterator iter = m_oInputList.begin(); iter != m_oInputList.end(); ++iter) {
			XDevice *oInputDev = iter.getDevice();
			// set input device from the named input value
			bool bIsBoolean;
			int iVal = input(oInputDev->getID(), bIsBoolean);
			oInputDev->setInputValue(iVal);
			oInputDev->setHasBooleanOutput(0, bIsBoolean);
		}
	}

	// set output values of named outputs by requesting output devices
	// output devices are part of this, and named outputs are connected to them
	if (if_type == IF_OUTPUT) {
		for (DeviceList::iterator iter = m_oOutputList.begin(); iter != m_oOutputList.end(); ++iter) {
			XDevice *oOutputDev = iter.getDevice();
			// set the named output value from output device
			setOutputValue(oOutputDev->getID(), oOutputDev->output());
			setHasBooleanOutput(oOutputDev->getID(), oOutputDev->hasBooleanOutput(0));
		}
	}
}

// virtual
// calculate all devices of this circuit and all sub circuits
void Circuit::Calculate(int burst_step)
{
	m_iCalcCalls++;
	switchInterface(IF_INPUT);
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		m_iCalcCalls++;
		iter.getDevice()->Calculate(burst_step);       // recursive by virtual
	}
}

// virtual
// propagate values to the outputs of this circuit and all sub circuits
void Circuit::Propagate(int burst_step)
{
	m_iPropCalls++;
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		m_iPropCalls++;
		iter.getDevice()->Propagate(burst_step);	// recursive by virtual
	}
	switchInterface(IF_OUTPUT);
}

/***************************************************/
/* import/export operations on a net-widget        */
/***************************************************/
// XML export method for library devices: export to XmlDevice
bool Circuit::exportXML(XmlNet& net)
{
	return exportXML((QTextStream *)NULL, &net, false, 0, 0, 0);
}

// XML export method for files: write directly to the file
bool Circuit::exportXML(QTextStream &ts, bool selected, int level, int dx, int dy)
{
	// top level circuit with selected flag ignored!
	XmlNet net(this, false, level, dx, dy);
	return exportXML(&ts, &net, selected, level, dx, dy);
}

// XML export method
bool Circuit::exportXML(QTextStream *ts, XmlNet *net, bool selected, int level, int dx, int dy)
{
	bool ownNet = (net == (XmlNet *)NULL);

	// circuit parameter and header
	if (ownNet) {
		net = new XmlNet(this, selected, level, dx, dy);
		if (!net->isValid()) {
			delete net;
			return false;
		}
	}

	if (ts) {
		*ts << net->getFieldStart();
		*ts << net->getFieldContent();
		level++;
	}

	// circuit wires
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		XmlWire wire(lw->Get(), selected, level, dx, dy);
		if (wire.isValid()) {
			if (ts) *ts << wire.getField();
			else net->addContent(wire.getField());
		}

		lw = lw->Next();
	}

	// circuit devices
	for (DeviceList::iterator iter = m_oDeviceList.begin(); iter != m_oDeviceList.end(); ++iter) {
		Circuit *poCircuit = iter.getDevice()->devIsCircuit();
		if (poCircuit) {
			XmlNet subCircuit(poCircuit, selected, level, dx, dy);
			if (subCircuit.isValid()) {
				// for sub-circuits always set selected=false!
				poCircuit->exportXML(ts, &subCircuit, false, level, 0, 0);	// recursion!

				// add net content (if ts avail: content already added)
				if (!ts) net->addContent(subCircuit.getField());
			}
		} else {
			XmlDevice dev(iter.getDevice(), selected, level, dx, dy);
			if (dev.isValid()) {
				if (ts) *ts << dev.getField();
				else net->addContent(dev.getField());
			}
		}
	}

	// circuit wire <-> wire connections
	lw = netwire.First();
	while(lw) {
		XmlWireConnection wire_conn(lw->Get(), selected, level, dx, dy);
		if (wire_conn.isValid()) {
			if (ts) *ts << wire_conn.getField();
			else net->addContent(wire_conn.getField());
		}

		lw = lw->Next();
	}

	// circuit device <-> wire connections
	lw = netwire.First();
	while(lw) {
		XmlDeviceConnection dev_conn(lw->Get(), selected, level, dx, dy);
		if (dev_conn.isValid()) {
			if (ts) *ts << dev_conn.getField();
			else net->addContent(dev_conn.getField());
		}

		lw = lw->Next();
	}

	if (ts) {
		level--;
		*ts << net->getFieldEnd();
	}

	if (ownNet) {
		delete net;
	}
	return true;
}

