#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <glib.h>

#include "kpsplitworkout.h"
#include "kppresetdataitem.h"
#include "kppresetdata.h"
#include "kputil.h"

static gboolean   get_preset_data_from_xml            (const gchar *file);
static gboolean   copy_preset_data_to_homedir         (void);
static void       check_dotdir_and_create_if_needed   (void);
static gchar     *default_file                        (void);

static GSList *preset_data[KP_PRESET_DATA_N];

struct {
  guint     n;
  gchar    *title;
} preset_data_titles[] = {
  { 0, "sport" },
  { 1, "detail" },
  { 2, "exercise_type" }
};


static gboolean
kp_preset_data_get_list_n (const gchar *title, guint *n)
{
  guint i;

  g_return_val_if_fail (title != NULL, FALSE);

  for (i=0; i < KP_PRESET_DATA_N; i++) {
    if (strcmp (preset_data_titles[i].title, title) == 0) {
      *n = i;
      return TRUE;
    }
  }
  
  return FALSE;
}

/**
 * kp_preset_data_init:
 * @filename: The name of the file that contains preset data
 *
 * The function loads preset data from the disk to the memory.
 * 
 * Returns: TRUE if successful and FALSE otherwise.
 */
gboolean
kp_preset_data_init (const gchar *filename)
{
  gchar *file;
  
  file = NULL;

  if (filename != NULL && g_file_test (filename, G_FILE_TEST_EXISTS))
    get_preset_data_from_xml (filename);

  file = default_file ();
  if (g_file_test (file, G_FILE_TEST_EXISTS))
    get_preset_data_from_xml (file);
  else {
    check_dotdir_and_create_if_needed ();
    copy_preset_data_to_homedir ();
    get_preset_data_from_xml (file);
  }

  g_free (file);
  
  return TRUE; 
}


/**
 * kp_preset_data_deinit:
 * 
 * Cleanup all stuff when preset data isn't needed anymore.
 */
void
kp_preset_data_deinit (void)
{
  guint i;
  for (i=0; i < KP_PRESET_DATA_N; i++) {
    kp_preset_data_clean (i);
    preset_data[i] = NULL;
  }
}


/**
 * kp_preset_data_write:
 * @file: The name of the file to write things, can be NULL
 *
 * Write preset data to the disk.
 */
void
kp_preset_data_write (const gchar *file)
{
  xmlNodePtr list;
  xmlNodePtr section;
  xmlNodePtr item;
  xmlDocPtr doc;
  GSList *node;
  gchar *lfile = NULL;
  guint i;
    
  lfile = (file) ? g_strdup (file) :  default_file ();
  
  kp_debug ("Writing preset data to file: %s!", lfile);
  
  doc = xmlNewDoc (BAD_CAST ("1.0"));

  list = xmlNewNode (NULL, BAD_CAST ("presetlist"));
  (void) xmlAddChild ((xmlNodePtr) doc, list);
 
  for (i=0; i < KP_PRESET_DATA_N; i++) {
    section = xmlNewNode (NULL, BAD_CAST ("section"));
    (void) xmlAddChild (list, section);
    (void) xmlNewChild (section, NULL, BAD_CAST ("title"), 
                                       BAD_CAST (preset_data_titles[i].title));
  
    for (node = preset_data[i]; node; node = node->next) {
      item = kp_preset_data_item_to_xml (KP_PRESET_DATA_ITEM (node->data));
      if (item) 
        (void) xmlAddChild (section, item);
      else 
        g_warning ("Couldn't transform item to XML: %p", node->data);
    }
  }

  xmlSaveFormatFile (lfile, doc, 1);
  xmlFreeDoc (doc);  
  
  if (lfile)
    g_free (lfile);
}


/**
 * kp_preset_data_clean:
 * @type: A #KPPresetDataType
 *
 * Remove all data from memory.
 */
void
kp_preset_data_clean (KPPresetDataType type)
{
  GSList *list;
  
  g_return_if_fail (type < KP_PRESET_DATA_N);

  for (list = preset_data[type]; list; list = list->next)
    kp_preset_data_item_free (KP_PRESET_DATA_ITEM (list->data));
  
  g_slist_free (preset_data[type]);
  preset_data[type] = NULL;
}


GSList *
kp_preset_data_get_items (KPPresetDataType type)
{
  g_return_val_if_fail (type < KP_PRESET_DATA_N, NULL);

  return preset_data[type];
}


KPPresetDataItem *
kp_preset_data_get_item (KPPresetDataType type, const gchar *name)
{
  GSList *list;
  gchar *str1, *str2;

  g_return_val_if_fail (type < KP_PRESET_DATA_N, NULL);
  g_return_val_if_fail (name != NULL, NULL);
  
  for (list = preset_data[type]; list; list = list->next) {
    str1 = g_utf8_strdown (KP_PRESET_DATA_ITEM (list->data)->name, -1);
    str2 = g_utf8_strdown (name, -1);
        
    if (g_utf8_collate (str1, str2) == 0) {
      g_free (str1);
      g_free (str2);
      
      return KP_PRESET_DATA_ITEM (list->data);
    }
    g_free (str1);
    g_free (str2);
  }
  return NULL;
}


void
kp_preset_data_add_item (KPPresetDataType type, KPPresetDataItem *item)
{
  KPPresetDataItem *old;
  GSList *list;
  
  g_return_if_fail (item != NULL);
  g_return_if_fail (item->name != NULL);
  g_return_if_fail (type < KP_PRESET_DATA_N);
 
  kp_debug ("Add item, title: %s\n", item->name);

  old = kp_preset_data_get_item (type, item->name);
  if (old != NULL)
    return;
  
  preset_data[type] = g_slist_prepend (preset_data[type], item);
}


static gboolean
import_section (const gchar *section, xmlNodePtr node)
{
  KPPresetDataItem *item;
  guint n;
  xmlNodePtr ptr;

  if (!kp_preset_data_get_list_n (section, &n)) {
    g_warning ("Unknown section: %s", section);
    return FALSE;
  }
  
  for (ptr = node->children; ptr; ptr = ptr->next) {
    if (strcmp ((const gchar *) ptr->name, "text") != 0
     && strcmp ((const gchar *) ptr->name, "title") != 0) {

      item = kp_preset_data_item_new_from_xml (ptr);

      if (item) {
        preset_data[n] = g_slist_prepend (preset_data[n], item);
      } else
        g_warning ("Preset data item is not valid! section = %s!", section);
    }
  }
  return TRUE;
}


static gchar *
get_section_title (xmlNodePtr section_node)
{
  xmlNodePtr ptr;

  for (ptr = section_node->children; ptr; ptr = ptr->next)
    if (strcmp ((const gchar *) ptr->name, "title") == 0) {
      return (gchar *) xmlNodeGetContent (ptr);
    }
  return NULL;
}


/**
 * kp_preset_data_import_tree:
 * @node: Node that represents <presetlist> 
 *
 * Loads the preset data into memory.
 *
 * Returns: TRUE if succesful and FALSE otherwise. 
 */
gboolean
kp_preset_data_import_tree (xmlNodePtr node)
{
  g_return_val_if_fail (node != NULL, FALSE);
  gchar *title;
  
  if (KP_TAG_MATCH (node, "presetlist") == FALSE)
    return FALSE;
  
  for (node = node->children; node; node = node->next) {
    if (KP_TAG_MATCH (node, "section")) {
      /* Try to insert data items to list or return FALSE
       * if an error occurs */
      title = get_section_title (node);
      if (title) {   
        if (!import_section (title, node)) {
          g_free (title);
          return FALSE;
        }
        g_free (title);
      } else {
        g_warning ("Can't find the title node for section!");
        return FALSE;
      }
    }
    else if (KP_TAG_MATCH (node, "text")) {
      /* Its OK to have text nodes.. */
    }
    else {
      g_warning ("Unknown tag: %s\n", node->name);
    }
  }

  return TRUE;
}

static gboolean
get_preset_data_from_xml (const gchar *file)
{
  xmlNodePtr node;
  xmlDocPtr doc;
  gboolean ret_val = FALSE;
 
  kp_debug ("Parsing started.");
  
  doc = xmlParseFile (file);
  if (!doc)
    goto err;

  node = xmlDocGetRootElement (doc);
  if (!node || !node->name || !KP_TAG_MATCH (node, "presetlist"))
    goto err;

  if (!kp_preset_data_import_tree (node)) {
    g_warning ("Importing tree failed!");
    goto err;
  }
    
  kp_debug ("Parsing finished.");
 
  ret_val = TRUE;
  
err:
  if (ret_val == FALSE)
    g_warning ("Parse error, couldn't parse file %s!", file);
  
  if (doc) { 
    xmlFreeDoc (doc);
  }
  
  xmlCleanupParser ();
 
  return ret_val;
}


/* JUST DEAL WITH THE CONFIG FILE ... */

static gchar *
default_file (void)
{ 
  return g_build_filename (G_DIR_SEPARATOR_S,
                           g_get_home_dir (),
                          ".kipina",
                           G_DIR_SEPARATOR_S,
                          "presetdata.xml",
                           NULL);
}

static void
check_dotdir_and_create_if_needed (void)
{
  gchar *dirname;
  int retval;

  dirname = g_build_path (G_DIR_SEPARATOR_S, g_get_home_dir (), ".kipina",
                          NULL);

  if (g_file_test (dirname, G_FILE_TEST_IS_DIR)) 
    return;
  else {
    retval = mkdir (dirname, 0777);

    if (retval == 0)
      return;

    g_warning (strerror (errno));
  }
}


static gboolean
copy_preset_data_to_homedir (void)
{
  pid_t pid;
  int ret;
  gchar *arg_list[4] = { "cp", NULL, NULL, NULL };

  arg_list[1] = g_build_filename (G_DIR_SEPARATOR_S, KIPINA_SYSCONF_DIR,
                                 "kipina-presetdata.xml", NULL);
  arg_list[2] = default_file ();
 
  pid = fork ();
  if (pid != 0)
    return TRUE;
  else {
    ret = execvp ("cp", arg_list);

    /* An error occurred! */
    fprintf (stderr, "Can't copy file: %s\n", strerror (errno));

    abort ();
  }

  g_free (arg_list[1]);
  g_free (arg_list[2]);

  return TRUE;
}


