#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kipina-i18n.h"
#include "../kputil.h"

#include "kpnewsplitworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpviewpopupmodel.h"
#include "kpentryviewmodel.h"
#include "kpworkouteditor.h"
#include "kpcontexts.h"
#include "kpguiutils.h"
#include "kplogstore.h"
#include "kptreeview.h"
#include "kpentries.h"

/* Callbacks */
static void       kp_tree_view_init             (KPTreeView *tv);

/* KPViewPopupModel implementation */
static void       kp_tree_view_popup_model_init (KPViewPopupModelIface *iface);
KPCalendarEntry  *kp_tree_view_get_active_entry (KPViewPopupModel *popup_model);

/* KPViewModel implementation */
static void       kp_tree_view_model_init       (KPViewModelIface *iface);
static void       kp_tree_view_activate         (KPViewModel *model);
static void       kp_tree_view_deactivate       (KPViewModel *model);
static gchar     *kp_tree_view_get_icon_name    (KPViewModel *view);
static void       kp_tree_view_set_view_type    (KPViewModel *view,
                                                 KPViewModelType type);
static  
KPViewModelType   kp_tree_view_get_view_type    (KPViewModel *view);
static void       kp_tree_view_set_dmy          (KPViewModel *model, 
                                                 guint d, 
                                                 guint m, 
                                                 guint y);
static void       kp_tree_view_get_dmy          (KPViewModel *model,
                                                 guint *d,
                                                 guint *m,
                                                 guint *y);
static void       kp_tree_view_set_log          (KPViewModel *model,
                                                 KPTrainingLog *log);
static void       kp_tree_view_unset_log        (KPViewModel *model);

/* Callback functions */
static void       view_popup_menu               (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 gpointer data);
static gboolean   on_popup_menu                 (GtkWidget *treeview,
                                                 gpointer data);
static gboolean   on_button_press_event         (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 gpointer data);
/* Follow log changes.. */
static void       log_connect_signals           (KPTrainingLog *log,
                                                 KPTreeView *tv);
static void       log_disconnect_signals        (KPTrainingLog *log,
                                                 KPTreeView *tv);
static void       log_changed                   (KPTrainingLog *log,
                                                 gpointer data);
static void       log_entry_removed             (KPTrainingLog *log,
                                                 guint d, guint m, guint y,
                                                 const gchar * mark_str,
                                                 KPTreeView *tv);
static void       log_entry_added               (KPTrainingLog *log,
                                                 KPCalendarEntry *entry,
                                                 KPTreeView *tv);
static gboolean   kp_tree_view_get_date         (KPTreeView *tv, 
                                                 KPDate *date);

typedef struct KPTreeViewPrivateData_
{
  KPEntryPopup     *popup_data;
  
  KPTrainingLog    *log;
} KPTreeViewPrivateData;

#define KP_TREE_VIEW_PRIVATE_DATA(widget) (((KPTreeViewPrivateData*) \
      (KP_TREE_VIEW (widget)->private_data)))


GType
kp_tree_view_get_type (void)
{
  static GType        kp_tree_view_type = 0;

  if (!kp_tree_view_type) {
    static const GTypeInfo kp_tree_view_info = {
      sizeof (KPTreeViewClass),
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      sizeof (KPTreeView),
      0,
      (GInstanceInitFunc) kp_tree_view_init,
      NULL
    };
    static const GInterfaceInfo view_popup_model_info = {
      (GInterfaceInitFunc) kp_tree_view_popup_model_init,
      NULL,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_tree_view_model_init,
      NULL,
      NULL
    };
    kp_tree_view_type = g_type_register_static (GTK_TYPE_TREE_VIEW,
                                               "KPTreeView",
                                               &kp_tree_view_info, 0);
    g_type_add_interface_static (kp_tree_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);
    g_type_add_interface_static (kp_tree_view_type,
                                 KP_TYPE_VIEW_POPUP_MODEL,
                                &view_popup_model_info);
  }
  return kp_tree_view_type;
}


static void
kp_tree_view_popup_model_init (KPViewPopupModelIface *iface)
{
  iface->get_active_entry = kp_tree_view_get_active_entry;
}


enum {
  COLUMN_ICON,
  COLUMN_TEXT
};

static void
kp_tree_view_init (KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  GtkTreeViewColumn *col;
  GtkCellRenderer *ren;
  
  tv->private_data = g_new (KPTreeViewPrivateData, 1);
  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);
  p_data->popup_data = kp_entries_get_popup (KP_VIEW_POPUP_MODEL (tv));
  
  g_object_set (G_OBJECT (tv), "headers-visible", FALSE, NULL);

  ren = gtk_cell_renderer_pixbuf_new ();
  col = gtk_tree_view_column_new ();
  gtk_tree_view_column_pack_start (col, ren, FALSE);
  gtk_tree_view_column_set_attributes (col, ren, "pixbuf", COLUMN_ICON, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tv), col);
  
  ren = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new ();
  gtk_tree_view_column_pack_start (col, ren, TRUE);
  gtk_tree_view_column_set_attributes (col, ren, "markup", COLUMN_TEXT, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tv), col);
  
  g_signal_connect (G_OBJECT (tv), "button-press-event",
                    G_CALLBACK (on_button_press_event), NULL);

  g_signal_connect (G_OBJECT (tv), "popup-menu",
                    G_CALLBACK (on_popup_menu), NULL);
}

  
static void
kp_tree_view_model_init (KPViewModelIface *iface)
{
  iface->get_dmy = kp_tree_view_get_dmy;
  iface->set_dmy = kp_tree_view_set_dmy;
  iface->set_log = kp_tree_view_set_log;
  iface->unset_log = kp_tree_view_unset_log;
  iface->set_view_type = kp_tree_view_set_view_type;
  iface->get_view_type = kp_tree_view_get_view_type;
  iface->get_icon_name = kp_tree_view_get_icon_name;
  iface->activate = kp_tree_view_activate;
  iface->deactivate = kp_tree_view_deactivate;
}


GtkWidget *
kp_tree_view_new (KPTrainingLog *log)
{
  KPTreeViewPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_tree_view_get_type (), NULL);

  p_data = KP_TREE_VIEW_PRIVATE_DATA (widget);
  p_data->log = log;
  
  if (p_data->log)
    log_connect_signals (p_data->log, KP_TREE_VIEW (widget));

  gtk_widget_show_all (GTK_WIDGET (widget));
  
  return widget;
}

void
kp_tree_view_set_log (KPViewModel *model, KPTrainingLog *log)
{
  KPTreeViewPrivateData *p_data;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  
  p_data = KP_TREE_VIEW_PRIVATE_DATA (model);
  p_data->log = log;
  
  log_connect_signals (log, KP_TREE_VIEW (model));
}

void
kp_tree_view_unset_log (KPViewModel *model)
{
  KPTreeViewPrivateData *p_data;
  p_data = KP_TREE_VIEW_PRIVATE_DATA (model);

  log_disconnect_signals (p_data->log, KP_TREE_VIEW (model));
  p_data->log = NULL;
}


static void
kp_tree_view_activate (KPViewModel *model) 
{
  /* nothing */
}


static void
kp_tree_view_deactivate (KPViewModel *model)
{
  /* nothing */
}


static gchar *
kp_tree_view_get_icon_name (KPViewModel *view)
{
  return g_strdup ("");
}


static void
kp_tree_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  /* nothing */
}


static KPViewModelType
kp_tree_view_get_view_type (KPViewModel *view)
{
  return KP_VIEW_MODEL_TYPE_ALL_TIME;
}


static void
kp_tree_view_set_dmy (KPViewModel *model, guint d, guint m, guint y)
{
  /* do nothing */   
}


static void
kp_tree_view_get_dmy (KPViewModel *model, guint *d, guint *m, guint *y)
{
  KPDate date;

  kp_tree_view_get_date (KP_TREE_VIEW (model), &date);
  
  if (d)
    *d = date.d;
  if (m)
    *m = date.m;
  if (y)
    *y = date.y;
}


KPCalendarEntry *
kp_tree_view_get_active_entry (KPViewPopupModel *popup_model)
{
  KPTreeViewPrivateData *p_data;
  KPCalendarEntry *entry = NULL;
  KPLogStoreRecord *rec;
  GtkTreeView *treeview = GTK_TREE_VIEW (popup_model);
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (popup_model);

  g_return_val_if_fail (GTK_IS_TREE_VIEW (treeview), NULL);
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    rec = kp_log_store_get_record (KP_LOG_STORE (model), &iter);
    
    if (rec->calendar_entry == FALSE) {
      return NULL;
    } else {
      entry = kp_training_log_get_entry (p_data->log, rec->d, rec->m, rec->y,
                                         rec->record);
    }
  }
  return entry;
}


static gboolean
kp_tree_view_get_date (KPTreeView *tv, KPDate *date)
{
  KPTreeViewPrivateData *p_data;
  GtkTreeView *treeview = GTK_TREE_VIEW (tv);
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    kp_log_store_get_date (KP_LOG_STORE (model), &iter,
                          &date->d, &date->m, &date->y);

    kp_debug ("Log store gives date: %u.%u.%u", date->d, date->m, date->y);
    
    if (g_date_valid_year (date->y && g_date_valid_month (date->m)
     && g_date_valid_day (date->d))) {
      return TRUE;
    } else if (g_date_valid_year (date->y)&& g_date_valid_month (date->m)) {
      date->d = 1;
      return TRUE;
    } else if (g_date_valid_year (date->y)) {
      date->d = 1;
      date->m = 1;
      return TRUE;
    }
  }
  
  kp_date_set_time (date, time (NULL));
  return TRUE;
}


static void
log_entry_removed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark_str, KPTreeView *tv)
{
  GtkTreeModel *store;
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_TREE_VIEW (tv));

  store = gtk_tree_view_get_model (GTK_TREE_VIEW (tv));
  g_return_if_fail (KP_IS_LOG_STORE (store));

  kp_debug ("Removing mark.");
  
  kp_log_store_remove_mark (KP_LOG_STORE (store), d, m, y, mark_str);
}


static void
log_entry_added (KPTrainingLog *log, KPCalendarEntry *entry, KPTreeView *tv)
{
  GtkTreeModel *store;
  const gchar *mark;
  KPDate date;
  
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_TREE_VIEW (tv));

  store = gtk_tree_view_get_model (GTK_TREE_VIEW (tv));
  g_return_if_fail (KP_IS_LOG_STORE (store));

  kp_calendar_entry_get_date (entry, &date);
 
  mark = kp_calendar_entry_to_string (entry);
  kp_log_store_add_mark (KP_LOG_STORE (store), date.d, date.m, date.y, mark);
  
  kp_debug ("%u.%u.%u (%s) added!", date.d, date.m, date.y, mark);
}
              

static void
log_changed (KPTrainingLog *log, gpointer data)
{
}
  
static void
log_connect_signals (KPTrainingLog *log, KPTreeView *tv)
{
  g_signal_connect (G_OBJECT (log), "entry-removed",
                    G_CALLBACK (log_entry_removed), tv);
  g_signal_connect (G_OBJECT (log), "new-entry-added",
                    G_CALLBACK (log_entry_added), tv);
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), tv);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPTreeView *tv)
{
  g_signal_handlers_disconnect_by_func (log, log_entry_removed, tv);
  g_signal_handlers_disconnect_by_func (log, log_entry_added, tv);
  g_signal_handlers_disconnect_by_func (log, log_changed, tv);
}


static gboolean
on_button_press_event (GtkWidget *treeview, GdkEventButton *event,
                       gpointer data)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  if (event->type != GDK_BUTTON_PRESS || event->button != 3)
    return FALSE;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    view_popup_menu (treeview, event, data);
  }

  return TRUE; /* we handled this */
}


static gboolean
on_popup_menu (GtkWidget *treeview, gpointer data)
{
  view_popup_menu (treeview, NULL, data);
  return TRUE; 
}


static void
view_popup_menu (GtkWidget *treeview, GdkEventButton *event, gpointer data)
{
  KPTreeViewPrivateData *p_data = KP_TREE_VIEW_PRIVATE_DATA (treeview);

  kp_entries_popup_prepare_dynamic (p_data->popup_data,
                                    KP_VIEW_POPUP_MODEL (treeview));
  
  gtk_menu_popup (GTK_MENU (p_data->popup_data->menu), NULL, NULL, NULL, NULL,
                  (event) ? event->button : 0,
                  gdk_event_get_time ((GdkEvent *) event));
}

