#include <string.h>
#include <time.h>

#include <gconf/gconf-client.h>
#include <gdk/gdkkeysyms.h>
#include <glade/glade.h>
#include <gtk/gtk.h>

#include "kpnewsplitworkoutdialog.h"
#include "kppreferencesdialog.h"
#include "kpnewcommentdialog.h"
#include "kphtmloutputdialog.h"
#include "kpresultsmanager.h"
#include "kpviewpopupmodel.h"
#include "kploginfodialog.h"
#include "kpresultseditor.h"
#include "kpworkouteditor.h"
#include "kpcalendarview.h"
#include "kpnavitreeview.h"
#include "kpmainwindow.h"
#include "kpchartview.h"
#include "kpstatusbar.h"
#include "kpviewmodel.h"
#include "kpstatsview.h"
#include "kptreeview.h"
#include "kplistview.h"
#include "kplogstore.h"
#include "kpguiutils.h"
#include "kpentries.h"
#include "kpmenubar.h"
#include "kpsidebar.h"
#include "kpchart.h"
#include "kpview.h"

#include "../kptraininglog.h"
#include "../kppresetdata.h"
#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kpcomment.h"
#include "../kpworkout.h"
#include "../kpplugin.h"
#include "../kputil.h"

/* Static functions */
static void     kp_main_window_deinit_log       (KPMainWindow *window);
static void     kp_main_window_class_init       (KPMainWindowClass *klass);
static void     kp_main_window_init             (KPMainWindow *dialog);
static void     kp_main_window_finalize         (GObject *object);

static void     kp_main_window_update_date_text (KPMainWindow *window);
static void     calendar_view_day_selected      (KPCalendarView *cv,
                                                 KPDate *date,
                                                 KPMainWindow *window);
static void     log_connect_signals             (KPTrainingLog *log,
                                                 KPMainWindow *window);
static void     log_disconnect_signals          (KPTrainingLog *log,
                                                 KPMainWindow *window);
static void     log_entry_removed               (KPTrainingLog *log, guint d,
                                                 guint m, guint y,
                                                 const gchar *mark,
                                                 KPMainWindow *window);
static void     log_entry_added                 (KPTrainingLog *log,
                                                 KPCalendarEntry *entry,
                                                 KPMainWindow *window);
static void     log_changed                     (KPTrainingLog *log,
                                                 KPMainWindow *window);
static void     log_saved                       (KPTrainingLog *log,
                                                 KPMainWindow *window);

/* Callbacks for KPMainWindow */
static gboolean on_main_window_close            (GtkWidget *widget,
                                                 GdkEvent *event,
                                                 KPMainWindow *window);
static gboolean main_window_key_press           (GtkWidget *widget,
                                                 GdkEventKey *key,
                                                 KPMainWindow *window);
static void     on_view_type_combo_changed_cb   (GtkComboBox *box, 
                                                 KPMainWindow *window);
static void     on_time_period_combo_changed_cb (GtkComboBox *box,
                                                 KPMainWindow *window);
static void     workout_tree_selection_changed  (GtkTreeSelection *selection,
                                                 KPMainWindow *window);
static void     on_paned_handle_move            (GObject *pane,
                                                 GParamSpec *spec,
                                                 KPMainWindow *window);
static void     next_button_cb                  (GtkButton *button,
                                                 KPMainWindow *window);
static void     prev_button_cb                  (GtkButton *button,
                                                 KPMainWindow *window);
static void     update_view_type_buttons_state_cb
                                                (KPViewModel *model, 
                                                 KPViewModelType type,
                                                 KPMainWindow *window);
static void     kp_view_view_set                (KPViewModel *model,
                                                 KPViewModelType type,
                                                 KPMainWindow *window);
static void     kp_view_date_set                (KPViewModel *model,
                                                 KPDate *date,
                                                 KPMainWindow *window);
static void     kp_view_viewer_set              (KPView *view,
                                                 KPViewModel *model,
                                                 KPMainWindow *window);
static void     update_toolbar_state            (KPMainWindow *window);


typedef struct KPMainWindowPrivateData_
{
  KPTrainingLog *log;
  GString *window_title;
  GString *logfile;

  KPCalendarView *cv;
  GtkWidget *tree;
  GtkWidget *date_label;
  GtkWidget *sidebar;

  GtkWidget *view;
  KPStatusbar *sbar;
 
  GtkWidget *toolbar_log;
  GtkWidget *toolbar_new_button;
  GtkWidget *toolbar_open_button;
  GtkWidget *toolbar_save_button;

  GtkWidget *paned_window;

  GtkWidget *menu_file_separator;
  GtkWidget *menu_import;
  GtkWidget *menu_export;
  GtkWidget *menubar;

  GtkWidget *time_period_combo;
  GtkWidget *view_type_combo;
  GtkWidget *toolbar_next_button;
  GtkWidget *toolbar_prev_button;

  GConfClient *client;
} KPMainWindowPrivateData;

#define KP_MAIN_WINDOW_PRIVATE_DATA(widget) ((KPMainWindowPrivateData*) \
        (KP_MAIN_WINDOW (widget)->private_data))

static GObjectClass *parent_class = NULL;
static KPMainWindow *_window = NULL;

GType
kp_main_window_get_type (void)
{
  static GType kp_main_window_type = 0;

  if (kp_main_window_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPMainWindowClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_main_window_class_init,
      NULL,
      NULL,
      sizeof (KPMainWindow),
      0,
      (GInstanceInitFunc) kp_main_window_init,
      NULL,
    };

    kp_main_window_type = g_type_register_static (GTK_TYPE_WINDOW,
                                                 "KPMainWindow",
                                                  &our_info, 0);
  }

  return kp_main_window_type;
}


static void
kp_main_window_class_init (KPMainWindowClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_main_window_finalize;
}

#define KEY_WINDOW_GEOMETRY   "/apps/kipina/preferences/save_window_geometry"
#define KEY_WINDOW_WIDTH      "/apps/kipina/preferences/window_width"
#define KEY_WINDOW_HEIGHT     "/apps/kipina/preferences/window_height"
#define KEY_HTML_OUTPUT_DIR   "/apps/kipina/preferences/html_output_dir"
#define KEY_PANED_WIDTH       "/apps/kipina/preferences/paned_width"

static void
kp_main_window_init (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkWidget *vbox;
  GtkWidget *sw_view;
  GtkWidget *navi;
  GdkPixbuf *icon;
  GladeXML *xml;
  gchar *file;
  gint idx;
  gint tper;
  gchar *vtype;
  gint width = -1;
  gint height = -1;
  
  xml = kp_gui_load ("main_window", "mainwin_vbox");
  
  window->private_data = g_new0 (KPMainWindowPrivateData, 1);
  _window = window;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  p_data->client = gconf_client_get_default ();
  p_data->window_title = g_string_new (NULL);
  p_data->logfile = g_string_new (NULL);
  p_data->log = NULL;
  p_data->paned_window = KP_W (xml, "h_paned");
   
  vbox = KP_W (xml, "mainwin_vbox");
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_box_pack_start (GTK_BOX (KP_W (xml, "menubar_box")), 
                      kp_menu_bar_new (&p_data->menu_import, &p_data->toolbar_log),
                      TRUE, TRUE, 0);

  p_data->sbar = KP_STATUSBAR (kp_statusbar_new (NULL));
  gtk_box_pack_end (GTK_BOX (vbox), GTK_WIDGET (p_data->sbar), FALSE, TRUE, 0);
  if (gconf_client_get_bool (p_data->client, KP_CONF_SHOW_STATUSBAR, NULL))
    gtk_widget_show (GTK_WIDGET (p_data->sbar));

  p_data->sidebar = kp_sidebar_new ();
  p_data->tree = kp_tree_view_new (NULL);
  p_data->view = GTK_WIDGET (kp_view_new (0, 0, 0));

  sw_view = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw_view),
                                  GTK_POLICY_AUTOMATIC, 
                                  GTK_POLICY_AUTOMATIC);
      
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw_view),
                                         GTK_WIDGET (p_data->view));
  gtk_widget_show (sw_view);
  gtk_widget_show (p_data->tree);

  p_data->cv = KP_CALENDAR_VIEW (kp_calendar_view_new (KP_VIEW_MODEL_TYPE_DAY, 
                                 0, 0, 0));
  g_object_set (G_OBJECT (p_data->cv), "line-spacing", 2, NULL);
  
  gtk_widget_set_sensitive (GTK_WIDGET (p_data->cv), TRUE);

  kp_view_add_viewer (KP_VIEW (p_data->view),
                    _("Calendar"),
                      KP_VIEW_MODEL (p_data->cv));
  kp_view_add_viewer (KP_VIEW (p_data->view),
                    _("Statistics"),  
                      KP_VIEW_MODEL (kp_stats_view_new ()));
  kp_view_add_viewer (KP_VIEW (p_data->view),
                    _("List"),
                      KP_VIEW_MODEL (kp_list_view_new ()));
  kp_view_add_viewer (KP_VIEW (p_data->view),
                    _("Chart"),
                      KP_VIEW_MODEL (kp_chart_view_new ()));

  g_signal_connect (G_OBJECT (p_data->view), "view-set",
                    G_CALLBACK (kp_view_view_set), window);
  g_signal_connect (G_OBJECT (p_data->view), "date-set",
                    G_CALLBACK (kp_view_date_set), window);
  g_signal_connect (G_OBJECT (p_data->view), "viewer-set",
                    G_CALLBACK (kp_view_viewer_set), window);

  /* Remember last time period */  
  tper = gconf_client_get_int (p_data->client, KP_CONF_ACTIVE_TIME_PERIOD, NULL);
  kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view), (tper == 0) ? KP_VIEW_MODEL_TYPE_MONTH : tper - 1);
  gtk_combo_box_set_active (GTK_COMBO_BOX (KP_W (xml, "time_period_combo")), tper-1);
  
  /* Remember last viewer */
  vtype = gconf_client_get_string (p_data->client, KP_CONF_ACTIVE_VIEW_TYPE, NULL);
  kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), (vtype) ? vtype : "KPCalendarView");
  
  /* Get current viewer real name :-) */
  vtype = g_object_get_data (G_OBJECT (kp_view_get_current_viewer (KP_VIEW (p_data->view))), "name");
  idx = kp_gui_get_combo_box_index (GTK_COMBO_BOX (KP_W (xml, "view_type_combo")), vtype);
  gtk_combo_box_set_active (GTK_COMBO_BOX (KP_W (xml, "view_type_combo")), idx);
  
  kp_statusbar_set_viewer_name (p_data->sbar,
    G_OBJECT_TYPE_NAME (kp_view_get_current_viewer (KP_VIEW (p_data->view))));

  if (GTK_WIDGET_CAN_FOCUS (GTK_WIDGET (p_data->cv)))
    gtk_widget_grab_focus (GTK_WIDGET (p_data->cv));

  navi = kp_navi_tree_view_new ();
  
  gtk_paned_pack1 (GTK_PANED (p_data->paned_window),
                   GTK_WIDGET (p_data->sidebar), TRUE, TRUE);
  gtk_paned_pack2 (GTK_PANED (p_data->paned_window), GTK_WIDGET (sw_view),
                   TRUE, TRUE);
  
  gtk_widget_show (p_data->sidebar);
  kp_sidebar_add_widget (KP_SIDEBAR (p_data->sidebar), navi, "Navi", FALSE);
  kp_sidebar_add_widget (KP_SIDEBAR (p_data->sidebar), p_data->tree, "Entry tree", TRUE);
  gtk_widget_show (GTK_WIDGET (p_data->paned_window));
  
  /* Put widgets to private data */
  p_data->date_label = KP_W (xml, "date_label");
  p_data->toolbar_open_button = KP_W (xml, "toolbar_open_button");
  p_data->toolbar_save_button = KP_W (xml, "toolbar_save_button");
  p_data->toolbar_new_button = KP_W (xml, "toolbar_new_button");
  
  p_data->time_period_combo = KP_W (xml, "time_period_combo");
  p_data->view_type_combo = KP_W (xml, "view_type_combo");

  p_data->toolbar_next_button = KP_W (xml, "toolbar_next_button");
  p_data->toolbar_prev_button = KP_W (xml, "toolbar_prev_button");
 
  g_signal_connect (G_OBJECT (p_data->toolbar_next_button), "clicked",
                    G_CALLBACK (next_button_cb), window);
  g_signal_connect (G_OBJECT (p_data->toolbar_prev_button), "clicked",
                    G_CALLBACK (prev_button_cb), window);
  g_signal_connect (G_OBJECT (p_data->view), "view-set",
                    G_CALLBACK (update_view_type_buttons_state_cb), window);
  g_signal_connect (G_OBJECT (p_data->view_type_combo), "changed",
                    G_CALLBACK (on_view_type_combo_changed_cb), window);
  g_signal_connect (G_OBJECT (p_data->time_period_combo), "changed",
                    G_CALLBACK (on_time_period_combo_changed_cb), window);
    
  g_return_if_fail (GTK_IS_TREE_VIEW (p_data->tree));
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->tree));
  gtk_widget_add_events (GTK_WIDGET (window), GDK_KEY_PRESS_MASK);
  g_return_if_fail (selection != NULL);

  kp_debug ("Key \"%s\" is %s, show sidebar.\n", KP_CONF_SHOW_SIDEBAR,
      (gconf_client_get_bool (p_data->client, KP_CONF_SHOW_SIDEBAR, NULL) ? "true" : "false"));
  
  if (gconf_client_get_bool (p_data->client, KP_CONF_SHOW_SIDEBAR, NULL)) {
    
    width = gconf_client_get_int (p_data->client, KEY_PANED_WIDTH, NULL);

    if (gconf_client_get_bool (p_data->client, KEY_WINDOW_GEOMETRY, NULL)
        && width > 0) {
      gtk_widget_set_size_request (GTK_WIDGET (p_data->sidebar), width, -1);
      gtk_paned_set_position (GTK_PANED (p_data->paned_window), width);
    }

    gtk_widget_show (GTK_WIDGET (p_data->sidebar));
  }
  else 
    gtk_widget_hide (GTK_WIDGET (p_data->sidebar));

  file = g_build_filename (KIPINA_PIXMAP_DIR, "runner.xpm", NULL);
  if ((icon = gdk_pixbuf_new_from_file (file, NULL)))
    gtk_window_set_icon (GTK_WINDOW (window), icon);
  g_free (file);
 
  if (gconf_client_get_bool (p_data->client, KEY_WINDOW_GEOMETRY, NULL)) {
    width = gconf_client_get_int (p_data->client, KEY_WINDOW_WIDTH, NULL);
    height = gconf_client_get_int (p_data->client, KEY_WINDOW_HEIGHT, NULL);
    
    if (width > 0 && height > 0)
      gtk_window_set_default_size (GTK_WINDOW (window), width, height);
  }
    
  g_signal_connect (G_OBJECT (p_data->cv), "day-selected",
                    G_CALLBACK (calendar_view_day_selected), window);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (workout_tree_selection_changed), window);
  g_signal_connect (G_OBJECT (window), "key_press_event",
                    G_CALLBACK (main_window_key_press), window);
  g_signal_connect (G_OBJECT (p_data->paned_window), "notify::position", 
                    G_CALLBACK (on_paned_handle_move), window);
 
  kp_main_window_update_date_text (window);
  kp_menu_bar_set_toggle_states ();
  
  
  g_object_unref (G_OBJECT (xml));

  g_signal_connect (G_OBJECT (window), "delete_event",
                    G_CALLBACK (on_main_window_close), window);

  kp_entries_init ();
}


static void
kp_main_window_finalize (GObject *object)
{
  KPMainWindow *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_MAIN_WINDOW (object));

  dialog = KP_MAIN_WINDOW (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


/**
 * kp_main_window_new:
 * @file: Name of the log file to be opened or NULL.
 *
 * Just create a new kipinämain window.
 *
 * Returns: New KPMainWindow widget
 */
GtkWidget *
kp_main_window_new (void)
{
  KPMainWindow *window;
 
  if (_window)
    return NULL;
  
  window = g_object_new (KP_TYPE_MAIN_WINDOW, NULL);

  kp_plugin_load_plugins ();
  update_toolbar_state (window);
 
  return GTK_WIDGET (window);
}


static void
update_view_type_buttons_state_cb (KPViewModel *model, KPViewModelType type,
                                   KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  gtk_combo_box_set_active (GTK_COMBO_BOX (p_data->time_period_combo), (guint) type);
}


void
kp_main_window_set_log_file (KPMainWindow *window, const gchar *file)
{
  kp_debug ("File: %p", file);
  kp_main_window_init_log (window, file);
  update_toolbar_state (window);
}


static void
update_window_title (KPMainWindow *window)
{
  KPTrainingLog *log;
  const gchar *str;
  gchar buf[256];
  gchar *name = NULL;

  log = kp_main_window_get_log ();
  g_return_if_fail (log != NULL);
  
  if ((str = kp_training_log_get_filename (log)) != NULL)
    name = g_filename_display_basename (str);
  else
    str = _("Untitled");

  g_snprintf (buf, sizeof (buf), "%s%s - Kipin\303\244", 
             (name != NULL) ? name : str, 
              kp_training_log_is_modified (log) ? _(" [modified]") : "");
  
  gtk_window_set_title (GTK_WINDOW (window), buf);

  g_free (name);

  kp_debug ("Updated title text");  
}


static void
set_log_file (KPMainWindow *window, const gchar *name)
{
  KPMainWindowPrivateData *p_data;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  g_string_assign (p_data->logfile, (name) ? name : "");

  update_window_title (window);
}


static void
kp_view_view_set (KPViewModel *model, KPViewModelType type,
                  KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  gconf_client_set_int (p_data->client, KP_CONF_ACTIVE_TIME_PERIOD, type + 1, NULL);
  
  kp_debug ("View set: %d", type);
  kp_main_window_update_date_text (window);
  kp_statusbar_set_view_type (p_data->sbar, type);

  update_toolbar_state (window);
}


static void
kp_view_date_set (KPViewModel *model, KPDate *date, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_main_window_update_date_text (window);

  kp_debug ("Date set: %d.%d.%d", date->d, date->m, date->y);
}


static void
kp_view_viewer_set (KPView *view, KPViewModel *model, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  const gchar *name;
  
  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  name = G_OBJECT_TYPE_NAME (model);
  kp_debug ("Viewer set: %s", name);

  gconf_client_set_string (p_data->client, KP_CONF_ACTIVE_VIEW_TYPE, name, NULL);
   
  kp_statusbar_set_viewer_name (p_data->sbar, G_OBJECT_TYPE_NAME (model));
}


static void
kp_main_window_update_date_text (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  struct tm tm;
  GDate *date;
  gchar buf[128];
  gchar *format;
  gchar *tmp = NULL;
  gchar *str;
  KPDate d;

  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_view_model_get_dmy (KP_VIEW_MODEL (p_data->view), &d.d, &d.m, &d.y);
  date = g_date_new_dmy (d.d, d.m, d.y);

  g_return_if_fail (date != NULL);

  format = kp_view_model_get_date_format (KP_VIEW_MODEL (p_data->view));

  kp_debug ("We are going to set the date title, date is %d.%d.%d\n", 
            g_date_get_day (date), g_date_get_month (date), g_date_get_year (date));
  g_date_to_struct_tm (date, &tm);
  strftime (buf, sizeof (buf) - 1, format, &tm);
  g_free (format);

  /*
   * If locale charset is not UTF-8, it can cause some trouble because
   * GTK+ needs valid UTF-8, so we need to check and convert if needed.
   */
  if (!g_utf8_validate (buf, -1, NULL)) {
    kp_debug ("Invalid utf-8 from strftime(), trying to convert..");
    
    if (!(tmp = g_locale_to_utf8 (buf, -1, NULL, NULL, NULL))) {
      kp_debug ("Can't convert, date str is now invalid, returning..", buf);
      return;
    }
    if (tmp) {
      strncpy (buf, tmp, sizeof (buf) - 1);
      g_free (tmp);
    }
  }
  str = g_strconcat ("<span size=\"x-large\"><b>", buf, "</b></span>", NULL);

  if (GTK_IS_LABEL (p_data->date_label))
    gtk_label_set_markup (GTK_LABEL (p_data->date_label), str);
  else
    return;
  
  g_date_free (date);
  g_free (str);
}


void
kp_main_window_init_log (KPMainWindow *window, const gchar *filename)
{
  KPMainWindowPrivateData *p_data;
  KPTrainingLog *log;
  KPLogStore *store;
  GError *err;
  gchar *file;
 
  file = (filename) ? g_filename_display_basename (filename) : NULL;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  
  /* If the filename was given, try to open the log on 
   * that name */
  if (filename) {
    err = NULL;
    /* Don't receive signals to be faster */
    kp_view_model_deactivate (KP_VIEW_MODEL (p_data->view));
       
    log = kp_training_log_new_from_file (filename, &err);
      
    if (log) {
      /* There is an old log */
      if (KP_IS_TRAINING_LOG (p_data->log))
        kp_main_window_deinit_log (window);
      
      p_data->log = log;
      g_return_if_fail (log != NULL);
      set_log_file (window, file);
      kp_statusbar_set_format_message (p_data->sbar, _("Log opened: %s"), file);
    } else {
      g_object_unref (log);
      kp_gui_err_message (GTK_WINDOW (window), 
                         (err) ? err->message : "Unknown error!");
      if (err)
        g_error_free (err);
      
      kp_debug ("Couldn't open the log, return!");
      return;
    }
  } else {
    /* The user wants to create a new log */
    log = kp_training_log_new ();

    if (KP_IS_TRAINING_LOG (p_data->log))
      kp_main_window_deinit_log (window);
    
    p_data->log = log;
    set_log_file (window, NULL);
    kp_statusbar_set_message (p_data->sbar, _("New log file created."));
  }

  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
  kp_debug ("TrainingLog has %u entries", kp_training_log_get_size (p_data->log));

  kp_statusbar_set_log (KP_STATUSBAR (p_data->sbar), p_data->log);

  log_connect_signals (p_data->log, window);
  kp_view_model_set_log (KP_VIEW_MODEL (p_data->view), p_data->log);
  /* Make View to update it's display */
  kp_view_model_activate (KP_VIEW_MODEL (p_data->view));
  
  /* Update views according to the new log */
  store = kp_log_store_new ();
  gtk_tree_view_set_model (GTK_TREE_VIEW (p_data->tree),
                           GTK_TREE_MODEL (store));
  kp_log_store_attach_log (store, p_data->log, 0);
  kp_view_model_set_log (KP_VIEW_MODEL (p_data->tree), p_data->log);

  g_free (file);
}


static void
kp_main_window_deinit_log (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_view_model_unset_log (KP_VIEW_MODEL (p_data->view));
  kp_view_model_unset_log (KP_VIEW_MODEL (p_data->tree));
  kp_statusbar_unset_log (KP_STATUSBAR (p_data->sbar));
  
  log_disconnect_signals (p_data->log, window);

  kp_debug ("Log deinit.");

  set_log_file (window, NULL);
  g_object_unref (p_data->log);
}


static void
on_paned_handle_move (GObject *pane, GParamSpec *spec, KPMainWindow *window)
                    
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
 
  g_return_if_fail (GTK_IS_HPANED (pane));
  
  gconf_client_set_int (p_data->client, KEY_PANED_WIDTH, 
                        gtk_paned_get_position (GTK_PANED (pane)), NULL);
}


/* 0 back, 1 forward */
static void
change_date_by_one (KPMainWindow *window, gint type, guint direction)
{
  KPMainWindowPrivateData *p_data;
  KPViewModelType cv_type;
  GDate *date;
  guint d, m, y;
  guint months;
  guint days;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_view_model_get_dmy (KP_VIEW_MODEL (p_data->view), &d, &m, &y);
 
  date = g_date_new_dmy (d, m, y);
  
  cv_type = (type == -1) ?
    kp_view_model_get_view_type (KP_VIEW_MODEL (p_data->view)) : (guint) type;
  
  switch (cv_type) {
    case KP_VIEW_MODEL_TYPE_DAY:
      days = 1;
      months = 0;
      break;
    case KP_VIEW_MODEL_TYPE_WEEK:
      days = 7;
      months = 0;
      break;
    case KP_VIEW_MODEL_TYPE_MONTH:
      days = 0;
      months = 1;
      break;
    case KP_VIEW_MODEL_TYPE_YEAR:
      days = 0;
      months = 12;
      break;
    case KP_VIEW_MODEL_TYPE_ALL_TIME:
      return;
      
    default:
      days = 0;
      months = 0;
      g_assert_not_reached ();
  }

  if (direction == 1) {
    g_date_add_months (date, months);
    g_date_add_days (date, days);
  } else {
    g_date_subtract_months (date, months);
    g_date_subtract_days (date, days);
  }
    
  kp_view_model_set_dmy (KP_VIEW_MODEL (p_data->view),
                         g_date_get_day (date),
                         g_date_get_month (date),
                         g_date_get_year (date));
  g_date_free (date);
}


static void
move_date_forward_by_one (KPMainWindow *window)
{
  change_date_by_one (window, -1, 1);
}


static void
move_date_backwards_by_one (KPMainWindow *window)
{
  change_date_by_one (window, -1, 0);
}

  
static void 
next_button_cb (GtkButton *button, KPMainWindow *window)
{ 
  move_date_forward_by_one (window);
}


static void
prev_button_cb (GtkButton *button, KPMainWindow *window)
{
  move_date_backwards_by_one (window);
}


static void
on_view_type_combo_changed_cb (GtkComboBox *box, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *str;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  model = gtk_combo_box_get_model (box);
  if (!gtk_combo_box_get_active_iter (box, &iter))
    return;
 
  gtk_tree_model_get (model, &iter, 0, &str, -1);

  kp_debug ("View type changed: %s\n", str);
  if (strcmp (str, _("Calendar")) == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPCalendarView");
  else if (strcmp (str, _("List")) == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPListView");
  else if (strcmp (str, _("Statistics")) == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPStatsView");
  else if (strcmp (str, _("Chart")) == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPChartView");
  else {
    g_warning ("Unknown viewer: %s\n", str);
    g_return_if_reached ();
  }
}


static void
on_time_period_combo_changed_cb (GtkComboBox *box, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  KPViewModelType type;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *str;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  model = gtk_combo_box_get_model (box);
  if (!gtk_combo_box_get_active_iter (box, &iter))
    return;
  
  gtk_tree_model_get (model, &iter, 0, &str, -1);

  kp_debug ("Time period combo box changed (selected: %s)\n", str);
  
  if (strcmp (str, _("Day")) == 0)
    type = KP_VIEW_MODEL_TYPE_DAY;
  else if (strcmp (str, _("Week")) == 0)
    type = KP_VIEW_MODEL_TYPE_WEEK;
  else if (strcmp (str, _("Month")) == 0)
    type = KP_VIEW_MODEL_TYPE_MONTH;
  else if (strcmp (str, _("Year")) == 0)
    type = KP_VIEW_MODEL_TYPE_YEAR;
  else if (strcmp (str, _("All Time")) == 0)
    type = KP_VIEW_MODEL_TYPE_ALL_TIME;
  else
    g_return_if_reached ();
  
  kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view), type);
}


/*
 * This is called when user selects a node in the sidebar tree.
 */
static void
workout_tree_selection_changed (GtkTreeSelection *selection,
                                KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  KPLogStoreRecordType type;
  KPViewModelType view_type;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint d, m, y;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_debug ("Selection changed.");
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    kp_log_store_get_date (KP_LOG_STORE (model), &iter, &d, &m, &y);
  else {
    kp_debug ("Can't get selected item, return.");
    return;
  }

  type = kp_log_store_get_iter_type (KP_LOG_STORE (model), &iter);

  switch (type)
  {
    case KP_LOG_STORE_REC_YEAR:
      view_type = KP_VIEW_MODEL_TYPE_YEAR;
      m = 1;
      d = 1;
      break;
      
    case KP_LOG_STORE_REC_MONTH:
      view_type = KP_VIEW_MODEL_TYPE_MONTH;
      d = 1;
      break;
      
    case KP_LOG_STORE_REC_DAY:
      view_type = KP_VIEW_MODEL_TYPE_DAY;
      break;

    case KP_LOG_STORE_REC_ROOT:
      view_type = KP_VIEW_MODEL_TYPE_ALL_TIME;
      kp_view_model_get_dmy (KP_VIEW_MODEL (p_data->view), &d, &m, &y);
      break;
      
    case KP_LOG_STORE_REC_ENTRY:
    case KP_LOG_STORE_REC_INVALID:
    default:
      return;
  }

  g_return_if_fail (g_date_valid_dmy (d, m, y));
 
  kp_view_model_set_dmy (KP_VIEW_MODEL (p_data->view), d, m, y);
  kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view), view_type);
}


static void
log_connect_signals (KPTrainingLog *log, KPMainWindow *window)
{
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), window);
  g_signal_connect (G_OBJECT (log), "saved",
                    G_CALLBACK (log_changed), window);
  g_signal_connect (G_OBJECT (log), "new-entry-added",
                    G_CALLBACK (log_entry_added), window);
  g_signal_connect (G_OBJECT (log), "entry-removed",
                    G_CALLBACK (log_entry_removed), window);
}


static void
log_disconnect_signals (KPTrainingLog *log, KPMainWindow *window)
{
  g_signal_handlers_disconnect_by_func (log, log_entry_removed, window);
  g_signal_handlers_disconnect_by_func (log, log_entry_added, window);
  g_signal_handlers_disconnect_by_func (log, log_changed, window);
  g_signal_handlers_disconnect_by_func (log, log_saved, window);
}


static void
log_entry_removed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark, KPMainWindow *window)
{
  kp_statusbar_set_message (kp_main_window_get_statusbar (),
                         _("Entry removed from the log."));
}


static void
log_entry_added (KPTrainingLog *log, KPCalendarEntry *entry, KPMainWindow *window)
{
  kp_statusbar_set_message (kp_main_window_get_statusbar (), 
                          _("New entry added to the log."));
}


static void
log_changed (KPTrainingLog *log, KPMainWindow *window)
{
  update_window_title (window);
}

  
static void
log_saved (KPTrainingLog *log, KPMainWindow *window)
{
  update_window_title (window);
}


static void
calendar_view_day_selected (KPCalendarView *cv, KPDate *d, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gchar buf[64];
  gchar *str;
  guint len;
 
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  
  len = g_list_length (kp_training_log_get_day (p_data->log, d->d, d->m, d->y));
  if (len == 0 || len == 1)
    strncpy (buf, (len == 0) ? _("No Entries") : _("1 Entry"), sizeof (buf)-1);
  else
    g_snprintf (buf, sizeof (buf)-1, _("%u Entries"), len);
  
  str = g_strdup_printf (_("%u.%u.%u (%s) selected."), d->d, d->m, d->y, buf);
  
  kp_statusbar_set_message (p_data->sbar, str);
  kp_view_model_set_date (KP_VIEW_MODEL (p_data->view), d);
  g_free (str);
  
  update_toolbar_state (window);
}


static void
update_toolbar_state (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gboolean all_time_set;
  gboolean day_has_mark;
  gboolean log;

  g_return_if_fail (KP_IS_MAIN_WINDOW (window));

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  log = (p_data->log != NULL);

  all_time_set = (kp_view_model_get_view_type (KP_VIEW_MODEL (p_data->view))
              ==  KP_VIEW_MODEL_TYPE_ALL_TIME); 
 
  if (GTK_IS_WIDGET (p_data->toolbar_next_button) == FALSE)
    return;
  
  gtk_widget_set_sensitive (p_data->toolbar_next_button, !all_time_set);
  gtk_widget_set_sensitive (p_data->toolbar_prev_button, !all_time_set);
  
  day_has_mark = (kp_calendar_view_get_current_mark (p_data->cv) != NULL);

  if (log) 
    gtk_widget_show (p_data->paned_window);
  else 
    gtk_widget_hide (p_data->paned_window);
}


KPTrainingLog *
kp_main_window_get_log (void)
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);
  g_return_val_if_fail (p_data != NULL, NULL);
 
  return p_data->log;
}


KPStatusbar *
kp_main_window_get_statusbar (void)
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);
  g_return_val_if_fail (p_data != NULL, NULL);
  
  return p_data->sbar;
}

KPView *
kp_main_window_get_view (void)
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);
  g_return_val_if_fail (p_data != NULL, NULL);
  
  return KP_VIEW (p_data->view);
}


#define _GTK_WIDGET_SET_VISIBILITY(widget,visible)  \
if((visible)) { gtk_widget_show(GTK_WIDGET (widget)); } \
else        { gtk_widget_hide(GTK_WIDGET (widget)); }
    

void
kp_main_window_set_toolbar_visible (gboolean visible)
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);
  _GTK_WIDGET_SET_VISIBILITY (p_data->toolbar_log, visible);
  gconf_client_set_bool (p_data->client, KP_CONF_SHOW_TOOLBAR, visible, NULL);
}


void
kp_main_window_set_sidebar_visible (gboolean visible)
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);
  guint width;

  gconf_client_set_bool (p_data->client, KP_CONF_SHOW_SIDEBAR, visible, NULL);
  _GTK_WIDGET_SET_VISIBILITY (p_data->sidebar, visible);
 
  width = gconf_client_get_int (p_data->client, KEY_PANED_WIDTH, NULL);
  /* Use 100 as default value */
  width = (width <= 0) ? 100 : width;
  gtk_paned_set_position (GTK_PANED (p_data->paned_window), width);
}


void
kp_main_window_set_statusbar_visible (gboolean visible)
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);
  _GTK_WIDGET_SET_VISIBILITY (p_data->sbar, visible);
  gconf_client_set_bool (p_data->client, KP_CONF_SHOW_STATUSBAR, visible, NULL);
}


/*
  will be added to KPMainWindowPrivateData:
  menu_import
  menu_export
  menubar  
 */
void
kp_main_window_add_to_import_menu (GtkMenuItem *item)
{
  KPMainWindowPrivateData *p_data;
  GtkWidget *menu;
 
  g_return_if_fail (item != NULL);
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);

  g_return_if_fail (p_data->menu_import != NULL);
  menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (p_data->menu_import));

  if (!menu) {
    menu = gtk_menu_new ();
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (p_data->menu_import), menu);
    gtk_widget_show (GTK_WIDGET (p_data->menu_import));
  }
  gtk_widget_show (GTK_WIDGET (item));

  gtk_menu_shell_append (GTK_MENU_SHELL (menu), GTK_WIDGET (item));
  gtk_widget_show_all (p_data->menu_import);
}


GtkWidget *
kp_main_window_add_to_export_menu (GtkMenuItem *item)
{
  KPMainWindowPrivateData *p_data;
  GtkWidget *menu;
 
  g_return_val_if_fail (item != NULL, NULL);
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);

  g_return_val_if_fail (p_data->menu_export != NULL, NULL);
  menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (p_data->menu_export));
  
  if (!menu) {
    menu = gtk_menu_new ();
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (p_data->menu_export), menu);
    gtk_widget_show (p_data->menu_file_separator);
  }
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), GTK_WIDGET (item));
  gtk_widget_show_all (p_data->menu_export);
  
  return NULL;
}

GtkWidget *
kp_main_window_add_to_menubar (GtkWidget *menu)
{
  /* TODO: Remove this */
  g_warning ("kp_main_window_add_to_menubar(): TODO: This doesn't work.");
  return NULL;
}


static gboolean
on_main_window_close (GtkWidget *widget, GdkEvent *event, KPMainWindow *window)
{
  /* TODO: Is there a better way to do this? */
  gtk_main_quit ();
  return TRUE;
}


KPMainWindow *
kp_main_window_get_window (void)
{
  return _window;
}


static gboolean
main_window_key_press (GtkWidget *widget, GdkEventKey *key,
                       KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  guint cv_type;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (KP_MAIN_WINDOW (widget));
  cv_type = kp_view_model_get_view_type (KP_VIEW_MODEL (p_data->view));

  switch (key->keyval)
  {
    
    case GDK_KP_Left:
    case GDK_Left:
      if (key->state & GDK_CONTROL_MASK) {
        if (cv_type > 0)
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view),
                                       cv_type - 1);
        else
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view),
                                       KP_VIEW_MODEL_TYPE_N - 1);
      } else if (key->state & GDK_MOD1_MASK) {
        move_date_backwards_by_one (window);
      } else
        return FALSE;
      break;
      
    case GDK_KP_Right:
    case GDK_Right:
      if (key->state & GDK_CONTROL_MASK) {
        if (cv_type < KP_VIEW_MODEL_TYPE_N - 1)
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view),
                                       cv_type + 1);
        else
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view), 0);
      } else if (key->state & GDK_MOD1_MASK) {
        move_date_forward_by_one (window);
      } else {
        return FALSE;
      }
      break;

    case GDK_KP_Up:
    case GDK_Up:
      if (key->state & GDK_CONTROL_MASK) {
        kp_view_set_viewer_next (KP_VIEW (p_data->view));
      } else {
        return FALSE;
      } 
      break;

    case GDK_KP_Down:
    case GDK_Down:
      if (key->state & GDK_CONTROL_MASK) {
        kp_view_set_viewer_prev (KP_VIEW (p_data->view));
      } else {
        return FALSE;
      }
      break;

    case GDK_s:
      if (key->state & GDK_CONTROL_MASK)
        /* TODO: How? */
    break;

    case GDK_o:
      if (key->state & GDK_CONTROL_MASK)
        /* TODO: How ? */
    break;

    case GDK_n:
      if (key->state & GDK_CONTROL_MASK)
        /* TODO: How ? */
    break;
      
    default:
      /* pass the event to some else place */
      return FALSE;
  }
  return TRUE;
}

