#include <gtk/gtk.h>
#include <string.h>
#include <time.h>

#include "../kipina-i18n.h"
#include "kpmainwindow.h"
#include "kpentrypropertiesdialog.h"

static void     kp_entry_properties_dialog_class_init     (KPEntryPropertiesDialogClass *klass);
static void     kp_entry_properties_dialog_init           (KPEntryPropertiesDialog *dialog);
static void     kp_entry_properties_dialog_finalize       (GObject *object);


typedef struct KPEntryPropertiesDialogPrivateData_
{
  GtkWidget   *table;

  GtkWidget   *titlebox;
  GtkWidget   *icon;

  GtkWidget   *notebook;

  gboolean     triplets_used;
  
  guint        n_pairs;
} KPEntryPropertiesDialogPrivateData;

#define KP_ENTRY_PROPERTIES_DIALOG_PRIVATE_DATA(widget) \
  (((KPEntryPropertiesDialogPrivateData*) \
  (KP_ENTRY_PROPERTIES_DIALOG (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_entry_properties_dialog_get_type (void)
{
  static GType kp_entry_properties_dialog_type = 0;

  if (kp_entry_properties_dialog_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPEntryPropertiesDialogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_entry_properties_dialog_class_init,
      NULL,
      NULL,
      sizeof (KPEntryPropertiesDialog),
      0,
      (GInstanceInitFunc) kp_entry_properties_dialog_init,
      NULL,
    };

    kp_entry_properties_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                             "KPEntryPropertiesDialog",
                                                             &our_info, 0);
  }
  return kp_entry_properties_dialog_type;
}


static void
kp_entry_properties_dialog_class_init (KPEntryPropertiesDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_entry_properties_dialog_finalize;
}


static void
kp_entry_properties_dialog_init (KPEntryPropertiesDialog *dialog)
{
  KPEntryPropertiesDialogPrivateData *p_data;
  GtkWidget *close;
  
  dialog->private_data = g_new0 (KPEntryPropertiesDialogPrivateData, 1);
  p_data = KP_ENTRY_PROPERTIES_DIALOG_PRIVATE_DATA (dialog);

  dialog->vbox = gtk_vbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), dialog->vbox);
  gtk_container_set_border_width (GTK_CONTAINER (dialog->vbox), 12);
  gtk_widget_show (dialog->vbox);

  p_data->titlebox = gtk_hbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (dialog->vbox), p_data->titlebox, FALSE, TRUE, 0);
  gtk_widget_show (p_data->titlebox);
  
  p_data->table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (p_data->table), 6);
  gtk_table_set_row_spacings (GTK_TABLE (p_data->table), 6);
  gtk_box_pack_start (GTK_BOX (dialog->vbox), p_data->table, TRUE, TRUE, 0);
  gtk_widget_show (p_data->table);
  p_data->n_pairs = 0;
  p_data->triplets_used = 0;  
  
  gtk_window_set_title (GTK_WINDOW (dialog), 
                      _("Calendar Entry Information"));
  
  close = gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CLOSE,
                                 GTK_RESPONSE_CLOSE);
  
  g_signal_connect_swapped (G_OBJECT (close), "clicked",
                            G_CALLBACK (gtk_widget_destroy), dialog);

  p_data->notebook = NULL;
}


static void
kp_entry_properties_dialog_finalize (GObject *object)
{
  KPEntryPropertiesDialog *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_ENTRY_PROPERTIES_DIALOG (object));

  dialog = KP_ENTRY_PROPERTIES_DIALOG (object);

  g_return_if_fail (dialog->private_data != NULL);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


static void
set_title_string (GtkLabel *label, KPCalendarEntry *entry)
{
  KPCalendarTime *ct = entry->datetime;
  struct tm tm;
  gchar buf[128];
  gchar format[128];
  KPTime time;
  gchar *tod;
  gchar *item;
 
  kp_time_set_hms (&time, ct->h, ct->m, ct->s);
  tod = kp_time_tod_to_string (&time);
  item = kp_calendar_entry_get_human_name (entry);
  
  g_snprintf (format, sizeof (format)-1, "<b><big>%s</big></b>\n<i>%%x %s</i>", 
              item, (tod) ? tod : "%X");
  g_free (item);
  if (tod)
    g_free (tod);
  
  tm.tm_year = g_date_get_year (ct->date) - 1900;
  tm.tm_mon = g_date_get_month (ct->date) - 1;
  tm.tm_mday = g_date_get_day (ct->date);
  tm.tm_hour = ct->h;
  tm.tm_min = ct->m;
  tm.tm_sec = ct->s;
  tm.tm_isdst = -1;
  
  (void) strftime (buf, sizeof (buf)-1, format, &tm);
  
  gtk_label_set_markup (GTK_LABEL (label), buf);

}
  
/**
 * kp_entry_properties_dialog_new:
 * @date: A #KPDate
 * @log: A #KPTrainingLog
 *
 * Create an instance of #KPEntryPropertiesDialog.
 *
 * Returns: A #KPEntryPropertiesDialog
 */
GtkWidget *
kp_entry_properties_dialog_new (KPCalendarEntry *entry)
{
  KPEntryPropertiesDialogPrivateData *p_data;
  GtkWidget *al;
  GtkWidget *label;
  gchar *icon_path;
  
  KPEntryPropertiesDialog *dialog;
  
  dialog = g_object_new (kp_entry_properties_dialog_get_type (), NULL);
  
  p_data = KP_ENTRY_PROPERTIES_DIALOG_PRIVATE_DATA (dialog);
 
  /* Create and pack the icon */
  icon_path = g_build_filename (KIPINA_PIXMAP_DIR, 
                                kp_calendar_entry_get_icon_name (entry), NULL);
  p_data->icon = gtk_image_new_from_file (icon_path);
  gtk_box_pack_start (GTK_BOX (p_data->titlebox), p_data->icon, FALSE, TRUE, 0);
  gtk_image_set_from_file (GTK_IMAGE (p_data->icon), icon_path);
  gtk_widget_show (p_data->icon);
  g_free (icon_path);
 
  /* Create the alignment and the title label and pack them */
  al = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_container_set_border_width (GTK_CONTAINER (al), 12);
  gtk_box_pack_start (GTK_BOX (p_data->titlebox), al, FALSE, TRUE, 0);
  gtk_widget_show (al);
  
  label = gtk_label_new (NULL);

  set_title_string (GTK_LABEL (label), entry);
  gtk_container_add (GTK_CONTAINER (al), label);
  gtk_widget_show (label);
 
  return GTK_WIDGET (dialog);
}


GtkWidget *
kp_entry_properties_dialog_new_with_tabs (KPCalendarEntry *entry, const gchar *title, ...)
{
  KPEntryPropertiesDialogPrivateData *p_data;
  KPEntryPropertiesDialog *dialog;
  const gchar *n_title;
  va_list args;

  g_return_val_if_fail (title != NULL, NULL);
  
  dialog = KP_ENTRY_PROPERTIES_DIALOG (kp_entry_properties_dialog_new (entry));
  p_data = KP_ENTRY_PROPERTIES_DIALOG_PRIVATE_DATA (dialog);

  p_data->notebook = gtk_notebook_new ();

  gtk_notebook_append_page (GTK_NOTEBOOK (p_data->notebook),
                            gtk_label_new ("something"),
                            gtk_label_new (title));
  
  va_start (args, title);

  while ((n_title = va_arg (args, const gchar *)) != NULL)
    gtk_notebook_append_page (GTK_NOTEBOOK (p_data->notebook),
                              gtk_label_new ("something"),  
                              gtk_label_new (n_title));

  va_end (args);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), p_data->notebook, FALSE, TRUE, 0);
  
  gtk_widget_show_all (p_data->notebook);
  
  return GTK_WIDGET (dialog);
}

void
kp_entry_properties_dialog_add_pair (KPEntryPropertiesDialog *dialog,
                                     const gchar *name,
                                     GtkWidget *value, gboolean expand)
{
  KPEntryPropertiesDialogPrivateData *p_data;
  GtkWidget *label;
  gchar *str;
  GtkWidget *al;
  
  g_return_if_fail (GTK_IS_WIDGET (value));
  g_return_if_fail (name != NULL);
  
  p_data = KP_ENTRY_PROPERTIES_DIALOG_PRIVATE_DATA (dialog);
  gtk_table_resize (GTK_TABLE (p_data->table), 3 + p_data->n_pairs, 2);
  
  str = g_strconcat ("<b>", name, "</b>", NULL);
  label = gtk_label_new (NULL);
  gtk_label_set_markup (GTK_LABEL (label), str);
  g_free (str);

  al = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (al), label);
  gtk_widget_show (al);
  
  gtk_table_attach (GTK_TABLE (p_data->table), al, 
                    0, 1, p_data->n_pairs, p_data->n_pairs + 1,
                    GTK_FILL,  GTK_FILL, 
                    0, 0);
 
  al = gtk_alignment_new (0.0, 0.0, 
                         (expand) ? 1.0 : 0.0, 
                         (expand) ? 1.0 : 0.0);
  gtk_container_add (GTK_CONTAINER (al), value);
  gtk_widget_show (al);

  gtk_table_attach (GTK_TABLE (p_data->table), al,
                    1, 2 + p_data->triplets_used, /* Use all the space if triplets are used */
                     p_data->n_pairs, p_data->n_pairs + 1,
                    (expand) ? GTK_EXPAND | GTK_FILL : GTK_FILL,
                    (expand) ? GTK_EXPAND | GTK_FILL : GTK_FILL,
                    0, 0);

  gtk_widget_show (label);
  gtk_widget_show (value);
  
  p_data->n_pairs++;
}

  
void
kp_entry_properties_dialog_add_triplet (KPEntryPropertiesDialog *dialog,
                                        const gchar *name, GtkWidget *value,
                                        GtkWidget *value2, gboolean expand)
{
  KPEntryPropertiesDialogPrivateData *p_data;
  GtkWidget *label;
  GtkWidget *al;
  
  g_return_if_fail (GTK_IS_WIDGET (value));
  g_return_if_fail (name != NULL);
  
  p_data = KP_ENTRY_PROPERTIES_DIALOG_PRIVATE_DATA (dialog);
  gtk_table_resize (GTK_TABLE (p_data->table), 3 + p_data->n_pairs, 3);
  
  /* Label (1st item) */
  label = gtk_label_new (name);

  al = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (al), label);
  gtk_widget_show (al);
  
  gtk_table_attach (GTK_TABLE (p_data->table), al, 
                    0, 1, p_data->n_pairs, p_data->n_pairs + 1,
                    GTK_FILL,  GTK_FILL, 
                    0, 0);
  gtk_widget_show (label);

  /* Value 1 (2nd item) */
  al = gtk_alignment_new (0.0, 0.0, 
                         (expand) ? 1.0 : 0.0, 
                         (expand) ? 1.0 : 0.0);
  
  gtk_container_add (GTK_CONTAINER (al), value);
  gtk_widget_show (al);
  
  gtk_table_attach (GTK_TABLE (p_data->table), al,
                    1, 2, p_data->n_pairs, p_data->n_pairs + 1,
                   (expand) ? GTK_EXPAND | GTK_FILL : GTK_FILL,
                   (expand) ? GTK_EXPAND | GTK_FILL : GTK_FILL,
                    0, 0);
  gtk_widget_show (value);
 
  /* Value 2 (3rd item) */
  al = gtk_alignment_new (0.0, 0.0,
                         (expand) ? 1.0 : 0.0,
                         (expand) ? 1.0 : 0.0);
  gtk_container_add (GTK_CONTAINER (al), value2);
  gtk_widget_show (al);

  gtk_table_attach (GTK_TABLE (p_data->table), al,
                    2, 3, p_data->n_pairs, p_data->n_pairs + 1,
                   (expand) ? GTK_EXPAND | GTK_FILL : GTK_FILL,
                   (expand) ? GTK_EXPAND | GTK_FILL : GTK_FILL,
                    0, 0);
  gtk_widget_show (value2);
  
  p_data->n_pairs++;
  p_data->triplets_used = TRUE;
}

