/***************************************************************************
 *   Copyright (C) 2004 by Emil Stoyanov                                   *
 *   emosto@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "iaxwrapper.h"

IaxWrapper* callbackCaller;

/*
	Global method used as callback function in libiaxclient
	Forwards incoming events to void IaxWrapper::handleIaxCEvent(void* callbackCaller, iaxc_event e) via customEvent(QCustomEvent)
*/

int iaxc_callback(iaxc_event e)
{
  // wrap iaxc_event into a QEvent
  IaxWrapperEvent * wrapperEvent = new IaxWrapperEvent(e);
  // and create a CustomEvent to be posted
  QCustomEvent * wev = new QCustomEvent(QEvent::User, wrapperEvent);
  /*
     we are not allowed to access GUI thread directly from external threads
     so, post event to be processed by the GUI thread
  */
  QApplication::postEvent(callbackCaller, wev);
  return 1;
}

/*
	Initialization and destruction of the wrapper
*/

IaxWrapper::IaxWrapper(QObject *parent, const char *name)
    : QObject(parent, name)
{
  showStats = false;
  registered=false; // we are not ready to call
  callbackCaller = this; // to use it in the class callback function

  //load the settings prior to do anything else
  settings = new KiaxPrefs(this,0);
  settings->loadSettings();
  
  /* allocate memory for the statistics vars*/
  rtt = (int*) malloc(sizeof(int));
  localNetstat = (struct iaxc_netstat*) malloc(sizeof(struct iaxc_netstat));
  remoteNetstat = (struct iaxc_netstat*) malloc(sizeof(struct iaxc_netstat));
  
  ringInTone = NULL;
  ringTimer = new QTimer(); // repeats ringing every 4 seconds

  callDurationTimer = new QTimer(); // shows duration of the time every second

  connect(ringTimer, SIGNAL(timeout()), this, SLOT(ring()));
  connect(callDurationTimer, SIGNAL(timeout()), this, SLOT(callDuration()));
  
  char* version = (char*)malloc(256);
  iaxc_ver = iaxc_version(version);
  
  debug("Using IAXClient ver. %s\n", iaxc_ver);

}

IaxWrapper::~IaxWrapper()
{
  shutdown_iax_client();
}

KiaxPrefs* IaxWrapper::getSettings()
{
  return settings;
}

/*
	High-level Methods interfacing the wrapped libiaxclient functions
*/

void IaxWrapper::registerWithServer()
{

  // start the action
  start_iax_client();
  registerMultipleAccounts();

  /* 
     Initially the idea was to enable controls for dialing only if we have Reigstration
     Accepted, but letting the people use the controls even if not registered is OK as we may
     have such servers that don't allow/require/need registration but allow calling.
  */

  registered = true;
  // and tell the GUI to enable "unregister" button or other
  emit signalRegistered();
}

int IaxWrapper::registerAccount(KiaxAccount* account) {
   const char* func = "IaxWrapper::registerMultipleAccounts()";
   debug ("%s registering accounts..\n", func);

    const char* accName = account->accAlias;
    const char* uname = account->username;
    const char* passwd = account->password;
    const char* hst = account->iaxServer;
    if ((account->registerAccount)&&(account->iaxServer!=""))
    {
      // tell to an Asterisk server where we are
      account->iaxRegistrationId = iaxc_register((char*)uname,(char*) passwd, (char*) hst);
      debug ("%s register %s [%s:xxx@%s], given ID = %d\n",  func, accName, uname, hst, account->iaxRegistrationId);
      emit signalRegistrationRequested(account);
    }
    else
    {
      debug ("%s skipped registration %s [%s:xxx@%s]\n", func, accName,uname, hst);
      account->iaxRegistrationId = -1; //skipped
    }
    return account->iaxRegistrationId;
}

int IaxWrapper::unregisterAccount(KiaxAccount* account) {
	if (account->iaxRegistrationId!=-1) {
		int result = iaxc_unregister(account->iaxRegistrationId);
		//emit signalUnregistrationRequested(account);
		account->setState(ACCOUNT_INACTIVE);
	} else return -1; // skipped
}

void IaxWrapper::registerMultipleAccounts()
{
  // loop through all accounts and call register

  for (uint i=0; i<settings->getAccounts().count(); i++)
  {
    KiaxAccount * account = settings->getAccount(i);
    registerAccount(account);
  }
}

void IaxWrapper::unregisterFromServer()
{
 for (int i=0; i<settings->getAccounts().count(); i++) {
 	KiaxAccount* account = settings->getAccount(i);
	if ((account->registerAccount) &&(account->state!=ACCOUNT_INACTIVE)) {
		unregisterAccount(account);
		account->disconnect(SIGNAL(signalRegistrationAccepted(KiaxAccount*)));
		account->disconnect(SIGNAL(signalRegistrationConnecting(KiaxAccount*)));
		account->disconnect(SIGNAL(signalRegistrationInactive(KiaxAccount*)));
		account->disconnect(SIGNAL(signalRegistrationRejected(KiaxAccount*)));
		account->disconnect(SIGNAL(signalRegistrationTimeout(KiaxAccount*)));
	}
 }
  stop_iax_client();
  registered=false;
  emit signalUnregistered();

}

int IaxWrapper::findAccountById(QString accountId)
{
  int accNumber =0;
  // locate the number of the account
  for (uint i=0; i<settings->getAccounts().count(); i++)
    if (accountId==settings->getAccount(i)->accId)
    {
      accNumber=i;
      break;
    }
  return accNumber;
}

void IaxWrapper::dial(QString accountId, QString contactName, QString contactNumber)
{

  KiaxCallRecord * callRecord = createCallRecord(accountId,contactName, contactNumber, true);
  int accountNumber = findAccountById(callRecord->getAccountId());

  KiaxAccount* account = settings->getAccount(accountNumber);
  QString uname = account->username;
  QString passwd = account->password;
  QString hst = account->iaxServer;

  if (registered) // if user is allowed to dial
  {
    QString full_url = uname+":"+passwd+"@"+hst+"/"+callRecord->getCallerIdNumber();
    const char *dest = "";
    configureCall(accountNumber);
    if (hst!="") {
      dest = full_url;
    } else {
      dest = (const char*) callRecord->getCallerIdNumber();
    }
    // first, emit signal to notify the GUI //
    // then call our friend
    debug( "IaxWrapper::dial(int, QString) dialing %s:%s@%s/%s ..\n", (const  char*)uname, "xxx", (const char*) hst, (const char*)callRecord->getCallerIdNumber());
    iaxc_call((char*) dest);
    callSession[getSelectedCall()] = *callRecord;
    emit signalCalling(getSelectedCall(), tr("Calling.."));
  }
}

/*
	Methods interfacing directly with libiaxclient
*/

bool IaxWrapper::init_iax_client()
{
  int result = iaxc_initialize(AUDIO_INTERNAL,getSettings()->getMaxCalls());
  debug( "IaxWrapper::iaxc_initialize() result = %d\n", result);
  if (result==-1)
  {
    // hopefully not many people will have to read this..
    fatal_error("cannot initialize iaxclient!\n");
    return false;
  }
  iaxc_set_formats(IAXC_FORMAT_GSM,IAXC_FORMAT_GSM|IAXC_FORMAT_SPEEX|IAXC_FORMAT_ULAW|IAXC_FORMAT_ILBC);
  return true;
}

void IaxWrapper::shutdown_iax_client()
{
  // shutdown
  debug( "IaxWrapper::shutdown_iax_client() shutting down..\n");
  iaxc_shutdown();
  debug( "IaxWrapper::shutdown_iax_client() done.\n");
}

void IaxWrapper::start_iax_client()
{
  // we want iaxevents to be passed to iaxc_callback
  iaxc_set_event_callback(iaxc_callback);
  // fire
  debug( "IaxWrapper::start_iax_client() starting processing thread..\n");
  int result = iaxc_start_processing_thread();
  debug( "IaxWrapper::iaxc_start_processing_thread() result = %d\n", result);
  if (result==-1)
    fatal_error("cannot start processing thread!\n");
}

void IaxWrapper::stop_iax_client()
{
  // close any active calls
  iaxc_dump_all_calls();
  // wait a bit
  iaxc_millisleep(1000);
  // stop the thread
  iaxc_stop_processing_thread();
  // reset type of call
}

void IaxWrapper::send_dtmf(char digit)
{
  // allow sending dtmf only if we are in a state of a call (ACTIVE)
  // TODO:  this is a cheap version - has to be changed
  if (registered)
  {
    printf ("IaxWrapper::send_dtmf(char digit) sending DTMF%c\n", digit);
    // I like playing with these voice menus
    iaxc_send_dtmf(digit);
  }
}

void IaxWrapper::answer_call(int callNumber)
{
  QString func = "IaxWrapper::answer_call()";
  // answer the call
  iaxc_answer_call(callNumber);
  debug (func+" answered call  %d\n", getSelectedCall());
  // notify GUI
  emit signalAnsweredCall(callNumber);
}

void IaxWrapper::transfer_call(int callNo, QString number)
{
  QString func = "IaxWrapper::transfer_call("+number+")";
  // answer the call
  const char* exten = (const char*) number;
  iaxc_blind_transfer_call(callNo, (char*) exten);
  debug (func+" transferred call  %d\n", callNo);
}

void IaxWrapper::select_call(int callNumber)
{
  QString func = "IaxWrapper::select_call("+QString::number(callNumber)+")";
  debug(func,"\n");
  iaxc_select_call(callNumber);
  debug (func+" selected call  %d\n", getSelectedCall());
}

void IaxWrapper::hold_call(int callNumber)
{
  QString func = "IaxWrapper::hold_call()";
  // answer the call
  iaxc_quelch(callNumber,0);
  debug (func+" hold call  %d\n", callNumber);
  // notify GUI
  KiaxCallRecord& record = callSession[callNumber];
  record.setCustomProperty("isHold","true");
  emit signalHoldCall(callNumber);
}

void IaxWrapper::resume_call(int callNumber)
{
  QString func = "IaxWrapper::resume_call()";
  // answer the call
  iaxc_unquelch(callNumber);
  //KiaxCallRecord * record = callSession.at(getSelectedCall());
  KiaxCallRecord& record = callSession[callNumber];
  record.setCustomProperty("isHold","false");
  debug (func+" resume call  %d\n", callNumber);
  // notify GUI
  emit signalResumedCall(callNumber);
}

void IaxWrapper::reject_call(int callNo)
{
  QString func = "IaxWrapper::reject_call()";
  debug(func+"  rejecting call %d..\n", callNo);
  iaxc_reject_call_number(callNo);
  debug(func+"  rejected call %d\n", callNo);
}

void IaxWrapper::interrupt_call()
{
  printf("IaxWrapper::interrupt_call() hanging up and exiting..\n");
  // hang up the call
  iaxc_dump_call();
  // wait a bit
  iaxc_millisleep(1000);
}

int IaxWrapper::getSelectedCall()
{
  return iaxc_selected_call();
}

KiaxAccount* IaxWrapper::getAccountByRegistrationId(int regId) {
	int number = settings->getAccounts().count();
	for (int i=0;i<number;i++) {
		KiaxAccount* account = settings->getAccount(i);
		if (account->iaxRegistrationId==regId)
			return account;
	}
	return NULL;
}

/*
     Procedures handling iaxclient events (forwarded from iaxc_callback(iaxc_event e)
*/

void IaxWrapper::handleIaxCEvent(iaxc_event e)
{
  switch(e.type)
  {
  case IAXC_EVENT_LEVELS:
    // monitor input and output signal levels (audio)
    event_level(e.ev.levels.input, e.ev.levels.output);
    break;
  case IAXC_EVENT_REGISTRATION:
    // monitor registration status
    event_registration(e.ev.reg.id, e.ev.reg.reply, e.ev.reg.msgcount);
    break;
  case IAXC_EVENT_TEXT:
    event_text(e.ev.text.type, e.ev.text.message);
    break;
  case IAXC_EVENT_STATE:
    event_state(e.ev.call.callNo, e.ev.call.state, e.ev.call.remote,
                e.ev.call.remote_name, e.ev.call.local,
                e.ev.call.local_context, e.ev.reg.id);
    break;
  default:
    event_unknown(e.type);
    break;
  }
}

void IaxWrapper::event_level(float in, float out)
{
  emit signalLevels((int)-in, (int) -out, (int) 0);
}

void IaxWrapper::event_registration(int id, int reply, int msgcount)
{
  debug("Registration id = %d, reply = %d, msgcount = %d\n", id, reply, msgcount);
  KiaxAccount * account = getAccountByRegistrationId(id);
  if (account!=NULL) {
  switch (reply) {
	case IAXC_REGISTRATION_REPLY_ACK : {
		//account->setRegistrationAccepted(true);
		//emit signalRegistrationAccepted(account);
		account->setState(ACCOUNT_ACCEPTED);
		break;
	}
	case IAXC_REGISTRATION_REPLY_REJ : {
		//account->setRegistrationAccepted(false);
		//emit signalRegistrationRejected(account);
		account->setState(ACCOUNT_REJECTED);
		break;
	}
	case IAXC_REGISTRATION_REPLY_TIMEOUT : {
		//emit signalRegistrationTimeout(account);
		account->setState(ACCOUNT_TIMEOUT);
		break;
	}
  }
  }
}

void IaxWrapper::event_state(int callNo, int state, char *remote, char *remote_name,
                             char *local, char *local_context, int reg_id)
{
  
  QString func = "IaxWrapper::event_state()";
  debug (func+" processing IAX event..\n");

  /* Call state masks */
  bool active   = state & IAXC_CALL_STATE_ACTIVE;
  bool outgoing = state & IAXC_CALL_STATE_OUTGOING;
  bool ringing  = state & IAXC_CALL_STATE_RINGING;
  bool complete = state & IAXC_CALL_STATE_COMPLETE;
  bool selected = state & IAXC_CALL_STATE_SELECTED;

  debug (func+" call state %d : active=%d, outgoing=%d, ringing=%d, complete=%d, selected=%d registrationId=%d\n", callNo, active, outgoing, ringing, complete, selected, reg_id);

  // remove the pass (when we are in outgoing call, it is visible)
  QString remoteStr = QString(remote_name);
  // store here remote_name with suppressed pass
  QString modifiedStr = "";
  int passStart = remoteStr.find(":", 0, false) + 1;
  int passEnd = remoteStr.find("@", 0, false);
  int passLength = passEnd-passStart;
  if (passStart>0)
  {
    modifiedStr = remoteStr.replace(passStart,passLength,"xxx");
  }

  debug (func+" call info %d : remote=%s, remote_name=%s, local=%s, local_context=%s\n", callNo, remote, (const char*)modifiedStr, local, local_context);

  QString remoteContext(remote);
  QString localName(local);
  QString localContext(local_context);
  QString remoteName(remote_name);

  QString remoteExten = remote;

  if (active)
  { // there is a call progress
    //ougoing calls
    if ((outgoing) && ringing)
    {
      debug(func+" ACTIVE_OUTGOING_RINGING %d : %s, reigstrationId = %d\n",callNo, (const char *) remoteExten, reg_id);
      // we are making the call so we have already created a record
      KiaxCallRecord& record = callSession[callNo];
      record.setOutgoing(true);
      emit signalOutgoingRinging(callNo);
    }
    // incoming calls
    if ((!outgoing)&& ringing)
    {
      debug(func+" ACTIVE_INCOMING_RINGING %d: %s, registrationId = %d\n",callNo, (const char *) remoteExten, reg_id);
      KiaxCallRecord * record = createCallRecord(settings->getDefaultAccountId(),remoteName, remoteExten,false);
      callSession[callNo] = *record;
      int accountNumber = findAccountById(record->getAccountId());
      configureCall(accountNumber);

      emit signalIncomingRinging(callNo);
    }
    // incoming and outgoing, but active
    if (complete && &selected)
    {
      debug(func+" ACTIVE_CALL_ESTABLISHED %d: %s\n",callNo, (const char *) remoteExten);
      // we have already answered the call so we have a created record
      KiaxCallRecord& record = callSession[callNo];

      if (!record.isAnswered())
      {
        record.setCallStartTime(QString::number(QDateTime::currentDateTime().toTime_t()));
      }
      if (!callDurationTimer->isActive())
      {
        callDurationTimer->start(1000); // notify every second
      }
      record.setAnswered(true);
      emit signalComplete(callNo);
    }

  }
  else
  { // there is no call progress, hangup
    debug(func+" INACTIVE %d\n", callNo);
    KiaxCallRecord& record = callSession[callNo];
    if (callSession.count()==1) //stop the timer only if this is the only call
      callDurationTimer->stop();
    if (record.isAnswered()) record.setCallEndTime(QString::number(QDateTime::currentDateTime().toTime_t()));
    record.save();
    emit signalInactive(callNo);
  }
}

void IaxWrapper::event_text(int type, char *message)
{
  if (message)
  {
    debug( "IaxWrapper::event_text() Message: Type=%d Message=%s\n", type, message);
    switch(type)
    {
    case IAXC_TEXT_TYPE_STATUS:
      emit signalTextMessage(message);
      break;
    }
  }
}

void IaxWrapper::event_unknown(int type)
{
  debug( "IaxWrapper::event_unknown() Uknown message: Type=%d\n", type);
}

void IaxWrapper::fatal_error(char *err)
{
  emit fatalError(QString(err));
}

/*
	Implements a customEvent handler for posted messages
	from iaxc_callback. This solves the problem of asynchronous
	access to the GUI thread.
*/

void IaxWrapper::customEvent(QCustomEvent * event)
{
  IaxWrapperEvent * wev = (IaxWrapperEvent *)event->data();
  iaxc_event e = wev->iaxClientEvent;
  handleIaxCEvent(e);
  delete(wev);
}

KiaxCallRecord* IaxWrapper::createCallRecord(QString accountId, QString name, QString number, bool outgoing)
{
  KiaxCallRecord * record = new KiaxCallRecord();
  record->setAccountId(accountId);
  record->setCallerIdName(name);
  record->setCallerIdNumber(number);
  record->setOutgoing(outgoing);
  record->setAnswered(false);
  return record;
}

/*
  Private methods
*/
uint IaxWrapper::getCodecMaskFromName(QString codecName)
{

  if (codecName=="ilbc")
    return IAXC_FORMAT_ILBC;
  if (codecName=="gsm")
    return IAXC_FORMAT_GSM;
  if (codecName=="ulaw")
    return IAXC_FORMAT_ULAW;
  if (codecName=="speex")
    return IAXC_FORMAT_SPEEX;
  // if none of them  for some reason..
  return IAXC_FORMAT_GSM;
}

void IaxWrapper::configureCall(int accountNumber)
{
  QString func = "IaxWrapper::configureCall(acc number "+QString::number(accountNumber)+")";
  debug( func+" configuring call..\n");
  KiaxAccount * account = settings->getAccount(accountNumber);
  debug( func+" account number=%d, account alias=%s\n",accountNumber,(const char*)account->accAlias);
  // get Caller ID info
  char * c_callerId = (char*)(const char *)account->callerId;
  debug( func+" callerId=%s\n",c_callerId);
  char * c_callerIdNumber = (char*)(const char *)(account->callerIdNumber);
  debug( func+" callerIdNumber=%s\n",c_callerIdNumber);
  iaxc_set_callerid(c_callerId,c_callerIdNumber);
  // get preferred codec - default ILBC
  iaxc_set_formats(getCodecMaskFromName(account->codec),IAXC_FORMAT_GSM|IAXC_FORMAT_SPEEX|IAXC_FORMAT_ULAW|IAXC_FORMAT_ILBC);
  debug( func+" codec=%s\n",(const char*)account->codec);
  // turn on filters
  int flag = settings->getFilterFlag();
  debug( func+" filter flag=%d\n",flag);
  iaxc_set_filters(flag);
  // set silence threshold (1-auto, 0 - mute)
  iaxc_set_silence_threshold(settings->getSilenceThreshold());
  debug( func+" silence threshold=%d\n",settings->getSilenceThreshold());
  // set up audio devices
  iaxc_audio_devices_set(settings->getInputDevice(),settings->getOutputDevice(), settings->getRingDevice());
  debug( func+" devices in=%d, out=%d, ring=%d\n",settings->getInputDevice(), settings->getOutputDevice(), settings->getRingDevice());
  iaxc_set_audio_output(0); //umute output (if muted for some reason)
}

/*
   Portions of tone generation code from iaxComm's Ringer Class 
   (c) Michael Van Donselaar
*/
iaxc_sound* IaxWrapper::initTone(int F1, int F2, int Dur, int Len, int Repeat)
{
  iaxc_sound* tone;
  tone = (iaxc_sound *)malloc(sizeof(iaxc_sound));
  tone->channel = 1;
  tone->len  = Len;
  tone->data = (short *)calloc(tone->len , sizeof(short));


  for( int i=0;i < Dur; i++ )
  {
    tone->data[i] = (short)(0x7fff*0.4*sin((double)i*F1*M_PI/8000))
                   + (short)(0x7fff*0.4*sin((double)i*F2*M_PI/8000));
  }

  tone->repeat = Repeat;
  return tone;

}

void IaxWrapper::ring()
{
 ringInTone = initTone(1900, 2400, 400,500,20);
 iaxc_play_sound(ringInTone, 1);
}

void IaxWrapper::startRing()
{
  ring(); // start ringing, the timer will wait and then ring again
  ringTimer->start(4000);
}

void IaxWrapper::stopRing()
{
  /* In case there was no ring signal.. we have to check*/
  if (ringInTone!= NULL) {
  	iaxc_stop_sound(ringInTone->id);
	delete ringInTone;
  }
  ringTimer->stop();
  ringInTone=NULL;
}

void IaxWrapper::callDuration()
{
  QMutexLocker locker(&callSessionMutex); /* to synchronize access to callSession */
  int callsNumber = callSession.count();
  for (int i=0;i<callsNumber;i++)
  {
    int hours = 0;
    int minutes = 0;
    int seconds = 0;
    int callNumber = callSession.keys()[i];
    int callStartTime=strtol(callSession[callNumber].getCallStartTime(),0,10);
    uint currentTime = QDateTime::currentDateTime().toTime_t();
    uint interval = currentTime - callStartTime;
    div_t hours_calc = div(interval, 3600);
    hours = hours_calc.quot;
    div_t minutes_calc = div(hours_calc.rem, 60);
    minutes = minutes_calc.quot;
    seconds = minutes_calc.rem;
    debugStatistics(callNumber, interval);
    if (callSession[callNumber].isAnswered()) /* display time only if the call was answered */
      emit callTime(callNumber, hours, minutes, seconds);
  }
}

void IaxWrapper::debug(const char* log, ...)
{
  QString dateStr = QDateTime::currentDateTime().toString();
  va_list ap;
  QString finalLog;
  finalLog = dateStr +" "+ log;
  char debug_buf[500];
  va_start (ap, log);
  vfprintf (stderr, (const char*)finalLog, ap);
  va_end (ap);
  va_start (ap, log);
  vsnprintf (debug_buf, 499, (const char*)finalLog, ap);
  va_end (ap);
  emit signalDebug(debug_buf);
}

void IaxWrapper::debugStatistics(int callNo, int callInterval)
{
  if (showStats) {
    iaxc_get_netstats(callNo, rtt, localNetstat, remoteNetstat);
    debug("----CALL %d [%s] [duration=%d]----\n",callNo,(const char*) callSession[callNo].getCallerIdName(), callInterval );
    debug("netstat local: jitter=%d, losspct=%d, losscnt=%d, packets=%d, delay=%d, dropped=%d, ooo=%d\n", localNetstat->jitter, localNetstat->losspct,localNetstat->losscnt,  localNetstat->packets, localNetstat->delay,localNetstat->dropped, localNetstat->ooo );
    debug("netstat remote: jitter=%d, losspct=%d, losscnt=%d, packets=%d, delay=%d, dropped=%d, ooo=%d\n", remoteNetstat->jitter, remoteNetstat->losspct,remoteNetstat->losscnt,  remoteNetstat->packets, remoteNetstat->delay,remoteNetstat->dropped, remoteNetstat->ooo );
  }
}

bool IaxWrapper::getShowStats() {
  return showStats;
}

void IaxWrapper::setShowStats(bool flag) {
  showStats = flag;
}

QString IaxWrapper::getIaxClientVersion() {
 return QString(iaxc_ver);
}
