/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

#ifndef KGRUBEDITOR_H
#define KGRUBEDITOR_H

//KDE
#include <kxmlguiwindow.h>

//Ui
#include "ui/ui_kgrubeditor.h"
#include "ui_settings_general.h"
#include "ui_settings_paths.h"

/**
 * @short Main window for KGRUBEditor.
 *
 * Handles everything that is GUI related.
 */
class KGRUBEditor : public KXmlGuiWindow, private Ui::KGRUBEditor
{
Q_OBJECT
public:
	/**
	* Default constructor.
	*
	* First, it creates the UI.
	* Secondly, it reads file input.
	* Thirdly, it show the input collected.
	*
	* The last 2 steps are done using refresh().
	*
	* @param parent QWidget parent. Self explanatory.
	*/
	KGRUBEditor( QWidget *parent = 0 );
private:
//SETUP
	/**
	* Sets up special objects such as timers etc.
	*/
	void setupObjects();
	/**
	* Setups all actions. It sets their text, icon, shortcut and
	* connects them to slots.
	*/
	void setupActions();
	/**
	* Setups the application's menus.
	*/
	void setupMenus();
	/**
	* Connects signals to slots. Action connections are
	* not done here. They are done in KGRUBEditor::setupActions().
	*/
	void setupConnections();
//GUI
	/**
	* Shows the input collected from all files.
	* Firsty it shows the GRUB Items in the GRUB Entries tab and
	* secondly it shows the GRUB Settings in the GRUB Settings tab.
	* It colors the default entry/entries using the 'defaultentry' color setting.
	*
	* It is only called through refresh() so every parameter filtering should be done there.
	*
	* @param entryListRow This row will be selected in the entry listbox after 
        *                     the GRUB Items are shown.
	*                     Defaults to 0 (first entry will be selected if not set).
	* @param mapListRow This row will be selected in the map listbox after 
        *                   the GRUB maps are shown.
	*                   Defaults to 0 (first entry will be selected if not set).
	* @param focusWidget The widget that will receive input focus. 0 = none.
	*/
	void showInput( const int entryListRow = 0, const int mapListRow = 0, QWidget* focusWidget = 0 );
	/**
	* Switches over to the GRUB Entries tab and sets the focus to the
	* listWidget_os listbox.
	*/
	void gotoMainView();
//OTHER
	/**
	* Outputs all the changes done to the GRUB Settings/Items
	* to GRUB's configuration file using fileOutput().
	*/
	void updateFile();
private slots:
	/**
	* Settings loaded:
	* @li paths used to retrieve input
	* @li default enrty's color used in KGRUBEditor::showInput
	*/
	void loadSettings();
	/**
	* Sets the listWidget_os current entry as default.
	*/
	void setDefault();
	/**
	* Sets the listWidget_os current entry as fallback.
	*/
	void setFallback();
//EVENTS
	/**
	* Called when the user right-clicks in the listWidget_os area.
	* Pop-ups a context menu showing entry-related actions.
	*/
	void entryContextMenu();
	/**
	* Called when the user selects another entry from the entry listbox
	* in order to update the info shown in the tree widget below, which
	* holds each entry's data.
	*
	* @param row Indicates which row has been selected, in order to know
	*            which entry's data to show.
	*/
	void entryRowChanged( const int row );

	/**
	* Called when the hiddenMenu checkbox is clicked,
	* in order to update the file's content.
	*/
	void checkBox_hiddenMenu_clicked();
	/**
	* Called when the timeout checkbox is clicked,
	* in order to enable/disable intSpinBox_timeout.
	*/
	void checkBox_timeout_clicked();
	/**
	* Called in order to update the file's content
	* according to intSpinBox_timeout state and value.
	*/
	void updateTimeout();
	/**
	* Called when the default checkbox is clicked,
	* in order to enable/disable comboBox_default.
	*/
	void checkBox_default_clicked();
	/**
	* Called in order to update the file's content
	* according to comboBox_default value.
	*
	* @param value The entry number to set default.
	*/
	void updateDefault( const int value );
	/**
	* Called when the default checkbox is clicked,
	* in order to enable/disable comboBox_fallback.
	*/
	void checkBox_fallback_clicked();
	/**
	* Called in order to update the file's content
	* according to comboBox_fallback value.
	*
	* @param value The entry number to set fallback.
	*/
	void updateFallback( const int value );
	/**
	* It launches the Password Assistant in order to create
	* a new password.
	*/
	void createPassword();
	/**
	* It launches the Password Assistant in order to edit
	* the current password.
	*/
	void editPassword();
	/**
	* It deletes the current password (the user is prompted first).
	*/
	void deletePassword();
	/**
	* Converts the generic splash image path to GRUB standard naming.
	* Eg. if /media/linux is recognised by GRUB as (hd0,0)
	*     then the path /media/linux/boot/splashimages/splash.xpm.gz
	*     would be converted to (hd0,0)/boot/splashimages/splash.xpm.gz.
	*
	* @param path The path to be converted.
	*/
	void convertSplashImagePath( const KUrl path );
	/**
	* Launches the Splash Image Creation Assistant in order to
	* create a new splash image.
	*/
	void createSplashImage();
	/**
	* Previews the splash image specified.
	*/
	void previewSplashImage();
	/**
	* Since the splash image line edit widget is read-only a means
	* to clear the splash image is required.
	*/
	void clearSplashImagePath();
	/**
	* Converts the generic gfxmenu path to GRUB standard naming.
	* Eg. if /media/linux is recognised by GRUB as (hd0,0)
	*     then the path /media/linux/boot/message
	*     would be converted to (hd0,0)/boot/message.
	*
	* @param path The path to be converted.
	*/
	void convertGfxmenuPath( const KUrl path );
	/**
	* Since the gfxmenu line edit widget is read-only a means
	* to clear the gfxmenu is required.
	*/
	void clearGfxmenuPath();
	/**
	* Enables the use of colors. Triggered when the 'Normal' check box is checked.
	*/
	void enableNormal( const bool state );
	/**
	* Enables the use of highlighted color. Triggered when the 'Highlight' check box is checked.
	*/
	void enableHighlight( const bool state );
	/**
	* Resets blinking properties to avoid asynchronous blinking.
	*/
	void blinkReset();
	/**
	* Updates the colors used in GRUB.
	*/
	void updateColors();
	/**
	* It launches the Map Assistant in order to add a new map.
	*/
	void addMap();
	/**
	* It launches the Map Assistant in order to edit the current map.
	*/
	void editMap();
	/**
	* Copies currently selected map to the end of the list and
	* updates the file contents.
	*
	* If no map is selected, it does nothing.
	*/
	void copyMap();
	/**
	* It deletes the currently selected map (the user is prompted first).
	*/
	void removeMap();
//ACTIONS
//file
	/** 
	* Displays a KFileDialog so that the user selects a file.
	* Then sets the 'menulst' setting to the file selected and 
	* calls KGRUBEditor::refresh().
	*
	* @see KGRUBEditor::refresh()
	*/
	void open();
	/**
	* Calls showInput() to update the info shown, after having called fileInput().
	* Should be called whenever we want to retrieve new file input
	* and update the info currently shown.
	*
	* @see KGRUBEditor::showInput()
	*/
	void refresh( int entryListRow = 0, int mapListRow = 0, QWidget* focusWidget = 0 );
//actions
	/**
	* It launches the Entry Assistant in order to create
	* a new entry.
	*/
	void add();
	/**
	* It launches the Entry Assistant in order to edit
	* the current entry.
	*/
	void edit();
	/**
	* Copies currently selected entry to the end of the list and
	* updates the file contents.
	*
	* If no entry is selected, it does nothing.
	*/
	void copy();
	/**
	* Removes current entry after prompting the user if he is sure.
	* After this, it updates file contents.
	*
	* If no entry is selected, it does nothing.
	*/
	void remove();
	/**
	* Moves current entry to the top of the enrty list.
	*/
	void moveTop();
	/**
	* Moves current entry one seat up.
	*/
	void moveUp();
	/**
	* Moves current entry one seat down.
	*/
	void moveDown();
	/**
	* Moves current entry to the bottom of the enrty list.
	*/
	void moveBottom();
	/**
	* Launches the Backup Assistant.
	*/
	void backup();
	/**
	* Launches the Restore Assistant.
	*/
	void restoreBackup();
	/**
	* Prompts the user to select a file and deletes it.
	*/
	void deleteBackup();
//tools
	/**
	* Creates a KDialog that shows all Device-relates input collected,
	* such as partition mountpoint, UUID, grub naming etc.
	*
	* For more info chech the Device class:
	* @see Device
	*/
	void view();
//settings
	/**
	* Toggles menubar.
	* If it is shown, it gets hidden.
	* If it is hidden, it gets shown.
	*/
	void toggleMenubar();
	/**
	* Shows the Configuration dialog.
	*/
	void preferences();
private:
	Ui::settings_general ui_settings_general;
	Ui::settings_paths ui_settings_paths;

	QTimer *timer_blink;

	GRUB::ConfigFile::Settings GrubSettings;
public:
	static QVector<GRUB::ConfigFile::Entry> GrubEntries;
	static QVector<GRUB::Misc::Device> DeviceList;
};

#endif
