/*  $Id: marsh.c 2293 2005-12-03 19:51:13Z rra $
**
**  Routines to marshall/unmarshall parameters and ticket-granting tickets.
*/

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#ifdef HAVE_KERBEROSIV_KRB_H
# include <kerberosIV/krb.h>
#else
# include <krb.h>
#endif

#include "kftgt.h"

/*
 * These macros were lifted from BIND 4.9.2 sources. 
 *
 */

/*
 * Inline versions of get/put short/long.  Pointer is advanced.
 * We also assume that a "u_short" holds 2 "chars"
 * and that a "u_int32_t" holds 4 "chars".
 *
 * These macros demonstrate the property of C whereby it can be
 * portable or it can be elegant but never both.
 */

/* these should work on all hosts I have to worry about :-) rjs */
typedef unsigned char my_u_char;
typedef unsigned int my_u_int32_t;
typedef unsigned short my_u_short;

#define GETSHORT(s, cp) { \
	register my_u_char *t_cp = (my_u_char*)(cp); \
	(s) = (((my_u_short)t_cp[0]) << 8) \
	    | (((my_u_short)t_cp[1])) \
	    ; \
	(cp) += 2; \
}

#define GETLONG(l, cp) { \
	register my_u_char *t_cp = (my_u_char*)(cp); \
	(l) = (((my_u_int32_t)t_cp[0]) << 24) \
	    | (((my_u_int32_t)t_cp[1]) << 16) \
	    | (((my_u_int32_t)t_cp[2]) << 8) \
	    | (((my_u_int32_t)t_cp[3])) \
	    ; \
	(cp) += 4; \
}

#define PUTSHORT(s, cp) { \
	register my_u_short t_s = (my_u_short)(s); \
	register my_u_char *t_cp = (my_u_char*)(cp); \
	*t_cp++ = t_s >> 8; \
	*t_cp   = t_s; \
	(cp) += 2; \
}

/*
 * Warning: PUTLONG --no-longer-- destroys its first argument.  if you
 * were depending on this "feature", you will lose.
 */
#define PUTLONG(l, cp) { \
	register my_u_int32_t t_l = (my_u_int32_t)(l); \
	register my_u_char *t_cp = (my_u_char*)(cp); \
	*t_cp++ = t_l >> 24; \
	*t_cp++ = t_l >> 16; \
	*t_cp++ = t_l >> 8; \
	*t_cp   = t_l; \
	(cp) += 4; \
}

/*****************************************************
 nobody to blame but myself for these lovely macros.

   I could have been more paranoid about the PUT macros
   but I'm assuming a valid cred stucture was passed to 
   cred_to_net.

******************************************************/

/*
   put a string with length stored first, string is not null terminated 


*/
  
#define PUT_COUNTED_STRING(string,dest) {       \
    KRB_UINT32 temp;                            \
    temp = strlen(string);                      \
    PUTLONG(temp,dest);                         \
    strncpy(dest,string,temp);                  \
    dest+= temp;                                \
  }

/* put arbitrary data */

#define PUT_BINARY(dest, src, len) { \
    memcpy(dest, src, len); \
    dest += len; \
  }

/* check len against max length  */

#define CHECK_LENGTH(len,max_len) { \
     if (len > max_len) return -1; \
  }

/* get a long, checking for space (in source, dest is already a long) */

#define GET_LONG(dest,src,cur_len,max_len) {\
  CHECK_LENGTH(cur_len+4, max_len); \
  GETLONG(dest,src);\
  cur_len += 4; \
  }

/* get a counted string, checking for space in source and dest */

#define GET_COUNTED_STRING(dest,src,dest_max, cur_len,max_len) {        \
    KRB_UINT32 count;                                                   \
    GET_LONG(count, src, cur_len, max_len);                             \
    CHECK_LENGTH(count, dest_max);                                      \
    CHECK_LENGTH(cur_len+count, max_len);                               \
    strncpy(dest,(char *) p,count);                                     \
    dest[count] = '\0';                                                 \
    cur_len += count;                                                   \
    src += count;                                                       \
  }


/* get arbitrary data, checking for space in source, dest length is given */

#define GET_BINARY(dest,src, len, cur_len, max_len) {\
  CHECK_LENGTH(cur_len + len, max_len); \
  memcpy(dest, src, len); \
  cur_len += len; \
  src += len; \
  } 

int 
marshall_cred (
       CREDENTIALS *cred,
       char *buffer,
       int max_buff)
{
  char *p;
  int len;
  KRB_UINT32 temp;

   p = buffer;

   if (max_buff < sizeof(CREDENTIALS)) return -1;

  PUTLONG(1, p); /* version */
  PUT_COUNTED_STRING(cred->service,p);
  PUT_COUNTED_STRING(cred->instance,p);
  PUT_COUNTED_STRING(cred->realm,p);
  PUT_BINARY(p, cred->session, sizeof(cred->session));
  temp = cred->lifetime;
  PUTLONG(temp,p);
  temp = cred->kvno;
  PUTLONG(temp,p);
  temp = cred->ticket_st.length; 
  PUTLONG(temp,p);
  PUT_BINARY(p, cred->ticket_st.dat, temp);
  temp = cred->issue_date;   
  PUTLONG(temp,p);
  PUT_COUNTED_STRING(cred->pname,p);
  PUT_COUNTED_STRING(cred->pinst,p);  

  len =  p-buffer;

  return len;

}

int 
unmarshall_cred(
       CREDENTIALS *cred,
       unsigned char *buffer,
       int max_len)
{

  unsigned char *p;
  int len;
  KRB_UINT32 temp, version;

  len = 0;
  p = buffer;

  GET_LONG(version, p, len, max_len);

  GET_COUNTED_STRING(cred->service, p, sizeof(cred->service)-1, len, max_len);
  GET_COUNTED_STRING(cred->instance, p, sizeof(cred->instance)-1,len, max_len);
  GET_COUNTED_STRING(cred->realm, p, sizeof(cred->realm)-1, len, max_len);

  GET_BINARY(cred->session, p, sizeof(cred->session), len, max_len);
  GET_LONG(temp, p, len, max_len); cred->lifetime = temp;
  GET_LONG(temp, p, len, max_len); cred->kvno = temp;
  GET_LONG(temp, p, len, max_len); cred->ticket_st.length = temp;
  CHECK_LENGTH(temp, sizeof(cred->ticket_st.dat));
  GET_BINARY(cred->ticket_st.dat, p, temp, len, max_len);
  cred->ticket_st.mbz = 0;
  GET_LONG(temp, p, len, max_len); cred->issue_date = temp;
  GET_COUNTED_STRING(cred->pname, p, sizeof(cred->pname)-1, len, max_len);
  GET_COUNTED_STRING(cred->pinst, p, sizeof(cred->pinst)-1, len, max_len);
   
  return len;
}


int marshall_params (
      char *user,
      char *ticketfile,
      char *buffer,
      int  max_buffer)
{
   int ulen, tlen;
   char *p;

   ulen = strlen(user);
   tlen = strlen(ticketfile);

   if (  (ulen > KFTGT_MAX_USERNAME)
       ||(tlen > KFTGT_MAX_PATHNAME)
       || (ulen+tlen+8) > max_buffer) return -1;

  p = buffer;
  PUT_COUNTED_STRING(user,p);
  PUT_COUNTED_STRING(ticketfile,p);

  return p-buffer;

}


int unmarshall_params(
      char *user,
      int max_user,
      char *ticketfile,
      int max_ticket,
      unsigned char *buffer,
      int  max_len)
{
    unsigned char *p;
    int len;
 
    p = buffer;
    len = 0;
    GET_COUNTED_STRING(user, p, max_user, len, max_len);
    GET_COUNTED_STRING(ticketfile, p, max_ticket, len, max_len);
    return len;
}
