/*  $Id: kftgt.c 2303 2005-12-22 00:58:47Z rra $
**
**  Client program for Kerberos v4 ticket forwarding.
*/

/*
 * INCLUDES
 */

#include "config.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <memory.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <stdarg.h>

#ifdef HAVE_KERBEROSIV_KRB_H
# include <kerberosIV/krb.h>
#else
# include <krb.h>
#endif

#include "kftgt.h"
#include "encrypt.h"
#include "marsh.h"

/** Exit values for various errors. */
enum kftgt_error {
    SUCCESS             = 0,
    ERROR_NOHOST        = 1,
    ERROR_TIMEOUT       = 2,
    ERROR_CONNECT       = 3,
    ERROR_KERBEROS      = 4,
    ERROR_SEND          = 5,
    ERROR_SYSTEM        = 6
};

/*
 * PROTOTYPES
 */
char **read_options(int argc, char **argv);
enum kftgt_error kftgt(char *user, char *server);
char *parse_username(char *server,char **ruser);
int read_null_term(int s, char *buffer, int max);
void warn (char *fmt, ...);
int print_version(void);
int usage(int exitcode);

/*
 * GLOBALS
 */
static char *prog = "kftgt";
static int debug=0;
static int quiet=0;
static char *l_val="";
static char *f_val = "";
static int t_val=60;  /* default timeout */

/*
 * FUNCTIONS
 */

RETSIGTYPE 
timeout(int signo)
{
  warn("timeout");
  /* FIXME */
  exit(ERROR_TIMEOUT);
}

int
main(int argc, char **argv)
{
  enum kftgt_error status = SUCCESS;

#ifdef HAVE_DECL_KRB_IGNORE_IP_ADDRESS
  krb_ignore_ip_address = 1;
#endif

  argv = read_options(argc, argv);
  while(*argv) {
    char *ruser = l_val, *server;
    server = parse_username(*argv,&ruser);
    status = kftgt(ruser, server);
    argv++;
  }
  exit(status);
}

char *
parse_username(char *server,char **ruser)
{
  char *retval = server;
  char *cp;

  if ((cp = strchr(server,'@')) != NULL) {
    retval = cp+1;
    *cp = '\0';
    *ruser = server;
  } else {
    *ruser = l_val;
  }
  return retval;
}

char **
read_options(int argc, char **argv)
{
  int c;
  extern int opterr;
  opterr = 0;

  prog = argv[0];

  /* A quick hack to honor --help and --version */
  if (argv[1])
    if (argv[1][0] == '-' && argv[1][1] == '-' && argv[1][2] != '\0') {
      switch(argv[1][2]) {
      case 'h':
	usage(0);
	break;
      case 'v':
	print_version();
	break;
      default:
	usage(1);
	break;
      }
    }
 
  while ((c = getopt(argc, argv, "hl:f:vqdt:")) != EOF) {
    switch (c) {
    case 'd': /* Debug */
      debug=1;
      break;
    case 'f': /* Ticket File */
      f_val=optarg;
      break;
    case 'h': /* Help */
      usage(0);
      break;
    case 'l': /* User name */
      l_val=optarg;
      break;
    case 'q': /* Quiet */
      quiet=1;
      break;
    case 't': /* Timeout */
      t_val=atoi(optarg);
      break;
    case 'v':
      print_version();
      break;
    default:
      usage(1);
      break;
    }
  }

  if (t_val <= 0 || optind >= argc) {
    usage(1);
  }

  return argv+optind;
}

enum kftgt_error
kftgt(char *ruser, char *server)
{
  struct hostent *hp;
  struct sockaddr_in saddr, caddr;
  struct servent *se;
  unsigned int addr;
  size_t clen;
  char *remote_host;
  int status;
  int sock, len;
  KTEXT_ST ticket;

  MSG_DAT msg_data;
  CREDENTIALS cred;
  CREDENTIALS tgtcred;
  Key_schedule sched;
  char buffer[KFTGT_MAX_BUFFER];
  MSG_DAT m_data;

  /* set alarm for timeout */
  /* FIXME */
  signal(SIGALRM, timeout);
  alarm(t_val);

  /*** Connect ***/

  /* Assign port */
  memset ((char *) &saddr, 0, sizeof (struct sockaddr_in));
  if ((se = getservbyname ("kftgt", "tcp")) != NULL ||
      (se = getservbyport(htons(SERVICE_PORT),"tcp")) != NULL) {
    saddr.sin_port = se->s_port;
  } else {
    saddr.sin_port = htons(SERVICE_PORT);
  }
  endservent();

  /* First check if valid IP address.  Otherwise check if valid name. */
  if ((addr = inet_addr(server)) != -1) {
    if ((hp = gethostbyaddr ((char *)&addr, sizeof(unsigned int),
			     AF_INET)) == NULL) {
      if (!quiet) {
	fprintf(stderr,"%s: unknown host",server);
      }
      return ERROR_NOHOST;
    }
  } else if ((hp = gethostbyname (server)) == NULL) {
    if (!quiet) {
      fprintf(stderr,"%s: unknown host",server);
    }
    return ERROR_NOHOST;
  }

  /* Set up socket connection */
  saddr.sin_family = AF_INET;
  memcpy (&saddr.sin_addr, hp->h_addr, sizeof(hp->h_addr));

  if ((sock = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
      if (!quiet) {
	perror("socket");
      }
    return ERROR_SYSTEM;
  }

  if (connect (sock, (struct sockaddr *) &saddr,
	       sizeof(struct sockaddr_in)) < 0 ) {
    if (!quiet) {
      perror("connect");
    }
    (void)close(sock); /* Back out */
    return ERROR_CONNECT;
  }

  /* copy the hostname into non-volatile storage */
  remote_host = (char*)malloc(strlen(hp->h_name) + 1);
  strcpy(remote_host, hp->h_name);

  /* find out who I am */
  clen = sizeof(caddr);
  if (getsockname(sock, (struct sockaddr *) &caddr, &clen) < 0) {
    if (!quiet)
      perror("getsockname");
    close(sock);
    return ERROR_SYSTEM;
  }

  /*
   * call Kerberos library routine to obtain an authenticator,
   * pass it over the socket to the server, and obtain mutual
   * authentication.
   */

  status = krb_sendauth((KRB_INT32)KOPT_DO_MUTUAL, sock, &ticket,
			SERVICE_PRINCIPAL,
			remote_host,
			krb_realmofhost(remote_host), 
			getpid(), 
			&msg_data,
			&cred,
			sched,
			&caddr, &saddr, KFTGT_PROTO_VERSION);

  if (status != KSUCCESS) {
    warn("cannot authenticate to server: %s",krb_err_txt[status]);
    close(sock);
    return ERROR_KERBEROS;
  }

  status = read_null_term(sock, buffer, sizeof(buffer));
  if (status < 0) {
    warn("read_null_term failed");
    close(sock);
    return ERROR_SEND;
  }

  if (strncmp(buffer,"ok",2)!=0) {
    warn("%s",buffer);
    close(sock);
    return ERROR_SEND;
  }

  /*
   * send over remote user name and ticket file, both NULL terminated
   * and encrypted
   */

  if ( (len = marshall_params(ruser, f_val, buffer, sizeof(buffer))) <0) {
    warn("unable to marshall params");
    close(sock);
    return ERROR_SYSTEM;
  }

  if (send_encrypted_chunk(buffer, len, sock, 
			   &cred.session, sched, &caddr, &saddr) < 0) {
    warn("error sending encrypted params");
    close(sock);
    return ERROR_SEND;
  }

  /* read encrypted response */

  len = receive_encrypted_chunk(&m_data, buffer, sizeof(buffer), sock,
				&cred.session, sched, &saddr, &caddr);

  if (strncmp((char *) m_data.app_data,"ok",2)!=0) {
    if (len > 0)
      warn("%s",m_data.app_data);
    else
      warn("error receiving encrypted chunk");
    close(sock);
    return ERROR_SEND;
  }

  /* now send tgt */
  /* get realm from cred for now */

  if (krb_get_cred("krbtgt",cred.realm,cred.realm,&tgtcred) != GC_OK) {
    warn("cannot get tgt from local ticket file");
    close(sock);
    return ERROR_KERBEROS;
  }

  len = marshall_cred(&tgtcred,buffer,sizeof(buffer));

  if (len <= 0 || len >  KFTGT_MAX_BUFFER) {
    warn("marshall of tgt failed");
    close(sock);
    return ERROR_SEND;
  }

  if (send_encrypted_chunk(buffer, len, sock, 
			   &cred.session, sched, &caddr, &saddr) < 0) {
    warn("error sending encrypted user name");
    close(sock);
    return ERROR_SEND;
  }

  len = receive_encrypted_chunk(&m_data,buffer,sizeof(buffer), sock,
				&cred.session, sched, &saddr, &caddr);

  if (strncmp((char *) m_data.app_data,"ok",2)!=0) {
    if (len > 0)
      warn("%s",m_data.app_data);
    else
      warn("error receiving encrypted chunk");
    close(sock);
    return ERROR_SEND;
  }

  if (!quiet) {
    printf("%s: tgt %s.%s@%s forwarded to ",
	   prog,
	   cred.pname,
	   cred.pinst,
	   cred.realm);
    if (ruser[0]) printf("%s at ",ruser);
    printf("%s\n", server);
  }
  close(sock);
  /* FIXME */
  alarm(0);
  return SUCCESS;
}

int
read_null_term(int s, char *buffer, int max)
{
  int len=0;
  while (max>0) {
    read(s, buffer, 1);
    len++;
    if ( *buffer == 0 ) return len;
    buffer++;
    --max;
  }
  return -1;
}

/* warn
 *
 * Print error message, terminate connection and download, and exit
 * Input:
 *	fmt		format of error message in printf style
 *	...		% arguments supplied for fmt
 * Returns:
 *	nothing
 */
void
warn (char *fmt, ...)
{
  va_list argptr;

  if (!quiet)
    fprintf(stderr,"%s: ",prog);

  va_start(argptr,fmt);
  if (!quiet)
    vfprintf(stderr,fmt,argptr);
  va_end(argptr);

  if (!quiet)
    fprintf(stderr,"\n");
}

int
print_version(void)
{
  fprintf(stderr,"%s version %s\n", prog, PACKAGE_VERSION);
  exit(0);
}

int
usage(int exitcode)
{
  fprintf(stderr,"Usage: %s [options] [user@]host [user2@host2 ...]\n",prog);
  fprintf(stderr,"   -l user     remote user to forward tickets to\n");
  fprintf(stderr,"   -f file     remote ticket filename\n");
  fprintf(stderr,"   -t secs     timeout, default is 60 seconds\n");
  fprintf(stderr,"   -v          version\n");
  fprintf(stderr,"   -q          quiet\n");
  fprintf(stderr,"   -d          debug info\n");
  fprintf(stderr,"\n");
  exit(exitcode);
}
