/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Specialized list to display a list of photos
 *
 * Copyright 2005 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */
#ifndef PHOTOLISTVIEW_H
#define PHOTOLISTVIEW_H

#include <kurl.h>
#include <kfileitem.h>

#include <qtimer.h>
#include <qpixmap.h>
#include <klistview.h>
#include <qdragobject.h>


namespace KIO { class Job; class PreviewJob; }

class Photo;
class QFile;
class PhotoListViewItem;

/**
 * Provides all the functionality of a KListViewItem object with the addition
 * of an image preview mechanism and specialized signals.
 * @short KListView to suit our needs.
 * @author M. Asselstine <asselsm@gmail.com>
 */
class PhotoListView : public KListView
{
Q_OBJECT

public:
    PhotoListView(QWidget *parent, const char *name);
    ~PhotoListView();

    /** Retrieve the border image to draw around each photograph
      * @return A pixmap of the border image.
      */
    const QPixmap& getBorderImage() { return m_border; }

    /** Retrieve the number of selected items.
      * @return The number of selected items
      */
    int numSelected() const;

    /** Remove all selected photos from the list */
    void removeSelected();

    /** Do a backup of all the photos in the list such that if there is a failure the user can
     *  possibly restore some of their work.
     *  @param f Name of Backup file
     */
    void doBackup(const QString& filename);

    /** Do a restore from the backup created by doBackup(). This function is part of the system
     *  used to allow the user to quickly recover from an upload that has crashed kflickr or
     *  if the user accidentally exited kflickr before uploading loaded photos.
     *  @param f Name of Backup file
     */
    void doRestore(const QString& filename);

protected:
    virtual QDragObject* dragObject();
    virtual void resizeEvent(QResizeEvent * e);
    virtual void viewportPaintEvent(QPaintEvent *e);
    virtual bool acceptDrag(QDropEvent *event) const;

public slots:
    /** Change the selection to select the next photograph */
    void selectNext();

    /** Change the selection to select the previous photograph */
    void selectPrevious();

private slots:
    /** Adds a request for a preview to the mix. We try to aggregrate
      * previews so that we can retrieve multiple previews with one
      * job. This is the start of this process and the actual preview
      * jobs are started in startPreviewJob().
      * @param item Pointer to the item added to the list
      */
    void addPreviewRequest(QListViewItem* item);

    /** A preview job has completed. We are not interested in the
      * exit status as we are not setup to handle errors we are
      * only interested in removing the job from out job tracker.
      * @param job Pointer to the preview job that is completing
      */
    void jobResult(KIO::Job* job);

    /** A preview job has hit some failure condition for one file. We handle
      * the failure by displaying a broken file icon instead of the image
      * preview.
      * @param fi The file the preview was for
      */
    void jobFailed(const KFileItem* fi);

    /** A preview job has returned a preview for one of the files.
      * Update the Photo in the list which corresponds to the file.
      * @param fi The file the preview is for
      * @param pm The preview pixmap
      */
    void gotPreview(const KFileItem* fi, const QPixmap& pm);

    /** Start a preview job to get one or many image previews. This
      * function will trigger by the aggregatorTimer to actually
      * start the PreviewJob.
      */
    void startPreviewJob();

    /** Call to display the list's context menu when MB3 is pressed
      * @param item List item under the mouse cursor
      * @param pt Point relative to the application window where the cursor is
      * @param ?? Unused
      */
    void showRMBMenu(QListViewItem* item, const QPoint &pt, int);

private:
    QTimer m_previewTimer;                      /// Timer used to get previews into one job
    QPtrList<KIO::Job> m_jobs;                  /// List of outstanding preview jobs
    QPtrList<QListViewItem> m_previewItems;     /// The list of items requiring a preview job

    QPixmap m_border;                           /// The pixmap used as the border image
};


/**
 * Provides all the functionality of a KListViewItem with some specific additions
 * to allow for image preview and setting and retrieval of values in the context
 * of their use instead of simply via index.
 * @short KListViewItem to suit our needs.
 * @author M. Asselstine <asselsm@gmail.com>
 */
class PhotoListViewItem : public QObject, public KListViewItem
{
Q_OBJECT

public:
    /**
     * Constructor.
     * @param parent Pointer to parent which is the listview that contains the this item.
     * @param url URL of the file containing the photo for this item.
     */
    PhotoListViewItem(PhotoListView *parent, const KURL& url);

    /**
     * Constructor.
     * @param parent Pointer to parent which is the listview that contains the this item.
     * @param url URL of the file containing the photo for this item.
     * @param after Pointer to another item which we will add this item after in the list order.
     */
    PhotoListViewItem(PhotoListView *parent, const KURL& url, PhotoListViewItem* after);

    /**
     * Destructor
     */
    ~PhotoListViewItem();

    /**
     * Common initialization for all constructors.
     */
    void init();

    /**
     * Called by the containing listview to determine how much space this item needs. We
     * override this function to fine tune how this listview item will be displayed.
     * @return The width this listview item uses.
     */
    virtual int width(const QFontMetrics &, const QListView *, int c) const;

    /**
     * Override the default painting of the item. We do this in order to display the preview image
     * in a nice border.
     */
    virtual void paintCell(QPainter *, const QColorGroup &cg, int column, int width, int alignment);

    /**
     * Use to get the photo this list item is associated with.
     * @return Reference to Photo this item is linked to.
     */
    Photo& photo() const {return m_photo;}

    /**
     * Called when the items selection state changes. We override this to notify the Photo object
     * the new selection state since in our architecture we query photos rather then items for
     * selection state.
     * @param s The new selection state.
     */
    virtual void setSelected(bool s);

protected:
    /**
     * Call to set a new height for the item.
     * @param height The new item height.
     */
    void setHeight(int height);

private slots:
    /**
     * Call to update the items appearance. Currently this is only the preview image but in
     * the future I am hoping to include icons in the item to display privacy and other photo
     * properties.
     * @param flag The update flag indicating which photo properties have changed.
     */
    void updateItem(Q_UINT16 flag);

private:
    Photo &m_photo;                 /// The Photo this item is linked to.
    friend class PhotoListView;
};

#endif
