/* ========================================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Description : Retrieve previews of photographs in a semi organized way
 *
 * Copyright 2007-2008 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ===================================================================== */

#include "previewmgr.h"

#include <QMap>
#include <QList>
#include <QSize>
#include <QTimer>
#include <QPixmap>

#include <KIcon>
#include <KApplication>

// Private data class
class PreviewMgr::PrivateData
{
public:
  PrivateData()
    : defaultPixmap(*(new QPixmap(KIcon("camera-photo").pixmap(32,32))))
  {
  }

  ~PrivateData()
  {
    previews.clear();
    delete &defaultPixmap;
  }

  QTimer timer;                        /// Timer used to get previews into one job
  KUrl::List urls;                     /// The list of URLs we are fetching previews for
  QMap<KUrl, quint32> idmap;

  QPixmap& defaultPixmap;
  QMap<quint32,QPixmap> previews;
};

PreviewMgr::PreviewMgr(QObject* parent)
  : QObject(parent)
  , d(*(new PreviewMgr::PrivateData()))
{
  d.timer.setSingleShot(true);

  // Create connections
  connect(&d.timer, SIGNAL(timeout()), SLOT(startPreviewJob()));
}

PreviewMgr::~PreviewMgr()
{
  delete &d;
}

PreviewMgr* PreviewMgr::instance()
{
  static PreviewMgr* singleton = 0;

  if (singleton == 0)
  {
    singleton = new PreviewMgr(KApplication::kApplication());
  }
  return singleton;
}

const QPixmap* PreviewMgr::defaultPreview() const
{
  return &(d.defaultPixmap);
}

const QPixmap* PreviewMgr::preview(int id) const
{
  if( !d.previews.contains(id) )
    return 0;

  return &(d.previews[id]);
}

PreviewMgr::PhotoOrientation PreviewMgr::orientation(int id, int rotation) const
{
  // Fallback to landscape
  PhotoOrientation orient = Landscape;

  if( d.previews.contains(id) )
  {
    QSize size = (d.previews[id]).size();
    if( size.width() < size.height() )
      orient = Portrait;

    // take into account the rotation
    if( rotation == 90 || rotation == 270 )
    {
      if( orient == Portrait )
      {
	orient = Landscape;
      }
      else
      {
	orient = Portrait;
      }
    }
  }

  return orient;
}

void PreviewMgr::previewRequest(int id, const KUrl& url)
{
  // Reset the timer used to group previews to allow for more requests to be
  // added to this job.
  d.timer.start(250);

  // Add to our list of URLs this new URL. This is part of the process of
  // attempting to group many preview requests into a single job.
  if( !d.urls.contains(url) )
  {
    d.idmap[url] = id;
    d.urls.append(url);
  }

  // Set (or possibly reset) to the default pixmap
  d.previews[id] = d.defaultPixmap;
  emit updatedPreview(id);
}

void PreviewMgr::startPreviewJob()
{
  KIO::PreviewJob *job;

  // start the IO slave
  job = KIO::filePreview(d.urls, 170, 170, 0, 0, true, false);
  job->setIgnoreMaximumSize(true);

  // make notification connections
  connect(job, SIGNAL(gotPreview(const KFileItem&, const QPixmap&)),
	  SLOT(gotPreview(const KFileItem&, const QPixmap&)));
  connect(job, SIGNAL(failed(const KFileItem&)), SLOT(jobFailed(const KFileItem&)));

  // clear out our URL list
  d.urls.clear();
}

void PreviewMgr::jobFailed(const KFileItem& fi)
{
  KUrl url = fi.url();
  d.idmap.remove(url);
  
}

void PreviewMgr::gotPreview(const KFileItem& fi, const QPixmap& pm)
{
  KUrl url = fi.url();
  quint32 id = d.idmap[url];

  d.previews[id] = pm;
  emit updatedPreview(id);

  d.idmap.remove(url);
}
