/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Description : Widget to edit a single photo's properties.
 *
 * Copyright 2008 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "editorwidget.h"

#include <QDebug>
#include <QRegExp>
#include <QStringList>
#include <QSqlTableModel>
#include <QModelIndexList>
#include <QStringListModel>

#include "previewmgr.h"
#include "ui_editorwidget.h"

// Private data class
class EditorWidget::PrivateData
{
public:
  PrivateData()
    : editRow(-1)
    , model(0)
    , sizes(QStringList())
  {
    QStringList str;
    str << i18n("Square") << i18n("Thumb") << i18n("Small") << i18n("Medium") << i18n("Large");
    str << i18n("Original") << i18n("Custom");
    sizes.setStringList(str);
  }
  ~PrivateData()
  {
  }

  QModelIndex getIndex(const QString& field)
  {
    Q_ASSERT( model && editRow >= 0 && editRow < model->rowCount() );
    return model->index(editRow, model->fieldIndex(field));
  }

  void setData(const QString& field, const QVariant& data)
  {
    if( !model || editRow < 0 || editRow > model->rowCount() )
      return;

    model->setData(model->index(editRow, model->fieldIndex(field)), data);
  }

  void tuneWidthAndHeight()
  {
    if( !model || editRow < 0 )
      return;

    if( ui.licenses->currentText() != i18n("Custom") )
    {
      int rotation(model->data(getIndex("rotation")).toInt());
      PreviewMgr::PhotoOrientation current(PreviewMgr::Landscape);
      
      if( ui.width->value() < ui.height->value() )
	current = PreviewMgr::Portrait;

      if( PreviewMgr::instance()->orientation(model->data(getIndex("id")).toInt()
					      , rotation) != current )
      {
	int tmp = ui.width->value();
	ui.width->setValue(ui.height->value());
	ui.height->setValue(tmp);
      }
    }
  }

public:
  Ui::editorwidget ui;              /// The widget UI

  int editRow;
  QSqlTableModel* model;
  QStringListModel tags;
  QStringListModel sizes;
};

EditorWidget::EditorWidget(QWidget* parent, Qt::WindowFlags f)
  : QFrame(parent, f)
  , d(*(new EditorWidget::PrivateData))
{
  d.ui.setupUi(this);

  d.ui.tags->setModel(&d.tags);
  d.ui.sizes->setModel(&d.sizes);
  d.ui.desc->setCheckSpellingEnabled(true);

  d.ui.rotateCW->setGuiItem(KGuiItem("","rotate_cw"));
  d.ui.rotateCCW->setGuiItem(KGuiItem("","rotate_ccw"));

  connect(d.ui.tags->selectionModel(),
	  SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	  SLOT(tagsSelectionChanged(const QItemSelection&, const QItemSelection&)));
}

EditorWidget::~EditorWidget()
{
  delete &d;
}

void EditorWidget::setModel(QSqlTableModel* model)
{
  d.model = model;
}

void EditorWidget::editRow(int row)
{
  int index;
  QString str;

  d.editRow = row;

  // Set the title and description
  d.ui.title->setText(d.model->data(d.getIndex("title")).toString());
  d.ui.desc->setText(d.model->data(d.getIndex("description")).toString());

  // Set the size data
  str = d.model->data(d.getIndex("size")).toString();
  index = static_cast<QStringListModel*>(d.ui.sizes->model())->stringList().indexOf(str);
  if( index < 0 )
  {
    index = 5;
  }
  if( index != d.ui.sizes->currentIndex() )
  {
    d.ui.sizes->setCurrentIndex(index);
  }
  else
  {
    d.tuneWidthAndHeight();
  }
  

  // Set the license
  str = d.model->data(d.getIndex("license")).toString();
  index = static_cast<QStringListModel*>(d.ui.licenses->model())->stringList().indexOf(str);
  if( index < 0 )
  {
    index = 0;
  }
  if( index != d.ui.licenses->currentIndex() )
  {
    d.ui.licenses->setCurrentIndex(index);
  }

  // Set the photoset
  str =  d.model->data(d.getIndex("photoset")).toString();
  index = static_cast<QStringListModel*>(d.ui.photosets->model())->stringList().indexOf(str);
  if( index >= 0 )
  {
    d.ui.photosets->setCurrentIndex(index);
  }
  else
  {
    d.ui.photosets->setEditText(str);
  }

  // Set the correct privacy radio button
  bool exposed = d.model->data(d.getIndex("exposed")).toBool();
  bool family  = d.model->data(d.getIndex("family")).toBool();
  bool friends = d.model->data(d.getIndex("friends")).toBool();
  if( exposed )
  {
    d.ui.pub->setChecked(true);
    d.ui.priv->setChecked(false);
    d.ui.family->setEnabled(false);
    d.ui.friends->setEnabled(false);
  }
  else
  {
    d.ui.pub->setChecked(false);
    d.ui.priv->setChecked(true);
    d.ui.family->setEnabled(true);
    d.ui.friends->setEnabled(true);
  }
  d.ui.family->setChecked(family);
  d.ui.friends->setChecked(friends);

  // Set the tags
  d.tags.setStringList(d.model->data(d.getIndex("tags")).toString().split(','));
}

void EditorWidget::setTagsModel(QStringListModel& model)
{
  d.ui.availableTags->setModel(&model);
}

void EditorWidget::setLicensesModel(QStringListModel& model)
{
  d.ui.licenses->setModel(&model);
}

void EditorWidget::setPhotosetsModel(QStringListModel& model)
{
  d.ui.photosets->setModel(&model);
}

void EditorWidget::on_title_textEdited(const QString& text)
{
  // Store the new value
  d.setData("title", QVariant(text));
}

void EditorWidget::on_rotateCW_clicked()
{
  int rcurr = d.model->data(d.getIndex("rotation")).toInt();
  if( rcurr == 270 )
  {
    d.setData("rotation", QVariant(0));
  }
  else
  {
    d.setData("rotation", QVariant(rcurr + 90));
  }

  // swap width and height if needed
  d.tuneWidthAndHeight();
}

void EditorWidget::on_rotateCCW_clicked()
{
  int rcurr = d.model->data(d.getIndex("rotation")).toInt();
  if( rcurr == 0 )
  {
    d.setData("rotation", QVariant(270));
  }
  else
  {
    d.setData("rotation", QVariant(rcurr - 90));
  }

  // swap width and height if needed
  d.tuneWidthAndHeight();
}

void EditorWidget::on_desc_textChanged()
{
  // Store the new value
  d.setData("description", QVariant(d.ui.desc->toPlainText()));
}

void EditorWidget::on_priv_clicked(bool b)
{
  // Store the new value
  d.setData("exposed", QVariant(!b));

  // Update widget states
  if( !b )
  {
    d.ui.family->setEnabled(false);
    d.ui.friends->setEnabled(false);
  }
  else
  {
    d.ui.family->setEnabled(true);
    d.ui.friends->setEnabled(true);
  }
}

void EditorWidget::on_pub_clicked(bool b)
{
  // Store the new value
  d.setData("exposed", QVariant(b));

  // Update widget states
  if( b )
  {
    d.ui.family->setEnabled(false);
    d.ui.friends->setEnabled(false);
  }
  else
  {
    d.ui.family->setEnabled(true);
    d.ui.friends->setEnabled(true);
  }
}

void EditorWidget::on_family_clicked(bool b)
{
  // Store the new value
  d.setData("family", QVariant(b));
}

void EditorWidget::on_friends_clicked(bool b)
{
  // Store the new value
  d.setData("friends", QVariant(b));
}

void EditorWidget::on_licenses_activated(int index)
{
  Q_UNUSED(index);

  // Store the new value
  d.setData("license", QVariant(d.ui.licenses->currentText()));
}

void EditorWidget::on_photosets_activated(int index)
{
  Q_UNUSED(index);

  // Store the new value
  d.setData("photoset", QVariant(d.ui.photosets->currentText()));
}

// We use currentIndexChanged() signal since we also want to handle programatic changes
// to the size type selection. We take this time to set the size spinbox values as well
// and ensure they are enabled or disabled accordingly.
void EditorWidget::on_sizes_currentIndexChanged(const QString& text)
{
    // Update the width and height spinboxes
  d.ui.width->setEnabled(false);
  d.ui.height->setEnabled(false);
  if( text == i18n("Square") )
  {
    d.ui.width->setValue(75);
    d.ui.height->setValue(75);
  }
  else if( text == i18n("Thumb") )
  {
    d.ui.width->setValue(100);
    d.ui.height->setValue(75);
  }
  else if( text == i18n("Small") )
  {
    d.ui.width->setValue(240);
    d.ui.height->setValue(180);
  }
  else if( text == i18n("Medium") )
  {
    d.ui.width->setValue(500);
    d.ui.height->setValue(375);
  }
  else if( text == i18n("Large") )
  {
    d.ui.width->setValue(1024);
    d.ui.height->setValue(768);
  }
  else if( text == i18n("Original") )
  {
    d.ui.width->setValue(74);
    d.ui.height->setValue(74);
  }
  else if( text == i18n("Custom") )
  {
    // Use the current width and height values as the starting place.
    d.ui.width->setValue(d.model->data(d.getIndex("width")).toInt());
    d.ui.height->setValue(d.model->data(d.getIndex("height")).toInt());
    d.ui.width->setEnabled(true);
    d.ui.height->setEnabled(true);
  }

  // Store the new size type
  d.setData("size", QVariant(text));

  // Swap width and height if necessary.
  d.tuneWidthAndHeight();
}

// Deal with any changes to the width spinbox value.
void EditorWidget::on_width_valueChanged(int value)
{
  // The only time the width and height values are actually used is when the Custom size
  // is selected. We use this to our advantage as it makes the size spinboxes more intuitive.
  if( d.ui.sizes->currentText() == i18n("Custom") && d.model && d.editRow >= 0 )
  { 
    d.setData("width", QVariant(value));
  }
}

// Deal with any changes to the height spinbox value.
void EditorWidget::on_height_valueChanged(int value)
{
  // The only time the width and height values are actually used is when the Custom size
  // is selected. We use this to our advantage as it makes the size spinboxes more intuitive.
  if( d.ui.sizes->currentText() == i18n("Custom") )
  {
    d.setData("height", QVariant(value));
  }
}

void EditorWidget::on_add_clicked(bool b)
{
  Q_UNUSED(b);

  QString str = d.ui.availableTags->currentText();
  QStringList list = d.tags.stringList().filter(QRegExp(".+"));

  if( str != QString::null && !list.contains(str) )
  {
    list << str;
    d.tags.setStringList(list);
    d.setData("tags", QVariant(list.join(",")));
  }
  d.ui.availableTags->clearEditText();
}

void EditorWidget::on_remove_clicked(bool b)
{
  Q_UNUSED(b);

  QModelIndexList list = d.ui.tags->selectionModel()->selection().indexes();
  qSort(list.begin(), list.end(), qGreater<QModelIndex>());

  foreach(QModelIndex index, list)
  {
    Q_ASSERT(index.row() >= 0 && index.row() < d.tags.rowCount());
    d.tags.removeRow(index.row());
  }

  QStringList tagList = d.tags.stringList().filter(QRegExp(".+"));
  d.setData("tags", QVariant(tagList.join(",")));
}

void EditorWidget::on_availableTags_editTextChanged(const QString& text)
{
  if( text.length() > 0 )
  {
    d.ui.add->setEnabled(true);
  }
  else
  {
    d.ui.add->setEnabled(false);
  }
}

void EditorWidget::on_availableTags_activated(const QString &text)
{
  QStringList list = d.tags.stringList().filter(QRegExp(".+"));

  if( text != QString::null && !list.contains(text) )
  {
    list << text;
    d.tags.setStringList(list);
    d.setData("tags", QVariant(list.join(",")));
  }
  d.ui.availableTags->clearEditText();
}

void EditorWidget::tagsSelectionChanged(const QItemSelection& selected,
					const QItemSelection& deselected)
{
  Q_UNUSED(selected);
  Q_UNUSED(deselected);

  if( d.ui.tags->selectionModel()->selection().size() > 0 )
    d.ui.remove->setEnabled(true);
  else
    d.ui.remove->setEnabled(false);

}
