/* ================================================================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Extend QSqlTableModel with D&D and other support
 *
 * Copyright 2008 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================================================== */

#include <QUrl>
#include <QDebug>
#include <QFileInfo>
#include <QMimeData>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QStringList>

#include <KLocale>

#include "previewmgr.h"

#include "sqlphototablemodel.h"


SqlPhotoTableModel::SqlPhotoTableModel(QObject* parent, QSqlDatabase db)
  : QSqlTableModel(parent, db)
{
  setEditStrategy(QSqlTableModel::OnManualSubmit);
}

SqlPhotoTableModel::~SqlPhotoTableModel()
{
}

QStringList SqlPhotoTableModel::mimeTypes() const
{
  QStringList types;

  types << "text/plain" << "text/uri-list" << "application/kflickr.photo.internal";
  return types;
}

Qt::DropActions SqlPhotoTableModel::supportedDropActions() const
{
  return Qt::CopyAction | Qt::MoveAction;
}

QMimeData* SqlPhotoTableModel::mimeData(const QModelIndexList& indexes) const
{
  QMimeData *mimeData = new QMimeData();
  QByteArray encodedData;

  QDataStream stream(&encodedData, QIODevice::WriteOnly);

  foreach(QModelIndex index, indexes)
  {
    if (index.isValid()) {
      // We can just use the row. If we start to use different views
      // then just the ListView we need to be smarter here to not
      // put duplicate rows on the stream.
      stream << index.row();
    }
  }

  mimeData->setData("application/kflickr.photo.internal", encodedData);
  return mimeData;
}

Qt::ItemFlags SqlPhotoTableModel::flags(const QModelIndex& index) const
{
  Qt::ItemFlags defaultFlags = QSqlTableModel::flags(index);

  if (index.isValid())
    return Qt::ItemIsDragEnabled | Qt::ItemIsDropEnabled | defaultFlags;
  else
    return Qt::ItemIsDropEnabled | defaultFlags;

}

bool SqlPhotoTableModel::dropMimeData(const QMimeData* data, Qt::DropAction action, int row,
				      int column, const QModelIndex& parent)
{
  Q_UNUSED(column);

  // Do nothing, get outta here
  if( action == Qt::IgnoreAction )
    return true;

  int beginRow;
      
  if( row != -1 )
    beginRow = row;
  else if( parent.isValid() )
    beginRow = parent.row();
  else
    beginRow = rowCount();

  // Preferred is to deal in URLs
  if( data->hasUrls() )
  {
    QUrl url;
    QList<QUrl> urls(data->urls());

    // Renumber all the items in rows after the insertion point.
    for( int i = beginRow + 1; i < rowCount(); ++i )
    {
      setData(index(i,fieldIndex("number")), i + urls.size());
    }

    foreach(url, urls)
    {
      if( url.isValid() )
      {
	int id = addPhotograph(url.path(), ++beginRow, false);
	if( id < 0 )
	{
	  // TODO: add a signal to notify the mainwindow of the error.
	  // For now silent failure is fine.
	}
	else
	{
	  // Start the process of fetching a preview
	  PreviewMgr::instance()->previewRequest(id, url);
	}
      }
    }
  }
  else if( data->hasText() )  // however we can handle text as well
  {
    QString filename;
    QStringList filenames(data->text().split("\n", QString::SkipEmptyParts));

    // Renumber all the items in rows after the insertion point.
    for( int i = beginRow + 1; i < rowCount(); ++i )
    {
      setData(index(i,fieldIndex("number")), i + filenames.size());
    }

    foreach( filename, filenames)
    {
      if( !filename.isNull() )
      {
	QUrl url(filename);
	int id = addPhotograph(url.path(), ++beginRow, false);
 	if( id < 0 )
	{
	  // TODO: add a signal to notify the mainwindow of the error.
	  // For now silent failure is fine.
	}
	else
	{
	  // Start the process of fetching a preview
	  PreviewMgr::instance()->previewRequest(id, url);
	}
     }
    }
  }
  else if( data->hasFormat("application/kflickr.photo.internal") )
  {
    int i;
    QSqlRecord rec;
    int numLtBegin = 0;
    QList<int> origins;
    QByteArray encodedData = data->data("application/kflickr.photo.internal");
    QDataStream stream(encodedData);

    // Get the rows being moved
    while (!stream.atEnd())
    {
      stream >> i;
      origins << i;
      if( i <= beginRow ) ++numLtBegin;
    }
    qSort(origins);

    int cnt = 0;
    for( i = 0; i < rowCount(); ++i )
    {
      // Check if this is a row that is not going to be moved.
      if( !origins.contains(i) )
      {
	// Update the 'number' of rows that are not be moved. Take into
	// account rows before the index that are being moved.
	if( i <= beginRow )
	{
	  setData(index(i,fieldIndex("number")), i - cnt);
        }
        else
        {
	  setData(index(i,fieldIndex("number")), i + origins.size() - cnt);
        }
      }
      else
      {
	// Perform the 'number' update on the rows being moved.
	setData(index(i,fieldIndex("number")), beginRow + 1 + cnt - numLtBegin);
        ++cnt;
      }
    }
    submitAll();
    sort(1, Qt::AscendingOrder);
  }
  else
  {
    return false;
  }
  
  return true;
}

int SqlPhotoTableModel::addPhotograph(const QString& filename, int at, bool shuffle)
{
  static int count = getLargestId();
  QFileInfo file(filename);

  // Check that the file exists
  if( !file.exists() )
  {
    return -1;
  }

  if( at < 0 )
  {
    at = rowCount();
  }
  else if( shuffle )
  {
    sort(1, Qt::AscendingOrder);
    for( int i = at; i < rowCount(); ++i )
    {
      setData(index(i,fieldIndex("number")), i + 1);
    }
    submitAll();
  }

  insertRows(0,1);
  setData(index(0, fieldIndex("id")), count);
  setData(index(0, fieldIndex("number")), at);
  setData(index(0, fieldIndex("filename")), file.filePath());
  setData(index(0, fieldIndex("title")), file.baseName());
  setData(index(0, fieldIndex("description")), "");
  setData(index(0, fieldIndex("size")), i18n("Original"));
  setData(index(0, fieldIndex("width")), 1024);
  setData(index(0, fieldIndex("height")), 768);
  setData(index(0, fieldIndex("license")), "All Rights Reserved");
  setData(index(0, fieldIndex("photoset")), i18n("<photostream only>"));
  setData(index(0, fieldIndex("exposed")), true);
  setData(index(0, fieldIndex("family")), false);
  setData(index(0, fieldIndex("friends")), false);
  setData(index(0, fieldIndex("rotation")), 0);
  
  submitAll();

  // I tried to leave the "id" field unset to allow it to autoincrement along with
  // query().lastInsertId().toInt() to return the auto value but got some really
  // really strange behaviour. So instead using the more brute force method seen
  // here with the getLargestId() func. Sucks but works.
  //  QSqlQuery query(database());
  return count++;
}

void SqlPhotoTableModel::removePhotographs(QList<int> rows)
{
  // Sort in decending order.
  qSort(rows.begin(), rows.end(), qGreater<int>());

  int row = 0;
  foreach(row, rows)
  {
    // Ensure it is a valid row
    if( row >= 0 && row < rowCount() )
    {
      // Remove it
      removeRow(row);
    }
  }

  // Recalculate the "number" field for the remaining photos.
  if( row != 0 )
    --row;

  while(row < rowCount())
  {
    setData(index(row,fieldIndex("number")), row);
    ++row;
  }

  // Send the new data to the DB
  submitAll();
}

int SqlPhotoTableModel::getLargestId() const
{
  int lv = 1;
  int rc = rowCount();

  for( int row = 0; row < rc; ++row )
  {
    if( lv < data(index(row, fieldIndex("id"))).toInt() )
    {
      lv = data(index(row, fieldIndex("id"))).toInt() + 1;
    }
  }
  return lv;
}
